% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fractional.operators.R
\name{matern.operators}
\alias{matern.operators}
\title{Rational approximations of stationary Gaussian Matern random fields}
\usage{
matern.operators(
  kappa = NULL,
  tau = NULL,
  alpha = NULL,
  sigma = NULL,
  range = NULL,
  nu = NULL,
  G = NULL,
  C = NULL,
  d = NULL,
  mesh = NULL,
  graph = NULL,
  range_mesh = NULL,
  loc_mesh = NULL,
  m = 1,
  type = c("covariance", "operator"),
  parameterization = c("spde", "matern"),
  compute_higher_order = FALSE,
  return_block_list = FALSE,
  type_rational_approximation = c("chebfun", "brasil", "chebfunLB"),
  compute_logdet = FALSE
)
}
\arguments{
\item{kappa}{Parameter kappa of the SPDE representation. If \code{NULL}, the range parameter will be used. If the range is also \code{NULL}, a starting value based on the mesh will be supplied.}

\item{tau}{Parameter tau of the SPDE representation. If both sigma and tau are \code{NULL}, a starting value based on the mesh will be supplied.}

\item{alpha}{Parameter alpha of the SPDE representation. If \code{alpha} is \code{NULL}, a starting value will be supplied.}

\item{sigma}{Standard deviation of the covariance function. Used if \code{parameterization} is \code{matern}. If \code{NULL}, tau will be used. If tau is also \code{NULL}, a starting value based on the mesh will be supplied.}

\item{range}{Range parameter of the covariance function. Used if \code{parameterization} is \code{matern}. If range is \code{NULL}, a starting value based on the mesh will be supplied.}

\item{nu}{Shape parameter of the covariance function. Used if \code{parameterization} is \code{matern}. If \code{NULL}, a starting value will be supplied.}

\item{G}{The stiffness matrix of a finite element discretization of the
domain of interest. Does not need to be given if either \code{mesh} or \code{graph} is supplied.}

\item{C}{The mass matrix of a finite element discretization of the domain
of interest. Does not need to be given if either \code{mesh} or \code{graph} is supplied.}

\item{d}{The dimension of the domain. Does not need to be given if either
\code{mesh} or \code{graph} is provided.}

\item{mesh}{An optional fmesher mesh. Replaces \code{d}, \code{C} and \code{G}.}

\item{graph}{An optional \code{metric_graph} object. Replaces \code{d}, \code{C} and \code{G}.}

\item{range_mesh}{The range of the mesh. Will be used to provide starting values for the parameters. Will be used if \code{mesh} and \code{graph} are \code{NULL}, and if one of the parameters (kappa or tau for spde parameterization, or sigma or range for matern parameterization) are not provided.}

\item{loc_mesh}{The mesh locations used to construct the matrices C and G. This option should be provided if one wants to use the \code{rspde_lme()} function and will not provide neither graph nor mesh. Only works for 1d data. Does not work for metric graphs. For metric graphs you should supply the graph using the \code{graph} argument.}

\item{m}{The order of the rational approximation, which needs to be a
positive integer. The default value is 1.}

\item{type}{The type of the rational approximation. The options are
"covariance" and "operator". The default is "covariance".}

\item{parameterization}{Which parameterization to use? \code{matern} uses range, std. deviation and nu (smoothness). \code{spde} uses kappa, tau and alpha. The default is \code{spde}.}

\item{compute_higher_order}{Logical. Should the higher order finite
element matrices be computed?}

\item{return_block_list}{Logical. For \code{type = "covariance"},
should the block parts of the precision matrix be returned
separately as a list?}

\item{type_rational_approximation}{Which type of rational
approximation should be used? The current types are
"chebfun", "brasil" or "chebfunLB".}

\item{compute_logdet}{Should log determinants be computed while building the model? (For covariance-based models)}
}
\value{
If \code{type} is "covariance", then \code{matern.operators}
returns an object of class "CBrSPDEobj".
This object is a list containing the
following quantities:
\item{C}{The mass lumped mass matrix.}
\item{Ci}{The inverse of \code{C}.}
\item{GCi}{The stiffness matrix G times \code{Ci}}
\item{Gk}{The stiffness matrix G along with the higher-order
FEM-related matrices G2, G3, etc.}
\item{fem_mesh_matrices}{A list containing the mass
lumped mass matrix, the stiffness matrix and
the higher-order FEM-related matrices.}
\item{m}{The order of the rational approximation.}
\item{alpha}{The fractional power of the precision operator.}
\item{type}{String indicating the type of approximation.}
\item{d}{The dimension of the domain.}
\item{nu}{Shape parameter of the covariance function.}
\item{kappa}{Range parameter of the covariance function}
\item{tau}{Scale parameter of the covariance function.}
\item{sigma}{Standard deviation of the covariance function.}
\item{type}{String indicating the type of approximation.}

If \code{type} is "operator", then \code{matern.operators}
returns an object of class "rSPDEobj". This object contains the
quantities listed in the output of \code{\link[=fractional.operators]{fractional.operators()}},
the \code{G} matrix, the dimension of the domain, as well as the
parameters of the covariance function.
}
\description{
\code{matern.operators} is used for computing a rational SPDE approximation
of a stationary Gaussian random fields on \eqn{R^d} with a Matern covariance
function
\deqn{C(h) = \frac{\sigma^2}{2^{\nu-1}\Gamma(\nu)}
(\kappa h)^\nu K_\nu(\kappa h)}{C(h) = (\sigma^2/(2^{\nu-1}\Gamma(\nu))
(\kappa h)^\nu K_\nu(\kappa h).}
}
\details{
If \code{type} is "covariance", we use the
covariance-based rational approximation of the fractional operator.
In the SPDE approach, we model \eqn{u} as the solution of the following SPDE:
\deqn{L^{\alpha/2}(\tau u) = \mathcal{W},}
where
\eqn{L  = -\Delta +\kappa^2 I} and \eqn{\mathcal{W}} is the standard
Gaussian white noise. The covariance operator of \eqn{u} is given
by \eqn{L^{-\alpha}}. Now, let \eqn{L_h} be a finite-element
approximation of \eqn{L}. We can use
a rational approximation of order \eqn{m} on \eqn{L_h^{-\alpha}} to
obtain the following approximation:
\deqn{L_{h,m}^{-\alpha} = L_h^{-m_\alpha} p(L_h^{-1})q(L_h^{-1})^{-1},}
where \eqn{m_\alpha = \lfloor \alpha\rfloor}, \eqn{p} and \eqn{q} are
polynomials arising from such rational approximation.
From this approximation we construct an approximate precision
matrix for \eqn{u}.

If \code{type} is "operator", the approximation is based on a
rational approximation of the fractional operator
\eqn{(\kappa^2 -\Delta)^\beta}, where \eqn{\beta = (\nu + d/2)/2}.
This results in an approximate model of the form \deqn{P_l u(s) = P_r W,}
where \eqn{P_j = p_j(L)} are non-fractional operators defined in terms
of polynomials \eqn{p_j} for \eqn{j=l,r}. The order of \eqn{p_r} is given
by \code{m} and the order of \eqn{p_l} is \eqn{m + m_\beta}
where \eqn{m_\beta} is the integer part of \eqn{\beta} if \eqn{\beta>1} and
\eqn{m_\beta = 1} otherwise.

The discrete approximation can be written as \eqn{u = P_r x} where
\eqn{x \sim N(0,Q^{-1})}{x ~ N(0,Q^{-1})} and \eqn{Q = P_l^T C^{-1} P_l}.
Note that the matrices \eqn{P_r} and \eqn{Q} may be be
ill-conditioned for \eqn{m>1}. In this case, the methods in
\code{\link[=operator.operations]{operator.operations()}} should be used for operations involving
the matrices, since these methods are more numerically stable.
}
\examples{
# Compute the covariance-based rational approximation of a
# Gaussian process with a Matern covariance function on R
kappa <- 10
sigma <- 1
nu <- 0.8
range <- sqrt(8 * nu) / kappa

# create mass and stiffness matrices for a FEM discretization
nobs <- 101
x <- seq(from = 0, to = 1, length.out = 101)
fem <- rSPDE.fem1d(x)

# compute rational approximation of covariance function at 0.5
op_cov <- matern.operators(
  loc_mesh = x, nu = nu,
  range = range, sigma = sigma, d = 1, m = 2,
  parameterization = "matern"
)

v <- t(rSPDE.A1d(x, 0.5))
# Compute the precision matrix
Q <- op_cov$Q
# A matrix here is the identity matrix
A <- Diagonal(nobs)
# We need to concatenate 3 A's since we are doing a covariance-based rational
# approximation of order 2
Abar <- cbind(A, A, A)
w <- rbind(v, v, v)
# The approximate covariance function:
c_cov.approx <- (Abar) \%*\% solve(Q, w)
c.true <- folded.matern.covariance.1d(
  rep(0.5, length(x)),
  abs(x), kappa, nu, sigma
)

# plot the result and compare with the true Matern covariance
plot(x, c.true,
  type = "l", ylab = "C(h)",
  xlab = "h", main = "Matern covariance and rational approximations"
)
lines(x, c_cov.approx, col = 2)


# Compute the operator-based rational approximation of a Gaussian
# process with a Matern covariance function on R
kappa <- 10
sigma <- 1
nu <- 0.8
range <- sqrt(8 * nu) / kappa

# create mass and stiffness matrices for a FEM discretization
x <- seq(from = 0, to = 1, length.out = 101)
fem <- rSPDE.fem1d(x)

# compute rational approximation of covariance function at 0.5
op <- matern.operators(
  range = range, sigma = sigma, nu = nu,
  loc_mesh = x, d = 1,
  type = "operator",
  parameterization = "matern"
)

v <- t(rSPDE.A1d(x, 0.5))
c.approx <- Sigma.mult(op, v)
c.true <- folded.matern.covariance.1d(
  rep(0.5, length(x)),
  abs(x), kappa, nu, sigma
)

# plot the result and compare with the true Matern covariance
plot(x, c.true,
  type = "l", ylab = "C(h)",
  xlab = "h", main = "Matern covariance and rational approximation"
)
lines(x, c.approx, col = 2)
}
\seealso{
\code{\link[=fractional.operators]{fractional.operators()}},
\code{\link[=spde.matern.operators]{spde.matern.operators()}},
\code{\link[=matern.operators]{matern.operators()}}
}
