% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/factional.computations.R
\name{predict.rSPDEobj}
\alias{predict.rSPDEobj}
\title{Prediction of a fractional SPDE using a rational SPDE approximation}
\usage{
\method{predict}{rSPDEobj}(object, A, Aprd, Y, sigma.e, compute.variances = FALSE, ...)
}
\arguments{
\item{object}{The rational SPDE approximation, computed using \code{\link{fractional.operators}},
\code{\link{matern.operators}}, or \code{\link{spde.matern.operators}}.}

\item{A}{A matrix linking the measurement locations to the basis of the FEM approximation of the latent model.}

\item{Aprd}{A matrix linking the prediction locations to the basis of the FEM approximation of the latent model.}

\item{Y}{A vector with the observed data, can also be a matrix where the columns are observations
of independent replicates of \eqn{u}.}

\item{sigma.e}{The standard deviation of the Gaussian measurement noise. Put to zero if the model
does not have measurement noise.}

\item{compute.variances}{Set to also TRUE to compute the kriging variances.}

\item{...}{further arguments passed to or from other methods.}
}
\value{
A list with elements
\item{mean }{The kriging predictor (the posterior mean of u|Y).}
\item{variance }{The posterior variances (if computed).}
}
\description{
The function is used for computing kriging predictions based on data \eqn{Y_i = u(s_i) + \epsilon_i},
where \eqn{\epsilon}{\epsilon} is mean-zero Gaussian measurement noise and \eqn{u(s)}{u(s)} is defined by
a fractional SPDE \eqn{L^\beta u(s) = W}{L^\beta u(s) = W}, where \eqn{W}{W} is Gaussian white noise.
}
\examples{
#Sample a Gaussian Matern process on R using a rational approximation
kappa <- 10
sigma <- 1
nu <- 0.8
sigma.e <- 0.3

#create mass and stiffness matrices for a FEM discretization
x <- seq(from = 0, to = 1, length.out = 101)
fem <- rSPDE.fem1d(x)

#compute rational approximation
op <- matern.operators(kappa = kappa, sigma = sigma,
                       nu = nu, G=fem$G, C = fem$C, d = 1)

#Sample the model
u <- simulate(op)

#Create some data
obs.loc <- runif(n = 10, min = 0, max = 1)
A <- rSPDE.A1d(x, obs.loc)
Y <- as.vector(A\%*\%u + sigma.e*rnorm(10))

#compute kriging predictions at the FEM grid
A.krig <- rSPDE.A1d(x, x)
u.krig <- predict(op, A = A, Aprd = A.krig, Y = Y, sigma.e = sigma.e,
                  compute.variances= TRUE)

plot(obs.loc, Y, ylab = "u(x)", xlab = "x", main = "Data and prediction",
     ylim = c(min(u.krig$mean - 2*sqrt(u.krig$variance)),
              max(u.krig$mean + 2*sqrt(u.krig$variance))))
lines(x, u.krig$mean)
lines(x, u.krig$mean + 2*sqrt(u.krig$variance), col = 2)
lines(x, u.krig$mean - 2*sqrt(u.krig$variance), col = 2)
}
