% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metadata.R
\name{write_yaml_metadata_block}
\alias{write_yaml_metadata_block}
\title{Write YAML Metadata Block for Quarto Documents}
\usage{
write_yaml_metadata_block(..., .list = NULL)
}
\arguments{
\item{...}{Named arguments to include in the metadata block. Names become
the metadata keys and values become the metadata values. These take
precedence over any conflicting keys in \code{.list}.}

\item{.list}{Optional list of additional metadata to include. This is useful
when you have metadata stored in a list variable. Keys in \code{.list} are
overridden by any matching keys provided in \code{...}.}
}
\value{
A character string containing the formatted YAML metadata block,
wrapped with \code{knitr::asis_output()} so it renders as raw markdown.
Returns \code{NULL} invisibly if no metadata is provided.
}
\description{
Creates a YAML metadata block that can be dynamically inserted into Quarto
documents from R code chunks. This allows setting metadata values based on
R computations, which can then be used with Quarto's conditional content
features like \code{when-meta} and \verb{\{\{< meta >\}\}} shortcodes.
}
\details{
The function converts R values to YAML format and wraps them in YAML
delimiters (\verb{---}). Logical values are converted to lowercase strings
("true"/"false") to ensure compatibility with Quarto's metadata system.

When both \code{...} and \code{.list} contain the same key, the value from \code{...}
takes precedence and will override the value from \code{.list}.

If no metadata is provided (empty \code{...} and \code{NULL} or empty \code{.list}),
the function returns \code{NULL} without generating any output.

This addresses the limitation where Quarto metadata must be static and
cannot be set dynamically from R code during document rendering.
\subsection{YAML 1.2 Compatibility:}{

To ensure compatibility with Quarto's YAML 1.2 parser (js-yaml), the function
automatically handles two key differences between R's yaml package (YAML 1.1)
and YAML 1.2:
\subsection{Boolean values:}{

R logical values (\code{TRUE}/\code{FALSE}) are converted to lowercase
YAML 1.2 format (\code{true}/\code{false}) using [yaml::verbatim_logical()]. This prevents
YAML 1.1 boolean representations like \code{yes}/\code{no} from being used.
}

\subsection{String quoting:}{

Strings with leading zeros that contain digits 8 or 9 (like \code{"029"}, \code{"089"})
are automatically quoted to prevent them from being parsed as octal numbers,
which would result in data corruption (e.g., \code{"029"} becoming \code{29}).
Valid octal numbers containing only digits 0-7 (like \code{"0123"}) are handled
by the underlying \pkg{yaml} package.

For manual control over string quoting behavior, use [yaml_quote_string()].
}

}

\subsection{Quarto Usage:}{

To use this function in a Quarto document, create an R code chunk with
the \code{output: asis} option:

\if{html}{\out{<div class="sourceCode">}}\preformatted{```\{r\}
#| output: asis
write_yaml_metadata_block(admin = TRUE, version = "1.0")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{
Without the `output: asis` option, the YAML metadata block will be
displayed as text rather than processed as metadata by Quarto.


[yaml::verbatim_logical()]: R:yaml::verbatim_logical()
[yaml_quote_string()]: R:yaml_quote_string()
}\if{html}{\out{</div>}}
}
}
\examples{
\dontrun{
# In a Quarto document R chunk with `#| output: asis`:
admin <- TRUE
user_level <- "advanced"

# Set metadata dynamically
write_yaml_metadata_block(
  admin = admin,
  level = user_level,
  timestamp = Sys.Date()
)

# Strings with leading zeros are automatically quoted for YAML 1.2 compatibility
write_yaml_metadata_block(
  zip_code = "029",    # Automatically quoted as "029"
  build_id = "0123"    # Quoted by yaml package (valid octal)
)

# Use with .list parameter
metadata_list <- list(version = "1.0", debug = FALSE)
write_yaml_metadata_block(.list = metadata_list)

# Direct arguments override .list values
base_config <- list(theme = "dark", debug = TRUE)
write_yaml_metadata_block(
  debug = FALSE,  # This overrides debug = TRUE from base_config
  author = "John",
  .list = base_config
)

# Then use in Quarto with conditional content:
# ::: {.content-visible when-meta="admin"}
# Admin-only content here
# :::
}

}
\seealso{
\code{\link[=yaml_quote_string]{yaml_quote_string()}} for explicitly controlling which strings are quoted
in YAML output when you encounter edge cases that need manual handling.
}
