% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/qic.R
\name{qic}
\alias{qic}
\title{Quality improvement charts}
\usage{
qic(y, n, x, data, chart = c("run", "i", "mr", "xbar", "s", "t", "p", "c",
  "u", "g"), notes = NULL, cl = NULL, ylim = NULL, target = NULL,
  freeze = NULL, breaks = NULL, exclude = NULL, negy = TRUE,
  dots.only = FALSE, multiply = 1, x.format = "\%Y-\%m-\%d", cex = 0.8,
  main, xlab = "Subgroup", ylab = "Indicator", decimals = NULL,
  pre.text = "Before data", post.text = "After data", runvals = FALSE,
  linevals = TRUE, plot.chart = TRUE, prnt = FALSE, primed = FALSE,
  standardised = FALSE, ...)
}
\arguments{
\item{y}{Numeric vector of counts or measures to plot. Mandatory.}

\item{n}{Numeric vector of sample sizes. Mandatory for P and U charts.}

\item{x}{Subgrouping vector used for aggregating data and making x-labels.
Mandatory for Xbar and S charts.}

\item{data}{Data frame containing variables.}

\item{chart}{Type of control chart. Possible types are: \itemize{ \item
"run": run chart (default). \item "i": individuals chart. \item "mr":
moving range chart. \item "xbar": sample average chart. \item "s": sample
standard deviation chart. \item "t": time between events chart. \item "p":
proportions chart. \item "c": counts chart. \item "u": rates chart. \item
"g": cases between events chart. }}

\item{notes}{Character vector of notes to be added to individual. data
points.}

\item{cl}{Value specifying the center line (if known).}

\item{ylim}{Range of y axis limits.}

\item{target}{Value specifying a target line to plot.}

\item{freeze}{Number identifying the last data point to include in
calculations of center and limits (ignored if \code{breaks} argument is
given).}

\item{breaks}{Numeric vector of break points.}

\item{exclude}{Numeric vector of data points to exclude from calculations of
center and control lines.}

\item{negy}{Logical value, if TRUE, the y axis is allowed to be negative
(only relevant for I and Xbar charts).}

\item{dots.only}{Logical value, if TRUE, data points are not connected by
lines and runs analysis is not performed. Useful for comparison and funnel
plots.}

\item{multiply}{Integer indicating a number to multiply y axis by, e.g. 100
for percents rather than proportions.}

\item{x.format}{Date format of x axis labels.}

\item{cex}{Number indicating the amount by which text and symbols should be
magnified.}

\item{main}{Character string specifying the title of the plot.}

\item{xlab}{Character string specifying the x axis label.}

\item{ylab}{Character string specifying the y axis label.}

\item{decimals}{Integer indicating the number of decimals shown for center
and limits on the plot. Default behaviour is smart rounding to at least two
significant digits.}

\item{pre.text}{Character string labelling pre-freeze period}

\item{post.text}{Character string labelling post-freeze period}

\item{runvals}{Logical value, if TRUE, prints statistics from runs analysis
on plot.}

\item{linevals}{Logical value, if TRUE, prints values for center and control
lines on plot.}

\item{plot.chart}{Logical value, if TRUE, prints plot.}

\item{prnt}{Logical value, if TRUE, prints return value}

\item{primed}{Logical value, if TRUE, control limits incorporate
between-subgroup variation as proposed by Laney (2002). This is recommended
for data involving very large sample sizes \code{n}. Only relevant for P
and U charts.}

\item{standardised}{Logical value, if TRUE, creates a standardised control
chart, where points are plotted in standard deviation units along with a
center line at zero and control limits at 3 and -3. Only relevant for P, U
and Xbar charts.}

\item{...}{Further arguments to plot function.}
}
\value{
A list of values and parameters of the qic plot.
}
\description{
Run and control charts for quality improvement and control
}
\details{
If \code{chart} is not specified, \code{qic} plots a \strong{run
  chart}. Non-random variation will be marked by a dashed, red center line
  (the median) if either the longest run of data points above or below the
  median is longer than predicted or if the graph crosses the median fewer
  times than predicted (see references for details).

  Only the \code{y} argument giving the count or measure of interest is
  mandatory for a run chart. If a denominator argument, \code{n}, is given,
  \eqn{y/n} will be plotted. If a subgrouping argument, \code{x}, is given,
  \eqn{sum(y)/sum(n)}, within each subgroup will be plotted.

  With \strong{controlcharts}, data aggregation by subgroups is handled
  according to chart type. For P, U, and I charts, data are aggregated as
  described for the run chart. For the C chart, the sum of counts,
  \code{sum(y)}, within each subgroups will be plotted.

  For Xbar and S charts, the subgrouping argument, \code{x}, is mandatory.
  However, the sample size argument, \code{n}, is irrelevant and will be
  ignored.

  The subgrouping argument, \code{x}, is irrelevant for T and G charts, and,
  if given, an error will occur if any subgroup has more than one element.

  If more than one \code{note} is present within any subgroup, the first
  \code{note} (alphabetically) is chosen.

  If both \code{primed} and \code{standardised} are \code{TRUE}, points are
  plotted in units corresponding to Laney's modified "standard deviation",
  which incorporates the variation between subgroups.
}
\examples{
set.seed(1)
# Run chart of 24 samples of a random continuous variable
# with an approximate mean = 12 and standard deviation = 3.
y <- rnorm(24, 12, 3)
qic(y)

# Add subgroup vector (dates) and a target
x <- seq.Date(as.Date('2013-08-04'), by = 'week', length = 24)
qic(y, x = x, target = 16)

# Individuals control chart
qic(y, x = x, chart = 'i')

# Xbar control chart, sample size = 5
y <- rnorm(5 * 24)
x <- rep(x, 5)
qic(y, x = x, chart = 'xbar')

# Create data frame with counts and sample sizes by week
d <- data.frame(week = seq.Date(as.Date('2013-08-04'),
                                by = 'week',
                                length = 36),
                y = c(rbinom(24, 20, 0.5), rbinom(12, 20, 0.8)),
                n = round(rnorm(36, 20, 2)))

# Proportions control chart
qic(y, n, x = week, data = d[1:24,], chart = 'p')

# Introduce change in process performance
qic(y, n, x = week, data = d, chart = 'p')

# Freeze baseline to first 24 samples
qic(y, n, x = week, data = d, chart = 'p', freeze = 24)

# Break control chart before and after change
qic(y, n, x = week, data = d, chart = 'p', breaks = 24)

# Introduce extreme sample value and notes
d$a <- ''
d$a[30] <- 'Extreme value'
d$y[30] <- 1
qic(y, n, x = week, data = d, chart = 'p',
    breaks = 24,
    notes = a)

# Exclude value from calculations
d$a[30] <- 'Value excluded from calculations'
qic(y, n, x = week, data = d, chart = 'p',
    breaks = 24,
    notes = a,
    exclude = 30)
}
\references{
Runs analysis: \itemize{ \item Mark F. Schilling (2012). The
  Surprising Predictability of Long Runs. Math. Mag. 85, 141-149. \item
  Zhenmin Chen (2010). A note on the runs test. Model Assisted Statistics and
  Applications 5, 73-77. } Calculation of control limits: \itemize{ \item
  Douglas C. Montgomery (2009). Introduction to Statistical Process Control,
  Sixth Edition, John Wiley & Sons. \item James C. Benneyan (2001).
  Number-Between g-Type Statistical Quality Control Charts for Monitoring
  Adverse Events. Health Care Management Science 4, 305-318. \item Lloyd P.
  Provost, Sandra K. Murray (2011). The Health Care Data Guide: Learning from
  Data for Improvement. San Fransisco: John Wiley & Sons Inc. \item David B.
  Laney (2002). Improved control charts for attributes. Quality Engineering,
  14(4), 531-537.}
}

