% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/matreg.R
\name{matreg}
\alias{matreg}
\alias{matrixreg}
\alias{lm_mat}
\alias{lm_matrix}
\title{Compute linear regression models and generate "lm" objects from covariance matrices.}
\usage{
matreg(formula, cov_mat, mean_vec = rep(0, ncol(cov_mat)), n = Inf,
  se_beta_method = "lm", ...)

matrixreg(formula, cov_mat, mean_vec = rep(0, ncol(cov_mat)), n = Inf,
  se_beta_method = "lm", ...)

lm_mat(formula, cov_mat, mean_vec = rep(0, ncol(cov_mat)), n = Inf,
  se_beta_method = "lm", ...)

lm_matrix(formula, cov_mat, mean_vec = rep(0, ncol(cov_mat)), n = Inf,
  se_beta_method = "lm", ...)
}
\arguments{
\item{formula}{Regression formula with a single outcome variable on the left-hand side and one or more predictor variables on the right-hand side (e.g., Y ~ X1 + X2).}

\item{cov_mat}{Covariance matrix containing the variables to be used in the regression.}

\item{mean_vec}{Vector of means corresponding to the variables in \code{cov_mat}.}

\item{n}{Sample size to be used in significance testing}

\item{se_beta_method}{Method to use to estimate the standard errors of standardized regression (beta) coefficients.
Current options include "lm" (estimate standard errors using conventional regression formulas) and "normal" (use the Jones-Waller normal-theory approach from the \code{fungible::seBeta()} function)}

\item{...}{Additional arguments.}
}
\value{
An object with the class "lm_mat" that can be used with summary, print, predict, and anova methods.
}
\description{
Compute linear regression models and generate "lm" objects from covariance matrices.
}
\examples{
## Generate data
S <- reshape_vec2mat(cov = c(.3 * 2 * 3,
                             .4 * 2 * 4,
                             .5 * 3 * 4),
                     var = c(2, 3, 4)^2,
                     var_names = c("X", "Y", "Z"))
mean_vec <- setNames(c(1, 2, 3), colnames(S))
dat <- data.frame(MASS::mvrnorm(n = 100, mu = mean_vec,
                                Sigma = S, empirical = TRUE))

## Compute regression models with lm
lm_out1 <- lm(Y ~ X, data = dat)
lm_out2 <- lm(Y ~ X + Z, data = dat)

## Compute regression models with matreg
matreg_out1 <- matreg(formula = Y ~ X, cov_mat = S, mean_vec = mean_vec, n = nrow(dat))
matreg_out2 <- matreg(formula = Y ~ X + Z, cov_mat = S, mean_vec = mean_vec, n = nrow(dat))

## Compare results of lm and matreg with one predictor
lm_out1
matreg_out1

## Compare summaries of lm and matreg with one predictor
summary(lm_out1)
summary(matreg_out1)

## Compare results of lm and matreg with two predictors
lm_out2
matreg_out2

## Compare summaries of lm and matreg with two predictors
summary(lm_out2)
summary(matreg_out2)

## Compare predictions made with lm and matreg
predict(object = matreg_out1, newdata = data.frame(X = 1:5))
predict(object = summary(matreg_out1), newdata = data.frame(X = 1:5))
predict(lm_out1, newdata = data.frame(X = 1:5))

## Compare predictions made with lm and matreg (with confidence intervals)
predict(object = matreg_out1, newdata = data.frame(X = 1:5),
        se.fit = TRUE, interval = "confidence")
predict(lm_out1, newdata = data.frame(X = 1:5),
        se.fit = TRUE, interval = "confidence")

## Compare predictions made with lm and matreg (with prediction intervals)
predict(object = matreg_out1, newdata = data.frame(X = 1:5),
        se.fit = TRUE, interval = "prediction")
predict(lm_out1, newdata = data.frame(X = 1:5),
        se.fit = TRUE, interval = "prediction")

## Compare model comparisons computed using lm and matreg objects
anova(lm_out1, lm_out2)
anova(matreg_out1, matreg_out2)

## Model comparisons can be run on matreg summaries, too:
anova(summary(matreg_out1), summary(matreg_out2))
## Or summaries and raw models can be mixed:
anova(matreg_out1, summary(matreg_out2))
anova(summary(matreg_out1), matreg_out2)


## Compare confidence intervals computed using lm and matreg objects
confint(object = lm_out1)
confint(object = matreg_out1)
confint(object = summary(matreg_out1))

confint(object = lm_out2)
confint(object = matreg_out2)
confint(object = summary(matreg_out2))
}
