% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_asym_connectivity_penalties.R
\name{add_asym_connectivity_penalties}
\alias{add_asym_connectivity_penalties}
\alias{add_asym_connectivity_penalties,ConservationProblem,ANY,ANY,Matrix-method}
\alias{add_asym_connectivity_penalties,ConservationProblem,ANY,ANY,matrix-method}
\alias{add_asym_connectivity_penalties,ConservationProblem,ANY,ANY,dgCMatrix-method}
\alias{add_asym_connectivity_penalties,ConservationProblem,ANY,ANY,data.frame-method}
\alias{add_asym_connectivity_penalties,ConservationProblem,ANY,ANY,array-method}
\title{Add asymmetric connectivity penalties}
\usage{
\S4method{add_asym_connectivity_penalties}{ConservationProblem,ANY,ANY,matrix}(x, penalty, zones, data)

\S4method{add_asym_connectivity_penalties}{ConservationProblem,ANY,ANY,Matrix}(x, penalty, zones, data)

\S4method{add_asym_connectivity_penalties}{ConservationProblem,ANY,ANY,data.frame}(x, penalty, zones, data)

\S4method{add_asym_connectivity_penalties}{ConservationProblem,ANY,ANY,dgCMatrix}(x, penalty, zones, data)

\S4method{add_asym_connectivity_penalties}{ConservationProblem,ANY,ANY,array}(x, penalty, zones, data)
}
\arguments{
\item{x}{\code{\link[=problem]{problem()}} object.}

\item{penalty}{\code{numeric} penalty that is used to scale the importance
of selecting planning units with strong connectivity between them compared
to the main problem objective (e.g., solution cost when the argument to
\code{x} has a minimum set objective set using
\code{\link[=add_min_set_objective]{add_min_set_objective()}}). Higher \code{penalty} values
can be used to obtain solutions with a high degree of connectivity,
and smaller \code{penalty} values can be used to obtain solutions with a
small degree of connectivity. Note that negative \code{penalty} values can
be used to obtain solutions that have very little connectivity.}

\item{zones}{\code{matrix} or \code{Matrix} object describing the
level of connectivity between different zones. Each row and column
corresponds to a different zone in the argument to \code{x}, and cell
values indicate the level of connectivity between each combination
of zones. Cell values along the diagonal of the matrix represent
the level of connectivity between planning units allocated to the
same zone. Cell values must lay between 1 and -1, where negative
values favor solutions with weak connectivity. The default argument to
\code{zones} is an identity matrix (i.e., a matrix with ones along the
matrix diagonal and zeros elsewhere), so that planning units are
only considered to be connected when they are allocated to the same zone.
This argument is required when working with multiple zones and the
argument to \code{data} is a \code{matrix} or \code{Matrix} object.
If the argument to \code{data} is an \code{array} or \code{data.frame} with data for
multiple zones (e.g., using the \code{"zone1"} and \code{"zone2"} column names),
this argument must explicitly be set to \code{NULL} otherwise an error will be
thrown.}

\item{data}{\code{matrix}, \code{Matrix}, \code{data.frame}, or
\code{array} object containing connectivity data. The connectivity values
correspond to the strength of connectivity between
different planning units. Thus connections between planning units
that are associated with higher values are more favorable in the solution.
See the Data format section for more information.}
}
\value{
An updated \code{\link[=problem]{problem()}} object with the penalties added to it.
}
\description{
Add penalties to a conservation planning problem to account for
asymmetric connectivity between planning units.
Asymmetric connectivity data describe connectivity information that is
directional.
For example, asymmetric connectivity data could describe
the strength of rivers flowing between different planning units. Since
river flow is directional, the level of connectivity
from an upstream planning unit to a downstream planning unit would
be higher than that from a downstream planning unit to an upstream planning
unit.
}
\details{
This function adds penalties to conservation planning problem to penalize
solutions that have low connectivity.
Specifically, it penalizes solutions that select planning units that
share high connectivity values with other planning units that are
not selected by the solution (based on Beger \emph{et al.} 2010).
}
\section{Mathematical formulation}{

The connectivity penalties are implemented using the following equations.
Let \eqn{I} represent the set of planning units
(indexed by \eqn{i} or \eqn{j}), \eqn{Z} represent the set
of management zones (indexed by \eqn{z} or \eqn{y}), and \eqn{X_{iz}}{Xiz}
represent the decision variable for planning unit \eqn{i} for in zone
\eqn{z} (e.g., with binary
values one indicating if planning unit is allocated or not). Also, let
\eqn{p} represent the argument to \code{penalty}, \eqn{D} represent the
argument to \code{data}, and \eqn{W} represent the argument
to \code{zones}.

If the argument to \code{data} is supplied as a \code{matrix} or
\code{Matrix} object, then the penalties are calculated as:

\deqn{
\sum_{i}^{I} \sum_{j}^{I} \sum_{z}^{Z} \sum_{y}^{Z}
(p \times X_{iz} \times D_{ij} \times W_{zy}) -
\sum_{i}^{I} \sum_{j}^{I} \sum_{z}^{Z} \sum_{y}^{Z}
(p \times X_{iz} \times X_{jy} \times D_{ij} \times W_{zy})}{
sum_i^I sum_j^I sum_z^Z sum_y^Z
(p * Xiz * Dij * Wzy) -
sum_i^I sum_j^I sum_z^Z sum_y^Z
(p * Xiz * Xjy * Dij * Wzy)
}

Otherwise, if the argument to \code{data} is supplied as an
\code{array} object, then the penalties are
calculated as:

\deqn{
\sum_{i}^{I} \sum_{j}^{I} \sum_{z}^{Z} \sum_{y}^{Z}
(p \times X_{iz} \times D_{ijzy}) -
\sum_{i}^{I} \sum_{j}^{I} \sum_{z}^{Z} \sum_{y}^{Z}
(p \times X_{iz} \times X_{jy} \times D_{ijzy})}{
sum_i^I sum_j^I sum_z^Z sum_y^Z
(p * Xiz * Dijzy) -
sum_i^I sum_j^I sum_z^Z sum_y^Z
(p * Xiz * Xjy * Dijzy)
}

Note that when the problem objective is to maximize some measure of
benefit and not minimize some measure of cost, the term \eqn{p} is
replaced with \eqn{-p}.
}

\section{Data format}{

The argument to \code{data} can be specified using several different formats.

\describe{

\item{\code{data} as a \code{matrix}/\code{Matrix} object}{where rows and columns represent
different planning units and the value of each cell represents the
strength of connectivity between two different planning units. Cells
that occur along the matrix diagonal are treated as weights which
indicate that planning units are more desirable in the solution.
The argument to \code{zones} can be used to control
the strength of connectivity between planning units in different zones.
The default argument for \code{zones} is to treat planning units
allocated to different zones as having zero connectivity.}

\item{\code{data} as a \code{data.frame} object}{containing columns that are named
\code{"id1"}, \code{"id2"}, and \code{"boundary"}. Here, each row
denotes the connectivity between a pair of planning units
(per values in the \code{"id1"} and \code{"id2"} columns) following the
\emph{Marxan} format.
If the argument to \code{x} contains multiple zones, then the
\code{"zone1"} and \code{"zone2"} columns can optionally be provided to manually
specify the connectivity values between planning units when they are
allocated to specific zones. If the \code{"zone1"} and
\code{"zone2"} columns are present, then the argument to \code{zones} must be
\code{NULL}.}

\item{\code{data} as an \code{array} object}{
containing four-dimensions where cell values
indicate the strength of connectivity between planning units
when they are assigned to specific management zones. The first two
dimensions (i.e., rows and columns) indicate the strength of
connectivity between different planning units and the second two
dimensions indicate the different management zones. Thus
the \code{data[1, 2, 3, 4]} indicates the strength of
connectivity between planning unit 1 and planning unit 2 when planning
unit 1 is assigned to zone 3 and planning unit 2 is assigned to zone 4.}

}
}

\examples{
\dontrun{
# load package
library(Matrix)

# set seed for reproducibility
set.seed(600)

# load data
sim_pu_polygons <- get_sim_pu_polygons()
sim_features <- get_sim_features()
sim_zones_pu_raster <- get_sim_zones_pu_raster()
sim_zones_features <- get_sim_zones_features()

# create basic problem
p1 <-
  problem(sim_pu_polygons, sim_features, "cost") \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(0.2) \%>\%
  add_default_solver(verbose = FALSE)

# create an asymmetric connectivity matrix. Here, connectivity occurs between
# adjacent planning units and, due to rivers flowing southwards
# through the study area, connectivity from northern planning units to
# southern planning units is ten times stronger than the reverse.
acm1 <- matrix(0, nrow(sim_pu_polygons), nrow(sim_pu_polygons))
acm1 <- as(acm1, "Matrix")
centroids <- sf::st_coordinates(
  suppressWarnings(sf::st_centroid(sim_pu_polygons))
)
adjacent_units <- sf::st_intersects(sim_pu_polygons, sparse = FALSE)
for (i in seq_len(nrow(sim_pu_polygons))) {
  for (j in seq_len(nrow(sim_pu_polygons))) {
    # find if planning units are adjacent
    if (adjacent_units[i, j]) {
      # find if planning units lay north and south of each other
      # i.e., they have the same x-coordinate
      if (centroids[i, 1] == centroids[j, 1]) {
        if (centroids[i, 2] > centroids[j, 2]) {
          # if i is north of j add 10 units of connectivity
          acm1[i, j] <- acm1[i, j] + 10
        } else if (centroids[i, 2] < centroids[j, 2]) {
          # if i is south of j add 1 unit of connectivity
          acm1[i, j] <- acm1[i, j] + 1
        }
      }
    }
  }
}

# rescale matrix values to have a maximum value of 1
acm1 <- rescale_matrix(acm1, max = 1)

# visualize asymmetric connectivity matrix
image(acm1)

# create penalties
penalties <- c(1, 50)

# create problems using the different penalties
p2 <- list(
  p1,
  p1 \%>\% add_asym_connectivity_penalties(penalties[1], data = acm1),
  p1 \%>\% add_asym_connectivity_penalties(penalties[2], data = acm1)
)

# solve problems
s2 <- lapply(p2, solve)

# create object with all solutions
s2 <- sf::st_sf(
  tibble::tibble(
    p2_1 = s2[[1]]$solution_1,
    p2_2 = s2[[2]]$solution_1,
    p2_3 = s2[[3]]$solution_1
 ),
 geometry = sf::st_geometry(s2[[1]])
)

names(s2)[1:3] <- c("basic problem", paste0("acm1 (", penalties,")"))

# plot solutions based on different penalty values
plot(s2, cex = 1.5)

# create minimal multi-zone problem and limit solver to one minute
# to obtain solutions in a short period of time
p3 <-
  problem(sim_zones_pu_raster, sim_zones_features) \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(matrix(0.15, nrow = 5, ncol = 3)) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(time_limit = 60, verbose = FALSE)

# crate asymmetric connectivity data by randomly simulating values
acm2 <- matrix(
  runif(ncell(sim_zones_pu_raster) ^ 2),
  nrow = ncell(sim_zones_pu_raster)
)

# create multi-zone problems using the penalties
p4 <- list(
  p3,
  p3 \%>\% add_asym_connectivity_penalties(penalties[1], data = acm2),
  p3 \%>\% add_asym_connectivity_penalties(penalties[2], data = acm2)
)

# solve problems
s4 <- lapply(p4, solve)
s4 <- lapply(s4, category_layer)
s4 <- terra::rast(s4)
names(s4) <- c("basic problem", paste0("acm2 (", penalties,")"))

# plot solutions
plot(s4, axes = FALSE)
}

}
\references{
Beger M, Linke S, Watts M, Game E, Treml E, Ball I, and Possingham, HP (2010)
Incorporating asymmetric connectivity into spatial decision making for
conservation, \emph{Conservation Letters}, 3: 359--368.
}
\seealso{
See \link{penalties} for an overview of all functions for adding penalties.

Other penalties: 
\code{\link{add_boundary_penalties}()},
\code{\link{add_connectivity_penalties}()},
\code{\link{add_feature_weights}()},
\code{\link{add_linear_penalties}()}
}
\concept{penalties}
