#ifndef matrice_hpp
#define matrice_hpp

#include <vector>
#include <map>
#include <cstring>
#include <limits>
#include <algorithm>
#include <cstdint>
#include "myException.h"

template <class TIPO>
class Matrice {
protected:
    TIPO *__dati;
    std::uint_fast64_t __size;
    std::uint_fast64_t __righe;
    std::uint_fast64_t __colonne;
public:
    Matrice(std::uint_fast64_t righe, std::uint_fast64_t colonne, TIPO init_val=TIPO()) {
        __righe = righe;
        __colonne = colonne;
        __size = __righe * __colonne;
        __dati = new TIPO[__size];
        std::fill(__dati, __dati + __size, init_val);
    }
    Matrice(std::uint_fast64_t size, TIPO init_val=TIPO()) {
        __righe = size;
        __colonne = size;
        __size = __righe * __colonne;
        __dati = new TIPO[__size];
        std::fill(__dati, __dati + __size, init_val);
    }
    
    virtual ~Matrice() {
        delete[] __dati;
    }
    
    virtual TIPO& operator()(std::uint_fast64_t row, std::uint_fast64_t col) {
        if (row >= __righe || col >= __colonne) {
            std::string err_str = "Matrice::operator= wrong position";
            throw_line(err_str);
        }
        return __dati[row * __colonne + col];
    }
    
    virtual TIPO at(std::uint_fast64_t row, std::uint_fast64_t col) const {
        if (row >= __righe || col >= __colonne) {
            std::string err_str = "Matrice::operator= wrong position";
            throw_line(err_str);
        }
        return __dati[row * __colonne + col];
    }
    
    virtual std::uint_fast64_t Rows() const {
        return __righe;
    }
    
    virtual std::uint_fast64_t Cols() const {
        return __colonne;
    }
    
    virtual bool operator==(Matrice& m) const {
        auto result = std::memcmp(__dati, m.__dati, sizeof(TIPO) * __size);
        bool risultato = (result == 0);
        return risultato;
    }
    
    virtual Matrice& operator=(Matrice& m) {
        if (__righe != m.__righe || __colonne != m.__colonne) {
            std::string err_str = "Matrice::operator= wrong size";
            throw_line(err_str);
        }
        std::memcpy(__dati, m.__dati, sizeof(TIPO) * __size);
        return *this;
    }
    
    virtual bool operator!=(Matrice& m) const {
        return !(*this == m);
    }
    
    virtual std::string to_string(char DELIMETER = ';') const {
        std::string r = "";
        for (size_t k = 0; k <  this->Rows(); ++k)
        {
            bool first = true;
            for (size_t h = 0; h < this->Cols(); ++h)
            {
                if (first)
                {
                    r += std::to_string(this->at(k, h));
                    first = false;
                }
                else
                {
                    r += DELIMETER + std::to_string(this->at(k, h));
                }
            }
            r += "\n";
        }
        
        return r;
    }
    
    virtual std::string to_string(std::vector<std::string> header, char DELIMETER = ';') const {
        std::string r = "";
        for (size_t h = 0; h < this->Cols(); ++h) {
            r += DELIMETER + header.at(h);
        }
        r += "\n";
        
        
        for (size_t k = 0; k <  this->Rows(); ++k) {
            r += header.at(k);
            for (size_t h = 0; h < this->Cols(); ++h) {
                r += DELIMETER + std::to_string(this->at(k, h));
            }
            r += "\n";
        }
        
        return r;
    }
    
};

//************************************
//************************************
//************************************

class MatriceSparsa : public Matrice<double> {
private:
    std::uint_fast64_t __size_allocata;
    std::map<std::uint_fast64_t, std::uint_fast64_t> __elements_conversion;
    double __valore_nan;
public:
    MatriceSparsa(std::uint_fast64_t size, std::vector<std::uint_fast64_t>& elements_used, double init_val=0.0) : Matrice(1, init_val) {
        delete[] __dati;
        __righe = size;
        __colonne = size;
        __size = __righe * __colonne;
        __size_allocata = elements_used.size();
        __dati = new double[__size_allocata * __size_allocata];
        std::fill(__dati, __dati + (__size_allocata * __size_allocata), init_val);
        for (std::uint_fast64_t p = 0; p < elements_used.size(); ++p) {
            __elements_conversion[elements_used[p]] = p;
        }
        __valore_nan = std::numeric_limits<double>::quiet_NaN();
    }
    
    double& operator()(std::uint_fast64_t row, std::uint_fast64_t col) {
        if (row >= __righe || col >= __colonne) {
            std::string err_str = "MatriceSparsa::operator= wrong position";
            throw_line(err_str);
        }
        __valore_nan = std::numeric_limits<double>::quiet_NaN();
        if (__elements_conversion.find(row) == __elements_conversion.end() || __elements_conversion.find(col) == __elements_conversion.end()) {
            return __valore_nan;
        }
        auto r = __elements_conversion.at(row);
        auto c = __elements_conversion.at(col);
        return __dati[r * __size_allocata + c];
    }
    
    double at(std::uint_fast64_t row, std::uint_fast64_t col) const {
        if (row >= __righe || col >= __colonne) {
            std::string err_str = "MatriceSparsa::operator= wrong position";
            throw_line(err_str);
        }
        if (__elements_conversion.find(row) == __elements_conversion.end() || __elements_conversion.find(col) == __elements_conversion.end()) {
            return std::numeric_limits<double>::quiet_NaN();
        }
        auto r = __elements_conversion.at(row);
        auto c = __elements_conversion.at(col);
        return __dati[r * __size_allocata + c];
    }
    
    bool operator==(MatriceSparsa& m) const {
        if (this->__elements_conversion != m.__elements_conversion) {
            return false;
        }
        auto result = std::memcmp(__dati, m.__dati, sizeof(double) * __size_allocata * __size_allocata);
        bool risultato = (result == 0);
        return risultato;
    }
    
    MatriceSparsa& operator=(MatriceSparsa& m) {
        if (this->__elements_conversion != m.__elements_conversion) {
            std::string err_str = "Matrice::operator= wrong size";
            throw_line(err_str);
        }
        std::memcpy(__dati, m.__dati, sizeof(double) * __size_allocata * __size_allocata);
        return *this;
    }
    
    bool operator!=(MatriceSparsa& m) const {
        return !(*this == m);
    }
    
    std::string to_string(char DELIMETER = ';') const {
        std::string risultato = "";
        for (std::uint_fast64_t row = 0; row < __righe; ++row) {
            bool first = true;
            for (std::uint_fast64_t col = 0; col < __colonne; ++col) {
                double valore = std::numeric_limits<double>::quiet_NaN();
                if (__elements_conversion.find(row) != __elements_conversion.end() && __elements_conversion.find(col) != __elements_conversion.end()) {
                    auto r = __elements_conversion.at(row);
                    auto c = __elements_conversion.at(col);
                    valore = __dati[r * __size_allocata + c];
                }
                if (first) {
                    risultato += std::to_string(valore);
                    first = false;
                } else {
                    risultato += DELIMETER + std::to_string(valore);
                }
            }
            risultato += "\n";
        }
        
        return risultato;
    }
    
    std::string to_string(std::vector<std::string> header, char DELIMETER = ';') const {
        std::string r = "";
        for (size_t h = 0; h < this->Cols(); ++h) {
            r += DELIMETER + header.at(h);
        }
        r += "\n";
        
        
        for (size_t k = 0; k <  this->Rows(); ++k) {
            r += header.at(k);
            for (size_t h = 0; h < this->Cols(); ++h) {
                r += DELIMETER + std::to_string(this->at(k, h));
            }
            r += "\n";
        }
        
        return r;
    }
    
};


#endif /* matrice_hpp */
