% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bootstraping.R
\name{aboot}
\alias{aboot}
\alias{bootstrap}
\title{Calculate a dendrogram with bootstrap support using any distance applicable 
to genind or genclone objects.}
\usage{
aboot(x, strata = NULL, tree = "upgma", distance = "nei.dist",
  sample = 100, cutoff = 0, showtree = TRUE, missing = "mean",
  mcutoff = 0, quiet = FALSE, root = NULL, ...)
}
\arguments{
\item{x}{a \linkS4class{genind}, \linkS4class{genpop},
\linkS4class{genclone}, \linkS4class{genlight}, \linkS4class{snpclone} or
\link{matrix}, object.}

\item{strata}{a formula specifying the strata to be used to convert x to a
genclone object if x is a genind object. Defaults to NULL. See details.}

\item{tree}{a text string or function that can calculate a tree from a 
distance matrix. Defaults to "upgma". Note that you must load the package 
with the function for it to work.}

\item{distance}{a character or function defining the distance to be applied 
to x. Defaults to \code{\link{nei.dist}}.}

\item{sample}{An integer representing the number of bootstrap replicates 
Default is 100.}

\item{cutoff}{An integer from 0 to 100 setting the cutoff value to return the
bootstrap values on the nodes. Default is 0.}

\item{showtree}{If \code{TRUE} (Default), a dendrogram will be plotted. If 
\code{FALSE}, nothing will be plotted.}

\item{missing}{any method to be used by \code{\link{missingno}}: "mean" 
(default), "zero", "loci", "genotype", or "ignore".}

\item{mcutoff}{a value between 0 (default) and 1 defining the percentage of 
tolerable missing data if the \code{missing} parameter is set to "loci" or 
"genotype". This should only be set if the distance metric can handle 
missing data.}

\item{quiet}{if \code{FALSE} (default), a progress bar will be printed to 
screen.}

\item{root}{is the tree rooted? This is a parameter passed off to 
\code{\link[ape]{boot.phylo}}. If the \code{tree} parameter returns a 
rooted tree (like UPGMA), this should be \code{TRUE}, otherwise (like 
neighbor-joining), it should be false. When set to \code{NULL} (default), 
the tree is considered rooted if \code{\link[ape]{is.ultrametric}} is true.}

\item{...}{any parameters to be passed off to the distance method.}
}
\value{
an object of class \code{\link[ape]{phylo}}.
}
\description{
Calculate a dendrogram with bootstrap support using any distance applicable 
to genind or genclone objects.
}
\details{
This function utilizes an internal class called 
  \code{\linkS4class{bootgen}} that allows bootstrapping of objects that 
  inherit the genind class. This is necessary due to the fact that columns in
  the genind matrix are defined as alleles and are thus interrelated. This 
  function will specifically bootstrap loci so that results are biologically 
  relevant. With this function, the user can also define a custom distance to
  be performed on the genind or genclone object.
  
  \subsection{the strata argument}{
  There is an argument called \code{strata}. This argument is useful for when
  you want to bootstrap by populations from a \code{\link[adegenet]{genind}}
  object. When you specify strata, the genind object will be converted to
  \code{\link[adegenet]{genpop}} with the specified strata.
  }
}
\note{
\code{\link{prevosti.dist}} and \code{\link{diss.dist}} are exactly the
  same, but \code{\link{diss.dist}} scales better for large numbers of 
  individuals (n > 125) at the cost of required memory. \subsection{missing 
  data}{Missing data is not allowed by many of the distances. Thus, one of 
  the first steps of this function is to treat missing data by setting it to 
  the average allele frequency in the data set. If you are using a distance 
  that can handle missing data (Prevosti's distance), you can set 
  \code{missing = "ignore"} to allow the distance function to handle any 
  missing data. See \code{\link{missingno}} for details on missing 
  data.}\subsection{Bruvo's Distance}{While calculation of Bruvo's distance 
  is possible with this function, it is optimized in the function 
  \code{\link{bruvo.boot}}.}
}
\examples{

data(nancycats)
nan9 <- popsub(nancycats, 9)

set.seed(9999)
# Generate a tree using nei's distance
neinan <- aboot(nan9, dist = nei.dist)

set.seed(9999)
# Generate a tree using custom distance
bindist <- function(x) dist(tab(x), method = "binary")
binnan <- aboot(nan9, dist = bindist)

\dontrun{
# Distances from other packages.
#
# Sometimes, distance functions from other packages will have the constraint
# that the incoming data MUST be genind. Internally, aboot uses the 
# bootgen class ( class?bootgen ) to shuffle loci, and will throw an error
# The function bootgen2genind helps fix that. Here's an example of a function
# that expects a genind class from above
bindist <- function(x){
  stopifnot(is.genind(x))
  dist(tab(x), method = "binary")
}
#
# Fails:
# aboot(nan9, dist = bindist)
## Error: is.genind(x) is not TRUE
#
# Add bootgen2genind to get it working!
# Works:
aboot(nan9, dist = function(x) bootgen2genind(x) \%>\% bindist)

# AFLP data
data(Aeut)

# Nei's distance
anei <- aboot(Aeut, dist = nei.dist, sample = 1000, cutoff = 50)

# Rogers' distance
arog <- aboot(Aeut, dist = rogers.dist, sample = 1000, cutoff = 50)

# This can also be run on genpop objects
strata(Aeut) <- other(Aeut)$population_hierarchy[-1]
Aeut.gc <- as.genclone(Aeut)
setPop(Aeut.gc) <- ~Pop/Subpop
Aeut.pop <- genind2genpop(Aeut.gc)
set.seed(5000)
aboot(Aeut.pop, sample = 1000) # compare to Grunwald et al. 2006

# You can also use the strata argument to convert to genpop inside the function.
set.seed(5000)
aboot(Aeut.gc, strata = ~Pop/Subpop, sample = 1000)

# And genlight objects 
# From glSim:
## 1,000 non structured SNPs, 100 structured SNPs
x <- glSim(100, 1e3, n.snp.struc=100, ploid=2)
aboot(x, distance = bitwise.dist)

# Utilizing other tree methods

library("ape")

aboot(Aeut.pop, tree = fastme.bal, sample = 1000)

# Utilizing options in other tree methods

myFastME <- function(x) fastme.bal(x, nni = TRUE, spr = FALSE, tbr = TRUE)
aboot(Aeut.pop, tree = myFastME, sample = 1000)

}
}
\seealso{
\code{\link{nei.dist}} \code{\link{edwards.dist}} 
  \code{\link{rogers.dist}} \code{\link{reynolds.dist}} 
  \code{\link{prevosti.dist}} \code{\link{diss.dist}} 
  \code{\link{bruvo.boot}} \code{\link[ape]{boot.phylo}} 
  \code{\link[adegenet]{dist.genpop}} \code{\link{dist}}
  \code{\link{bootgen2genind}} \code{\linkS4class{bootgen}}
}
\keyword{bootstrap}

