% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/create_agent.R
\name{create_agent}
\alias{create_agent}
\title{Create a \strong{pointblank} \emph{agent} object}
\usage{
create_agent(
  tbl = NULL,
  read_fn = NULL,
  tbl_name = NULL,
  label = NULL,
  actions = NULL,
  end_fns = NULL,
  embed_report = FALSE,
  lang = NULL,
  locale = NULL
)
}
\arguments{
\item{tbl}{The input table. This can be a data frame, a tibble, a \code{tbl_dbi}
object, or a \code{tbl_spark} object. Alternatively, a function can be used to
read in the input data table with the \code{read_fn} argument (in which case,
\code{tbl} can be \code{NULL}).}

\item{read_fn}{A function that's used for reading in the data. Even if a
\code{tbl} is provided, this function will be invoked to obtain the data (i.e.,
the \code{read_fn} takes priority). There are two ways to specify a \code{read_fn}:
(1) using a function (e.g., \verb{function() \{ <table reading code> \}}) or, (2)
with an R formula expression (e.g., \verb{~ \{ <table reading code> \}}).}

\item{tbl_name}{A optional name to assign to the input table object. If no
value is provided, a name will be generated based on whatever information
is available. This table name will be displayed in the header area of the
agent report generated by printing the \emph{agent} or calling
\code{\link[=get_agent_report]{get_agent_report()}}.}

\item{label}{An optional label for the validation plan. If no value is
provided, a label will be generated based on the current system time.
Markdown can be used here to make the label more visually appealing (it
will appear in the header area of the agent report).}

\item{actions}{A option to include a list with threshold levels so that all
validation steps can react accordingly when exceeding the set levels. This
is to be created with the \code{\link[=action_levels]{action_levels()}} helper function. Should an
action levels list be used for a specific validation step, the default set
specified here will be overridden.}

\item{end_fns}{A list of function calls that should be performed at the end
of an interrogation. Each function call should be in the form of a
one-sided R formula expression, so overall this construction should be
used: \verb{end_fns = list(~ <R statements>, ~ <R statements>, ...)}. An example
of a function that can be sensibly used here is \code{\link[=email_blast]{email_blast()}}, where an
email of the validation report is generated and sent based on sending
condition.}

\item{embed_report}{An option to embed a \strong{gt}-based validation report into
the \code{ptblank_agent} object. If \code{FALSE} (the default) then the table object
will be not generated and available with the \emph{agent} upon returning from
the interrogation.}

\item{lang}{The language to use for automatic creation of briefs (short
descriptions for each validation step) and for the \emph{agent report} (a
summary table that provides the validation plan and the results from the
interrogation. By default, \code{NULL} will create English (\code{"en"}) text. Other
options include French (\code{"fr"}), German (\code{"de"}), Italian (\code{"it"}), Spanish
(\code{"es"}), Portuguese, (\code{"pt"}), Chinese (\code{"zh"}), and Russian (\code{"ru"}).}

\item{locale}{An optional locale ID to use for formatting values in the
\emph{agent report} summary table according the locale's rules. Examples include
\code{"en_US"} for English (United States) and \code{"fr_FR"} for French (France);
more simply, this can be a language identifier without a country
designation, like "es" for Spanish (Spain, same as \code{"es_ES"}).}
}
\value{
A \code{ptblank_agent} object.
}
\description{
The \code{create_agent()} function creates an \emph{agent} object, which is used in a
\emph{data quality reporting} workflow. The overall aim of this workflow is to
generate useful reporting information for assessing the level of data quality
for the target table. We can supply as many validation functions as the user
wishes to write, thereby increasing the level of validation coverage for that
table. The \emph{agent} assigned by the \code{create_agent()} call takes validation
functions, which expand to validation steps (each one is numbered). This
process is known as developing a \emph{validation plan}.

The validation functions, when called on an \emph{agent}, are merely instructions
up to the point the \code{\link[=interrogate]{interrogate()}} function is called. That kicks off the
process of the \emph{agent} acting on the \emph{validation plan} and getting results
for each step. Once the interrogation process is complete, we can say that
the \emph{agent} has intel. Calling the \emph{agent} itself will result in a reporting
table. This reporting of the interrogation can also be accessed with the
\code{\link[=get_agent_report]{get_agent_report()}} function, where there are more reporting options.
}
\details{
A very detailed list object, known as the x-list, can be obtained by
using the \code{\link[=get_agent_x_list]{get_agent_x_list()}} function on the \emph{agent}. This font of
information can be taken as a whole, or, broken down by the step number (with
the \code{i} argument).

Sometimes it is useful to see which rows were the failing ones. By using the
\code{\link[=get_data_extracts]{get_data_extracts()}} function on the \emph{agent}, we either get a list of
tibbles (for those steps that have data extracts) or one tibble if the
validation step is specified with the \code{i} argument.

If we just need to know whether all validations completely passed (i.e., all
steps had no failing test units), the \code{\link[=all_passed]{all_passed()}} function could be used
on the \emph{agent}. However, in practice, it's not often the case that all data
validation steps are free from any failing units.
}
\section{Figures}{

\if{html}{\figure{man_create_agent_1.png}{options: width=100\%}}
}

\section{Function ID}{

1-2
}

\examples{
# Let's walk through a data quality
# analysis of an extremely small table;
# it's actually called `small_table` and
# we can find it as a dataset in this
# package
small_table

# We ought to think about what's
# tolerable in terms of data quality so
# let's designate proportional failure
# thresholds to the `warn`, `stop`, and
# `notify` states using `action_levels()`
al <- 
  action_levels(
      warn_at = 0.10,
      stop_at = 0.25,
    notify_at = 0.35
  )

# Now create a pointblank `agent` object
# and give it the `al` object (which
# serves as a default for all validation
# steps which can be overridden); the
# static thresholds provided by `al` will
# make the reporting a bit more useful
agent <- 
  create_agent(
    read_fn = ~ small_table,
    label = "An example.",
    actions = al
  )

# Then, as with any `agent` object, we
# can add steps to the validation plan by
# using as many validation functions as we
# want; then, we use `interrogate()` to
# physically perform the validations and
# gather intel
agent <-
  agent \%>\% 
  col_exists(vars(date, date_time)) \%>\%
  col_vals_regex(
    vars(b), "[0-9]-[a-z]{3}-[0-9]{3}"
  ) \%>\%
  rows_distinct() \%>\%
  col_vals_gt(vars(d), 100) \%>\%
  col_vals_lte(vars(c), 5) \%>\%
  col_vals_equal(
    vars(d), vars(d),
    na_pass = TRUE
  ) \%>\%
  col_vals_between(
    vars(c),
    left = vars(a), right = vars(d),
    na_pass = TRUE
  ) \%>\%
  interrogate()
  
# Calling `agent` in the console
# prints the agent's report; but we
# can get a `gt_tbl` object directly
# with `get_agent_report(agent)`
report <- get_agent_report(agent)
class(report)

# What can you do with the report?
# Print it from an R Markdown code
# chunk, use it in a **blastula** email,
# put it in a webpage, or further
# modify it with the **gt** package

# From the report we know that Step
# 4 had two test units (rows, really)
# that failed; we can see those rows
# with `get_data_extracts()` 
agent \%>\% get_data_extracts(i = 4)

# We can get an x-list for the whole
# validation (8 steps), or, just for
# the 4th step with `get_agent_x_list()`
xl_step_4 <-
  agent \%>\% get_agent_x_list(i = 4)
 
# And then we can peruse the different
# parts of the list; let's get the
# fraction of test units that failed
xl_step_4$f_failed

# Just printing the x-list will tell
# us what's available therein
xl_step_4

# An x-list not specific to any step
# will have way more information and a
# slightly different structure; see
# `help(get_agent_x_list)` for more info
# get_agent_x_list(agent)

}
\seealso{
Other Planning and Prep: 
\code{\link{action_levels}()},
\code{\link{col_schema}()},
\code{\link{create_informant}()},
\code{\link{db_tbl}()},
\code{\link{scan_data}()},
\code{\link{validate_rmd}()}
}
\concept{Planning and Prep}
