#
#                _         _    _      _                _    
#               (_)       | |  | |    | |              | |   
#  _ __    ___   _  _ __  | |_ | |__  | |  __ _  _ __  | | __
# | '_ \  / _ \ | || '_ \ | __|| '_ \ | | / _` || '_ \ | |/ /
# | |_) || (_) || || | | || |_ | |_) || || (_| || | | ||   < 
# | .__/  \___/ |_||_| |_| \__||_.__/ |_| \__,_||_| |_||_|\_\
# | |                                                        
# |_|                                                        
# 
# This file is part of the 'rich-iannone/pointblank' package.
# 
# (c) Richard Iannone <riannone@me.com>
# 
# For full copyright and license information, please look at
# https://rich-iannone.github.io/pointblank/LICENSE.html
#


#' Write an *agent* and *informant* to a **pointblank** YAML file
#' 
#' @description 
#' With `yaml_write()` we can take an existing *agent* and write that *agent*'s
#' validation plan to a YAML file. With **pointblank** YAML, we can modify the
#' YAML markup if so desired, or, use as is to create a new agent with the
#' [yaml_read_agent()] function. That *agent* will have a validation plan and is
#' ready to [interrogate()] the data. We can go a step further and perform an
#' interrogation directly from the YAML file with the [yaml_agent_interrogate()]
#' function. That returns an agent with intel (having already interrogated the
#' target data table). An *informant* object can also be written to YAML with
#' `yaml_write()`.
#'
#' One requirement for writing the *agent* to YAML is that we need to have a
#' table-reading function (`read_fn`) specified (it's a function that is used to
#' read the target table when [interrogate()] is called). This option can be set
#' when using [create_agent()] or with [set_read_fn()] (for use with an existing
#' *agent*).
#' 
#' @param agent An *agent* object of class `ptblank_agent`.
#' @param informant An *informant* object of class `ptblank_informant`.
#' @param filename The name of the YAML file to create on disk. It is
#'   recommended that either the `.yaml` or `.yml` extension be used for this
#'   file.
#' @param path An optional path to which the YAML file should be saved (combined
#'   with `filename`).
#'   
#' @examples
#' # Let's go through the process of
#' # developing an agent with a validation
#' # plan (to be used for the data quality
#' # analysis of the `small_table` dataset),
#' # and then offloading that validation
#' # plan to a pointblank YAML file
#' 
#' # We ought to think about what's
#' # tolerable in terms of data quality so
#' # let's designate proportional failure
#' # thresholds to the `warn`, `stop`, and
#' # `notify` states using `action_levels()`
#' al <- 
#'   action_levels(
#'     warn_at = 0.10,
#'     stop_at = 0.25,
#'     notify_at = 0.35
#'   )
#' 
#' # Now create a pointblank `agent` object
#' # and give it the `al` object (which
#' # serves as a default for all validation
#' # steps which can be overridden); the
#' # data will be referenced in a `read_fn`
#' # (a requirement for writing to YAML)
#' agent <- 
#'   create_agent(
#'     read_fn = ~small_table,
#'     label = "A simple example with the `small_table`.",
#'     actions = al
#'   )
#' 
#' # Then, as with any `agent` object, we
#' # can add steps to the validation plan by
#' # using as many validation functions as we
#' # want
#' agent <-
#'   agent %>% 
#'   col_exists(vars(date, date_time)) %>%
#'   col_vals_regex(
#'     vars(b), "[0-9]-[a-z]{3}-[0-9]{3}"
#'   ) %>%
#'   rows_distinct() %>%
#'   col_vals_gt(vars(d), 100) %>%
#'   col_vals_lte(vars(c), 5)
#'
#' # The agent can be written to a pointblank
#' # YAML file with `yaml_write()`
#' # yaml_write(
#' #   agent = agent,
#' #   filename = "agent-small_table.yml"
#' # )
#' 
#' # The 'agent-small_table.yml' file is
#' # available in the package through
#' # `system.file()`
#' yml_file <- 
#'   system.file(
#'     "agent-small_table.yml",
#'     package = "pointblank"
#'   )
#' 
#' # We can view the YAML file in the console
#' # with the `yaml_agent_string()` function
#' yaml_agent_string(path = yml_file)
#' 
#' # The YAML can also be printed in the console
#' # by supplying the agent as the input
#' yaml_agent_string(agent = agent)
#' 
#' # At a later time, the YAML file can
#' # be read into a new agent with the
#' # `yaml_read_agent()` function
#' agent <- 
#'   yaml_read_agent(path = yml_file)
#' 
#' class(agent)
#' 
#' # We can interrogate the data (which
#' # is accessible through the `read_fn`)
#' # with `interrogate()` and get an
#' # agent with intel, or, we can
#' # interrogate directly from the YAML
#' # file with `yaml_agent_interrogate()`
#' agent <- 
#'   yaml_agent_interrogate(path = yml_file)
#' 
#' class(agent)
#' 
#' @family pointblank YAML
#' @section Function ID:
#' 9-1
#' 
#' @export
yaml_write <- function(agent = NULL,
                       informant = NULL,
                       filename,
                       path = NULL) {

  if (!is.null(path)) {
    filename <- file.path(path, filename)
  }
  
  if (is.null(agent) && is.null(informant)) {
    stop("An agent or informant object must be supplied to `yaml_write()`.",
         call. = FALSE)
  }
  
  if (!is.null(agent) && !is.null(informant)) {
    x <- c(as_agent_yaml_list(agent), as_informant_yaml_list(informant))
    # TODO: manage conflicts between both YAML representations
    
  } else if (!is.null(agent)) {
    x <- as_agent_yaml_list(agent)
  } else {
    x <- as_informant_yaml_list(informant)
  }
  
  yaml::write_yaml(
    x = x,
    file = filename,
    handlers = list(
      logical = function(x) {
        result <- ifelse(x, "true", "false")
        class(result) <- "verbatim"
        result
      }
    )
  )
}

#' Display **pointblank** YAML using an agent or a YAML file
#' 
#' With **pointblank** YAML, we can serialize an agent's validation plan (with
#' [yaml_write()]), read it back later with a new agent (with
#' [yaml_read_agent()]), or perform an interrogation on the target data table
#' directly with the YAML file (with [yaml_agent_interrogate()]). The
#' `yaml_agent_string()` function allows us to inspect the YAML generated by
#' [yaml_write()] in the console, giving us a look at the YAML without needing
#' to open the file directly. Alternatively, we can provide an *agent* to the
#' `yaml_agent_string()` and view the YAML representation of the validation plan
#' without needing to write the YAML to disk beforehand.
#'
#' @param agent An *agent* object of class `ptblank_agent`.
#' @param path A path to a YAML file that specifies a validation plan for an
#'   *agent*.
#'   
#' @examples 
#' # Let's create a validation plan for the
#' # data quality analysis of the `small_table`
#' # dataset; we need an agent and its
#' # table-reading function enables retrieval
#' # of the target table
#' agent <- 
#'   create_agent(
#'     read_fn = ~small_table,
#'     label = "A simple example with the `small_table`.",
#'     actions = action_levels(
#'       warn_at = 0.10,
#'       stop_at = 0.25,
#'       notify_at = 0.35
#'     )
#'   ) %>%
#'   col_exists(vars(date, date_time)) %>%
#'   col_vals_regex(
#'     vars(b), "[0-9]-[a-z]{3}-[0-9]{3}"
#'   ) %>%
#'   rows_distinct() %>%
#'   col_vals_gt(vars(d), 100) %>%
#'   col_vals_lte(vars(c), 5)
#'
#' # We can view the YAML file in the console
#' # with the `yaml_agent_string()` function,
#' # providing the `agent` object as the input
#' yaml_agent_string(agent = agent)
#'
#' # The agent can be written to a pointblank
#' # YAML file with `yaml_write()`
#' # yaml_write(
#' #   agent = agent,
#' #   filename = "agent-small_table.yml"
#' # )
#' 
#' # The 'agent-small_table.yml' file is
#' # available in the package through `system.file()`
#' yml_file <- 
#'   system.file(
#'     "agent-small_table.yml",
#'     package = "pointblank"
#'   )
#' 
#' # The `yaml_agent_string()` function can
#' # be used with the YAML file as well
#' yaml_agent_string(path = yml_file)
#' 
#' # At a later time, the YAML file can
#' # be read into a new agent with the
#' # `yaml_read_agent()` function
#' agent <- yaml_read_agent(path = yml_file)
#' class(agent)
#'   
#' @family pointblank YAML
#' @section Function ID:
#' 9-5
#' 
#' @export
yaml_agent_string <- function(agent = NULL,
                              path = NULL) {
  
  if (is.null(agent) && is.null(path)) {
    stop(
      "An `agent` object or a `path` to a YAML file must be specified.",
      call. = FALSE
    )
  }
  
  if (!is.null(agent) && !is.null(path)) {
    stop("Only one of `agent` or `path` should be specified.", call. = FALSE)
  }
  
  if (!is.null(agent)) {
    
    message(
      as_agent_yaml_list(agent) %>%
        yaml::as.yaml(
          handlers = list(
            logical = function(x) {
              result <- ifelse(x, "true", "false")
              class(result) <- "verbatim"
              result
            }
          )
        )
    )
    
  } else {
    message(readLines(path) %>% paste(collapse = "\n"))
  }
}

as_vars_fn <- function(columns) {
  paste0("vars(", columns, ")")
}

as_list_preconditions <- function(preconditions) {
  if (is.null(preconditions[[1]])) {
    return(NULL)
  } else {
    return(as.character(preconditions))
  }
}

to_list_action_levels <- function(actions) {
  
  agent_actions <- actions
  agent_actions[sapply(agent_actions, is.null)] <- NULL
  agent_actions$fns[sapply(agent_actions$fns, is.null)] <- NULL
  
  if (length(agent_actions$fns) == 0) agent_actions$fns <- NULL
  
  if (length(agent_actions$fns) == 0) {
    agent_actions$fns <- NULL
  } else {
    agent_actions$fns <-
      lapply(
        agent_actions$fns,
        FUN = function(x) {
          if (!is.null(x)) x %>% as.character() %>% paste(collapse = "")
        }
      )
  }
  
  list(actions = agent_actions)
}

as_action_levels <- function(actions, actions_default = NULL) {

  agent_actions <- actions
  agent_actions[sapply(agent_actions, is.null)] <- NULL
  agent_actions$fns[sapply(agent_actions$fns, is.null)] <- NULL
  
  if (length(agent_actions$fns) == 0) {
    agent_actions$fns <- NULL
  } else {
    agent_actions$fns <-
      lapply(
        agent_actions$fns,
        FUN = function(x) {
          if (!is.null(x)) x %>% as.character() %>% paste(collapse = "")
        }
      )
  }

  if (!is.null(actions_default)) {
    if (identical(agent_actions, actions_default)) {
      return(NULL)
    }
  }
  
  agent_actions
}

get_schema_list <- function(schema) {

  vals <- schema

  complete <- schema$`__complete__`
  in_order <- schema$`__in_order__`
  is_exact <- schema$`__is_exact__`
  
  type <- ifelse(inherits(schema, "r_type"), "r", "sql")
  
  vals <- 
    vals[!(names(vals) %in% c("__complete__", "__in_order__", "__is_exact__"))]
  
  if (type == "sql") {
    vals <- c(vals, list(`.db_col_types` = "sql"))
  }

  list(
    schema = vals, 
    complete = complete,
    in_order = in_order,
    is_exact = is_exact
  )
}

to_list_read_fn <- function(read_fn) {
  list(read_fn = paste(as.character(read_fn), collapse = ""))
}

to_list_label <- function(label) {
  list(label = label)
}

to_list_tbl_name <- function(tbl_name) {
  list(tbl_name = tbl_name)
}

get_arg_value <- function(value) {
  
  if (inherits(value, "list") && inherits(value[[1]], "quosures")) {
    out <- paste0("vars(", rlang::as_label(value[[1]][[1]]), ")")
  } else if (inherits(value, "list") && inherits(value[[1]], "numeric")) {
    out <- value[[1]] 
  } else {
    out <- as.character(value[[1]])
  }
  
  out
}

get_arg_value_lr <- function(value) {

  if (inherits(value, "quosure")) {
    out <- paste0("vars(", rlang::as_label(value), ")")
  } else if (inherits(value, "numeric")) {
    out <- value
  } else {
    out <- as.character(value)
  }
  
  out
}

prune_lst_step <- function(lst_step) {
  
  if ("preconditions" %in% names(lst_step[[1]]) &&
      is.null(lst_step[[1]][["preconditions"]])) {
    lst_step[[1]]["preconditions"] <- NULL
  }
  if ("na_pass" %in% names(lst_step[[1]]) &&
      !lst_step[[1]][["na_pass"]]) {
    lst_step[[1]]["na_pass"] <- NULL
  }
  if ("active" %in% names(lst_step[[1]]) &&
      lst_step[[1]][["active"]]) {
    lst_step[[1]]["active"] <- NULL
  }
  if ("complete" %in% names(lst_step[[1]]) &&
      lst_step[[1]][["complete"]]) {
    lst_step[[1]]["complete"] <- NULL
  }
  if ("in_order" %in% names(lst_step[[1]]) &&
      lst_step[[1]][["in_order"]]) {
    lst_step[[1]]["in_order"] <- NULL
  }
  if ("is_exact" %in% names(lst_step[[1]]) &&
      lst_step[[1]][["is_exact"]]) {
    lst_step[[1]]["is_exact"] <- NULL
  }
  if ("actions" %in% names(lst_step[[1]])) {
    if (length(lst_step[[1]][["actions"]][["action_levels"]]) == 1 &&
        length(lst_step[[1]][["actions"]][["action_levels"]][["fns"]]) == 0) {
      
      lst_step[[1]][["actions"]] <- NULL
    } else if (is.null(lst_step[[1]][["actions"]])) {
      lst_step[[1]][["actions"]] <- NULL
    }
  }
  
  lst_step
}

as_agent_yaml_list <- function(agent) {

  if (is.null(agent$read_fn)) {
    stop(
      "The agent must have a `read_fn` value to transform it into YAML.",
       call. = FALSE
    )
  }

  action_levels_default <- as_action_levels(agent$actions)
  end_fns <- agent$end_fns %>% unlist()
  
  lst_label <- to_list_label(agent$label)
  lst_tbl_name <- to_list_tbl_name(agent$tbl_name)
  lst_read_fn <- to_list_read_fn(agent$read_fn)
  
  if (is.null(action_levels_default)) {
    lst_action_levels <- NULL
  } else {
    lst_action_levels <- list(actions = action_levels_default)
  }
  
  if (is.null(end_fns)) {
    lst_end_fns <- NULL
  } else {
    lst_end_fns <- list(end_fns = as.character(end_fns))
  }
  
  if (is.null(agent$embed_report) || 
      (!is.null(agent$embed_report) && !agent$embed_report)) {
    lst_embed_report <- NULL
  } else {
    lst_embed_report <- list(embed_report = agent$embed_report)
  }

  if (is.null(agent$lang) || 
      (!is.null(agent$lang) && agent$lang == "en")) {
    lst_lang <- NULL
  } else {
    lst_lang <- list(lang = agent$lang)
  }
  
  if (is.null(agent$locale)) {
    lst_locale <- NULL
  } else {
    lst_locale <- list(locale = agent$locale)
  }

  # Select only the necessary columns from the agent's `validation_set` 
  agent_validation_set <- 
    agent$validation_set %>% 
    dplyr::select(
      assertion_type, column, values, na_pass,
      preconditions, actions, brief, active
    )
  
  all_steps <- list()
  
  for (i in seq_len(nrow(agent_validation_set))) {
    
    step_list <- agent_validation_set[i, ] %>% as.list()
    
    validation_fn <- step_list$assertion_type
    
    if (validation_fn %in% c(
      "col_vals_lt", "col_vals_lte",
      "col_vals_equal", "col_vals_not_equal",
      "col_vals_gte", "col_vals_gt"
    )) {
      
      lst_step <- 
        list(
          validation_fn = list(
            columns = as_vars_fn(step_list$column[[1]]),
            value = get_arg_value(step_list$values),
            na_pass = step_list$na_pass,
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(
              step_list$actions[[1]],
              action_levels_default
            ),
            active = step_list$active
          )
        )
      
    } else if (grepl("between", validation_fn)) {

      lst_step <- 
        list(
          validation_fn = list(
            columns = as_vars_fn(step_list$column[[1]]),
            left = get_arg_value_lr(step_list$values[[1]][[1]]),
            right = get_arg_value_lr(step_list$values[[1]][[2]]),
            inclusive = as.logical(
              c(
                names(step_list$values[[1]][1]),
                names(step_list$values[[1]][1])
              )
            ),
            na_pass = step_list$na_pass,
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(
              step_list$actions[[1]],
              action_levels_default
            ),
            active = step_list$active
          )
        )
      
    } else if (grepl("in_set", validation_fn)) {

      lst_step <- 
        list(
          validation_fn = list(
            columns = as_vars_fn(step_list$column[[1]]),
            set = step_list$values[[1]],
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(
              step_list$actions[[1]],
              action_levels_default
            ),
            active = step_list$active
          )
        )

    } else if (grepl("null", validation_fn)) {
      
      lst_step <- 
        list(
          validation_fn = list(
            columns = as_vars_fn(step_list$column[[1]]),
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(
              step_list$actions[[1]],
              action_levels_default
            ),
            active = step_list$active
          )
        )
      
    } else if (validation_fn == "col_vals_regex") {

      lst_step <- 
        list(
          validation_fn = list(
            columns = as_vars_fn(step_list$column[[1]]),
            regex = get_arg_value(step_list$values),
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(
              step_list$actions[[1]],
              action_levels_default
            ),
            active = step_list$active
          )
        )
      
    } else if (grepl("col_is_", validation_fn) ||
               validation_fn == "col_exists") {

      lst_step <- 
        list(
          validation_fn = list(
            columns = as_vars_fn(step_list$column[[1]]),
            actions = as_action_levels(
              step_list$actions[[1]],
              action_levels_default
            ),
            active = step_list$active
          )
        )
      
    } else if (validation_fn == "col_vals_expr") {

      lst_step <- 
        list(
          validation_fn = list(
            expr = paste0("~", rlang::as_label(step_list$values[[1]])),
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(
              step_list$actions[[1]],
              action_levels_default
            ),
            active = step_list$active
          )
        )
      
    } else if (validation_fn == "rows_distinct") {

      if (is.na(step_list$column[[1]][[1]])) {
        vars_cols <- NULL
      } else {
        vars_cols <- as_vars_fn(step_list$column[[1]])
      }
      
      lst_step <- 
        list(
          validation_fn = list(
            columns = vars_cols,
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(
              step_list$actions[[1]],
              action_levels_default
            ),
            active = step_list$active
          )
        )
    
    } else if (validation_fn == "col_schema_match") {
      
      schema_list <- get_schema_list(schema = step_list$values[[1]])

      lst_step <- 
        list(
          validation_fn = list(
            schema = schema_list$schema,
            complete = schema_list$complete,
            in_order = schema_list$in_order,
            is_exact = schema_list$is_exact,
            actions = as_action_levels(
              step_list$actions[[1]],
              action_levels_default
            ),
            active = step_list$active
          )
        )
      
    } else if (validation_fn == "conjointly") {
      
      lst_step <- 
        list(
          validation_fn = list(
            fns = as.character(step_list$values[[1]]),
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(
              step_list$actions[[1]],
              action_levels_default
            ),
            active = step_list$active
          )
        )
    }

    # Remove list elements that are representative of defaults
    lst_step <- prune_lst_step(lst_step)

    # Set the top level list-element name to that of
    # the validation function
    names(lst_step) <- validation_fn
    all_steps <- c(all_steps, list(lst_step))
  }
  
  c(
    lst_read_fn,
    lst_tbl_name,
    lst_label,
    lst_action_levels,
    lst_end_fns,
    lst_embed_report,
    lst_lang,
    lst_locale,
    list(steps = all_steps)
  )
}

as_informant_yaml_list <- function(informant) {

  if (is.null(informant$read_fn)) {
    stop(
      "The informant must have a `read_fn` value to transform it into YAML.",
      call. = FALSE
    )
  }
  
  lst_read_fn <- to_list_read_fn(informant$read_fn)
  
  if (length(informant$meta_snippets) > 0) {
    
    lst_meta_snippets <- 
      list(
        meta_snippets = 
          lapply(
            informant$meta_snippets,
            FUN = function(x) {
              paste(as.character(x), collapse = "")
            }
          )
      )
  } else {
    lst_meta_snippets <- NULL
  }
  
  if (is.null(informant$lang) || 
      (!is.null(informant$lang) && informant$lang == "en")) {
    lst_lang <- NULL
  } else {
    lst_lang <- list(lang = informant$lang)
  }
  
  if (is.null(informant$locale)) {
    lst_locale <- NULL
  } else {
    lst_locale <- list(locale = informant$locale)
  }
  
  c(
    lst_read_fn,
    lst_lang,
    lst_locale,
    lst_meta_snippets,
    informant$metadata
  )
}
