skip_on_cran()

# Create a pointblank agent for testing
agent <-
  create_agent(
    tbl = ~ small_table,
    actions = action_levels(stop_at = 0.1)
  ) %>%
  col_vals_gt(
    vars(date_time), vars(date),
    na_pass = TRUE
  ) %>%
  col_vals_gt(
    vars(b), vars(g), na_pass = TRUE,
    label = "b > g"
  ) %>%
  col_is_character(
    vars(b, f),
    label = "Verifying character-type columns" 
  ) %>%
  rows_distinct(
    vars(d, e, f),
    label = "Distinct rows across 'd', 'e', and 'f'"
  ) %>%
  col_is_integer(
    vars(a),
    label = "`a` must be an integer",
    active = FALSE
  ) %>%
  interrogate()

work_path <- "./generated_testthat_files"

if (fs::dir_exists(path = work_path)) {
  fs::dir_delete(path = work_path)
}

fs::dir_create(path = work_path)

expect_fixed_match <- function(fulltext, text) {
  expect_match(fulltext, text, fixed = TRUE)
}

test_that("a testthat test file can be written using an agent", {
  
  sub_dir <- fs::dir_create(path = "./generated_testthat_files")
  
  write_testthat_file(
    agent = agent,
    name = "testthat",
    path = sub_dir
  )
  
  expect_true(
    fs::file_exists(path = fs::path(work_path, "test-testthat.R"))
  )
  
  testthat_file <- 
    readLines(con = fs::path(work_path, "test-testthat.R")) %>%
    paste0(collapse = "\n")
  
  expect_fixed_match(testthat_file,
    "# Generated by pointblank"
  )
  expect_fixed_match(testthat_file,
    "tbl <- small_table"
  )
  expect_fixed_match(testthat_file,
    "test_that(\"values in `date_time` should be > `date`\", {"
  )
  expect_fixed_match(testthat_file,
    "test_that(\"values in `b` should be > `g`\", {"
  )
  expect_fixed_match(testthat_file,
    "test_that(\"column `b` is of type: character\", {"
  )
  expect_fixed_match(testthat_file,
    "test_that(\"column `f` is of type: character\", {"
  )
  expect_fixed_match(testthat_file,
    "test_that(\"entirely distinct rows across `d, e, f`\", {"
  )
  expect_fixed_match(testthat_file,
    "test_that(\"column `a` is of type: integer\", {"
  )
  
  # Expect an error if a `read_fn` is not available in agent
  expect_error(
    write_testthat_file(
      agent = remove_read_fn(agent),
      name = "testthat",
      path = sub_dir
    )
  )
  
  # Expect an error if the supplied path does not exist
  expect_error(
    write_testthat_file(
      agent = agent,
      name = "testthat",
      path = "does/not/exist"
    )
  )
  
})

if (fs::dir_exists(path = work_path)) {
  fs::dir_delete(path = work_path)
}

test_that("the `process_skips_text()` creates the correct text strings", {
  
  skips_keywords <-
    c(
      "cran", "travis", "appveyor", "ci", "covr", "bioc",
      "windows", "mac", "linux", "solaris"
    )
  
  skips_text <-
    c(
      "skip_on_cran()\n\n", "skip_on_travis()\n\n", "skip_on_appveyor()\n\n", 
      "skip_on_ci()\n\n", "skip_on_covr()\n\n", "skip_on_bioc()\n\n", 
      "skip_on_os(\"windows\")\n\n", "skip_on_os(\"mac\")\n\n",
      "skip_on_os(\"linux\")\n\n", "skip_on_os(\"solaris\")\n\n"
    )
  
  # Expect that the above keywords generate the corresponding text
  for (i in seq_along(skips_keywords)) {
    expect_equal(process_skips_text(skips_keywords[i]), skips_text[i])
  }
  
  # Expect no text (`NULL`) if certain inputs are provided
  expect_null(process_skips_text(NULL))
  expect_null(process_skips_text(3))
  
  # Expect an error if invalid keywords are supplied
  expect_error(process_skips_text("crane"))
  expect_error(process_skips_text(c("cran", "covr", "crane")))
})
