% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/specially.R
\name{specially}
\alias{specially}
\alias{expect_specially}
\alias{test_specially}
\title{Perform a specialized validation with a user-defined function}
\usage{
specially(
  x,
  fn,
  preconditions = NULL,
  actions = NULL,
  step_id = NULL,
  label = NULL,
  brief = NULL,
  active = TRUE
)

expect_specially(object, fn, preconditions = NULL, threshold = 1)

test_specially(object, fn, preconditions = NULL, threshold = 1)
}
\arguments{
\item{x}{A data frame, tibble (\code{tbl_df} or \code{tbl_dbi}), Spark DataFrame
(\code{tbl_spark}), or, an \emph{agent} object of class \code{ptblank_agent} that is
created with \code{\link[=create_agent]{create_agent()}}.}

\item{fn}{A function that performs the specialized validation on the data. It
must either return a logical vector or a table where the last column is a
logical column.}

\item{preconditions}{An optional expression for mutating the input table
before proceeding with the validation. This can either be provided as a
one-sided R formula using a leading \code{~} (e.g.,
\code{~ . \%>\% dplyr::mutate(col = col + 10)} or as a function (e.g.,
\code{function(x) dplyr::mutate(x, col = col + 10)}. See the \emph{Preconditions}
section for more information.}

\item{actions}{A list containing threshold levels so that the validation step
can react accordingly when exceeding the set levels. This is to be created
with the \code{\link[=action_levels]{action_levels()}} helper function.}

\item{step_id}{One or more optional identifiers for the single or multiple
validation steps generated from calling a validation function. The use of
step IDs serves to distinguish validation steps from each other and provide
an opportunity for supplying a more meaningful label compared to the step
index. By default this is \code{NULL}, and \strong{pointblank} will automatically
generate the step ID value (based on the step index) in this case. One or
more values can be provided, and the exact number of ID values should (1)
match the number of validation steps that the validation function call will
produce (influenced by the number of \code{columns} provided), (2) be an ID
string not used in any previous validation step, and (3) be a vector with
unique values.}

\item{label}{An optional label for the validation step. This label appears in
the \emph{agent} report and for the best appearance it should be kept short.}

\item{brief}{An optional, text-based description for the validation step. If
nothing is provided here then an \emph{autobrief} is generated by the \emph{agent},
using the language provided in \code{\link[=create_agent]{create_agent()}}'s \code{lang} argument (which
defaults to \code{"en"} or English). The \emph{autobrief} incorporates details of the
validation step so it's often the preferred option in most cases (where a
\code{label} might be better suited to succinctly describe the validation).}

\item{active}{A logical value indicating whether the validation step should
be active. If the validation function is working with an \emph{agent}, \code{FALSE}
will make the validation step inactive (still reporting its presence and
keeping indexes for the steps unchanged). If the validation function will
be operating directly on data (no \emph{agent} involvement), then any step with
\code{active = FALSE} will simply pass the data through with no validation
whatsoever. Aside from a logical vector, a one-sided R formula using a
leading \code{~} can be used with \code{.} (serving as the input data table) to
evaluate to a single logical value. With this approach, the \strong{pointblank}
function \code{\link[=has_columns]{has_columns()}} can be used to determine whether to make a
validation step active on the basis of one or more columns existing in the
table (e.g., \code{~ . \%>\% has_columns(vars(d, e))}). The default for \code{active}
is \code{TRUE}.}

\item{object}{A data frame, tibble (\code{tbl_df} or \code{tbl_dbi}), or Spark
DataFrame (\code{tbl_spark}) that serves as the target table for the expectation
function or the test function.}

\item{threshold}{A simple failure threshold value for use with the
expectation (\code{expect_}) and the test (\code{test_}) function variants. By
default, this is set to \code{1} meaning that any single unit of failure in data
validation results in an overall test failure. Whole numbers beyond \code{1}
indicate that any failing units up to that absolute threshold value will
result in a succeeding \strong{testthat} test or evaluate to \code{TRUE}. Likewise,
fractional values (between \code{0} and \code{1}) act as a proportional failure
threshold, where \code{0.15} means that 15 percent of failing test units results
in an overall test failure.}
}
\value{
For the validation function, the return value is either a
\code{ptblank_agent} object or a table object (depending on whether an agent
object or a table was passed to \code{x}). The expectation function invisibly
returns its input but, in the context of testing data, the function is
called primarily for its potential side-effects (e.g., signaling failure).
The test function returns a logical value.
}
\description{
The \code{specially()} validation function allows for custom validation with a
function that \emph{you} provide. The major proviso for the provided function is
that it must either return a logical vector or a table where the final column
is logical. The function will operate on the table object, or, because you
can do whatever you like, it could also operate on other types of objects. To
do this, you can transform the input table in \code{preconditions} or inject an
entirely different object there. During interrogation, there won't be any
checks to ensure that the data is a table object.
}
\section{Preconditions}{

Providing expressions as \code{preconditions} means \strong{pointblank} will preprocess
the target table during interrogation as a preparatory step. It might happen
that a particular validation requires a calculated column, some filtering of
rows, or the addition of columns via a join, etc. Especially for an
\emph{agent}-based report this can be advantageous since we can develop a large
validation plan with a single target table and make minor adjustments to it,
as needed, along the way. Within \code{specially()}, because this function is
special, there won't be internal checking as to whether the
\code{preconditions}-based output is a table.

The table mutation is totally isolated in scope to the validation step(s)
where \code{preconditions} is used. Using \strong{dplyr} code is suggested here since
the statements can be translated to SQL if necessary (i.e., if the target
table resides in a database). The code is most easily supplied as a one-sided
\strong{R} formula (using a leading \code{~}). In the formula representation, the \code{.}
serves as the input data table to be transformed (e.g., \code{~ . \%>\% dplyr::mutate(col_b = col_a + 10)}). Alternatively, a function could instead
be supplied (e.g., \code{function(x) dplyr::mutate(x, col_b = col_a + 10)}).
}

\section{Actions}{

Often, we will want to specify \code{actions} for the validation. This argument,
present in every validation function, takes a specially-crafted list
object that is best produced by the \code{\link[=action_levels]{action_levels()}} function. Read that
function's documentation for the lowdown on how to create reactions to
above-threshold failure levels in validation. The basic gist is that you'll
want at least a single threshold level (specified as either the fraction of
test units failed, or, an absolute value), often using the \code{warn_at}
argument. This is especially true when \code{x} is a table object because,
otherwise, nothing happens. For the \verb{col_vals_*()}-type functions, using
\code{action_levels(warn_at = 0.25)} or \code{action_levels(stop_at = 0.25)} are good
choices depending on the situation (the first produces a warning when a
quarter of the total test units fails, the other \code{stop()}s at the same
threshold level).
}

\section{Briefs}{

Want to describe this validation step in some detail? Keep in mind that this
is only useful if \code{x} is an \emph{agent}. If that's the case, \code{brief} the agent
with some text that fits. Don't worry if you don't want to do it. The
\emph{autobrief} protocol is kicked in when \code{brief = NULL} and a simple brief will
then be automatically generated.
}

\section{YAML}{

A \strong{pointblank} agent can be written to YAML with \code{\link[=yaml_write]{yaml_write()}} and the
resulting YAML can be used to regenerate an agent (with \code{\link[=yaml_read_agent]{yaml_read_agent()}})
or interrogate the target table (via \code{\link[=yaml_agent_interrogate]{yaml_agent_interrogate()}}). When
\code{specially()} is represented in YAML (under the top-level \code{steps} key as a
list member), the syntax closely follows the signature of the validation
function. Here is an example of how a complex call of \code{specially()} as a
validation step is expressed in R code and in the corresponding YAML
representation.

R statement:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent \%>\% 
  specially(
    fn = function(x) \{ ... \},
    preconditions = ~ . \%>\% dplyr::filter(a < 10),
    actions = action_levels(warn_at = 0.1, stop_at = 0.2), 
    label = "The `specially()` step.",
    active = FALSE
  )
}\if{html}{\out{</div>}}

YAML representation:

\if{html}{\out{<div class="sourceCode yaml">}}\preformatted{steps:
- specially:
    fn: function(x) \{ ... \}
    preconditions: ~. \%>\% dplyr::filter(a < 10)
    actions:
      warn_fraction: 0.1
      stop_fraction: 0.2
    label: The `specially()` step.
    active: false
}\if{html}{\out{</div>}}

In practice, both of these will often be shorter as only the expressions for
validation steps are necessary. Arguments with default values won't be
written to YAML when using \code{\link[=yaml_write]{yaml_write()}} (though it is acceptable to include
them with their default when generating the YAML by other means). It is also
possible to preview the transformation of an agent to YAML without any
writing to disk by using the \code{\link[=yaml_agent_string]{yaml_agent_string()}} function.
}

\section{Examples}{


For all examples here, we'll use a simple table with three numeric columns
(\code{a}, \code{b}, and \code{c}). This is a very basic table but it'll be more useful when
explaining things later.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl <-
  dplyr::tibble(
    a = c(5, 2, 6),
    b = c(3, 4, 6),
    c = c(9, 8, 7)
  )
  
tbl
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## # A tibble: 3 × 3
##       a     b     c
##   <dbl> <dbl> <dbl>
## 1     5     3     9
## 2     2     4     8
## 3     6     6     7
}\if{html}{\out{</div>}}
\subsection{A: Using an \code{agent} with validation functions and then \code{interrogate()}}{

Validate that the target table has exactly three rows. This single validation
with \code{specially()} has 1 test unit since the function executed on \code{x} (the
target table) results in a logical vector with a length of 1. We'll determine
if this validation has any failing test units (there is 1 test unit).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent <-
  create_agent(tbl = tbl) \%>\%
  specially(fn = function(x) nrow(x) == 3) \%>\%
  interrogate()
}\if{html}{\out{</div>}}

Printing the \code{agent} in the console shows the validation report in the
Viewer. Here is an excerpt of validation report, showing the single entry
that corresponds to the validation step demonstrated here.

\if{html}{
\out{
<img src="https://raw.githubusercontent.com/rich-iannone/pointblank/main/images/man_specially_1.png" style="width:100\%;">
}
}
}

\subsection{B: Using the validation function directly on the data (no \code{agent})}{

This way of using validation functions acts as a data filter. Data is passed
through but should \code{stop()} if there is a single test unit failing. The
behavior of side effects can be customized with the \code{actions} option.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl \%>\% specially(fn = function(x) nrow(x) == 3)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## # A tibble: 3 × 3
##       a     b     c
##   <dbl> <dbl> <dbl>
## 1     5     3     9
## 2     2     4     8
## 3     6     6     7
}\if{html}{\out{</div>}}
}

\subsection{C: Using the expectation function}{

With the \verb{expect_*()} form, we would typically perform one validation at a
time. This is primarily used in \strong{testthat} tests.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{expect_specially(tbl, fn = function(x) nrow(x) == 3)
}\if{html}{\out{</div>}}
}

\subsection{D: Using the test function}{

With the \verb{test_*()} form, we should get a single logical value returned to
us.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl \%>\% test_specially(fn = function(x) nrow(x) == 3)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] TRUE
}\if{html}{\out{</div>}}
}

\subsection{Variations}{

We can do more complex things with \code{specially()} and its variants.

Check the class of the target table.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl \%>\% 
  test_specially(
    fn = function(x) \{
      inherits(x, "data.frame")
    \}
  )
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] TRUE
}\if{html}{\out{</div>}}

Check that the number of rows in the target table is less than \code{small_table}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl \%>\% 
  test_specially(
    fn = function(x) \{
      nrow(x) < nrow(small_table)
    \}
  )
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] TRUE
}\if{html}{\out{</div>}}

Check that all numbers across all numeric column are less than \code{10}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl \%>\% 
  test_specially(
    fn = function(x) \{
      (x \%>\% 
         dplyr::select(where(is.numeric)) \%>\%
         unlist()
      ) < 10
    \}
  )
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] TRUE
}\if{html}{\out{</div>}}

Check that all values in column \code{c} are greater than b and greater than \code{a}
(in each row) and always less than \code{10}. This creates a table with the new
column \code{d} which is a logical column (that is used as the evaluation of test
units).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl \%>\% 
  test_specially(
    fn = function(x) \{
      x \%>\%
        dplyr::mutate(
          d = c > b & c > a & c < 10
        )
    \}
  )
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] TRUE
}\if{html}{\out{</div>}}

Check that the \code{game_revenue} table (which is not the target table) has
exactly 2000 rows.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl \%>\% 
  test_specially(
    fn = function(x) \{
      nrow(game_revenue) == 2000
    \}
  )
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] TRUE
}\if{html}{\out{</div>}}
}
}

\section{Function ID}{

2-36
}

\seealso{
Other validation functions: 
\code{\link{col_count_match}()},
\code{\link{col_exists}()},
\code{\link{col_is_character}()},
\code{\link{col_is_date}()},
\code{\link{col_is_factor}()},
\code{\link{col_is_integer}()},
\code{\link{col_is_logical}()},
\code{\link{col_is_numeric}()},
\code{\link{col_is_posix}()},
\code{\link{col_schema_match}()},
\code{\link{col_vals_between}()},
\code{\link{col_vals_decreasing}()},
\code{\link{col_vals_equal}()},
\code{\link{col_vals_expr}()},
\code{\link{col_vals_gte}()},
\code{\link{col_vals_gt}()},
\code{\link{col_vals_in_set}()},
\code{\link{col_vals_increasing}()},
\code{\link{col_vals_lte}()},
\code{\link{col_vals_lt}()},
\code{\link{col_vals_make_set}()},
\code{\link{col_vals_make_subset}()},
\code{\link{col_vals_not_between}()},
\code{\link{col_vals_not_equal}()},
\code{\link{col_vals_not_in_set}()},
\code{\link{col_vals_not_null}()},
\code{\link{col_vals_null}()},
\code{\link{col_vals_regex}()},
\code{\link{col_vals_within_spec}()},
\code{\link{conjointly}()},
\code{\link{row_count_match}()},
\code{\link{rows_complete}()},
\code{\link{rows_distinct}()},
\code{\link{serially}()},
\code{\link{tbl_match}()}
}
\concept{validation functions}
