% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hessian.R
\name{Hessian}
\alias{Hessian}
\title{Numerical cross-derivatives with parallel capabilities}
\usage{
Hessian(
  FUN,
  x,
  side = 0,
  acc.order = 2,
  h = NULL,
  symmetric = TRUE,
  h0 = NULL,
  control = list(),
  f0 = NULL,
  cores = 1,
  preschedule = TRUE,
  func = NULL,
  report = 1L,
  ...
)
}
\arguments{
\item{FUN}{A function returning a numeric scalar.
If the function returns a vector, the output will be is a Jacobian.
If instead of \code{FUN}, \code{func} is passed, as in \code{numDeriv::grad},
it will be reassigned to \code{FUN} with a warning.}

\item{x}{Numeric vector or scalar: point at which the derivative is estimated.
\code{FUN(x)} must return a finite value.}

\item{side}{Integer scalar or vector indicating difference type:
\code{0} for central, \code{1} for forward, and \code{-1} for backward differences.
Central differences are recommended unless computational cost is prohibitive.}

\item{acc.order}{Integer specifying the desired accuracy order.
The error typically scales as \eqn{O(h^{\mathrm{acc.order}})}{O(h^acc.order)}.}

\item{h}{Numeric scalar, vector, or character specifying the step size for the numerical
difference. If character (\code{"CR"}, \code{"CRm"}, \code{"DV"}, or \code{"SW"}),
calls \code{gradstep()} with the appropriate step-selection method.
Must be length 1 or match \code{length(x)}. Matrices of step sizes are not
supported. Suggestions how to handle all pairs of coordinates are welcome.}

\item{symmetric}{Logical: if \code{TRUE}, then, almost halves computation time
by exploiting Hessian symmetry.}

\item{h0}{Numeric scalar of vector: initial step size for automatic search with
\code{gradstep()}.}

\item{control}{A named list of tuning parameters passed to \code{gradstep()}.}

\item{f0}{Optional numeric scalar or vector: if provided and applicable, used
where the stencil contains zero (i.e. \code{FUN(x)} is part of the sum)
to save time.
TODO: Currently ignored.}

\item{cores}{Integer specifying the number of parallel processes to use. Recommended
value: the number of physical cores on the machine minus one.}

\item{preschedule}{Logical: if \code{TRUE}, enables pre-scheduling for \code{mclapply()}
and disables load balancing with \code{parLapplyLB()}. Minimises overhead at the cost
of potentially unequal loads at the end of a job. Recommended for functions that
take less than 0.1 s per evaluation.}

\item{func}{Deprecated; for \code{numDeriv::grad()} compatibility only.}

\item{report}{Integer: if \code{0}, returns a gradient without any attributes; if \code{1},
attaches the step size and its selection method: \code{2} or higher, attaches the full
diagnostic output (overrides \code{diagnostics = FALSE} in \code{control}).}

\item{...}{Additional arguments passed to \code{FUN}.}
}
\value{
Depends on the output of \code{FUN}. If \code{FUN} returns a scalar:
returns a gradient vector matching the length of \code{x}. If \code{FUN} returns a vector:
returns a Jacobian matrix with dimensions \code{length(FUN(x)), length(x)}.
Unlike the output of \code{numDeriv::grad} and \code{numDeriv::jacobian},
this output preserves the names of \code{x} and \code{FUN(x)}.
}
\description{
Computes the derivative of a function with respect to two different arguments.
Arbitrary accuracies and sides for different coordinates
of the argument vector are supported.
}
\details{
The optimal step size for 2nd-order-accurate central-differences-based Hessians
is of the order Mach.eps^(1/4)
to balance the Taylor series truncation error with the rounding error.
However, selecting the best step size typically requires knowledge
of higher-order cross derivatives and is highly technically involved. Future releases
will allow character arguments to invoke automatic data-driven step-size selection.

The use of \code{f0} can reduce computation time similar to the use of \code{f.lower}
and \code{f.upper} in \code{uniroot()}.
}
\examples{
f <- function(x) prod(sin(x))
Hessian(f, 1:4)
# Large matrices
\donttest{
  system.time(Hessian(f, 1:100))
}
}
\seealso{
\code{\link[=gradstep]{gradstep()}} for automatic step-size selection.
}
