% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ps_dissim.R
\name{ps_dissim}
\alias{ps_dissim}
\title{Quantitative phylogenetic dissimilarity}
\usage{
ps_dissim(
  ps,
  method = "sorensen",
  fun = c("vegdist", "designdist", "chaodist"),
  endemism = FALSE,
  normalize = FALSE,
  ...
)
}
\arguments{
\item{ps}{phylospatial object.}

\item{method}{Character indicating the dissimilarity index to use:
\itemize{
\item "sorensen": Sorensen's dissimilarity, a.k.a. Bray-Curtis distance (the default)
\item "sorensen_turnover": The turnover component of Sorensen's dissimilarity, a.k.a. Simpson's.
\item "sorensen_nestedness": The nestedness component of Sorensen's dissimilarity.
\item Any other valid \code{method} passed to \code{fun}. For options, see the documentation for those functions.
}}

\item{fun}{Character indicating which general distance function from the \code{vegan} library to use: "\link[vegan]{vegdist}"
(the default), "\link[vegan]{designdist}", or "\link[vegan]{chaodist}". (While these functions are not explicitly
designed to calculate phylogenetic beta diversity, their use here incorporates the phylogenetic components.)
This argument is ignored if one of the three "sorensen" methods is selected.}

\item{endemism}{Logical indicating whether community values should be divided by column totals (taxon range sizes)
to derive endemism before computing distances.}

\item{normalize}{Logical indicating whether community values should be divided by row totals (community sums) before
computing distances. If \code{TRUE}, dissimilarity is based on proportional community composition. Normalization is
applied after endemism.}

\item{...}{Additional arguments passed to \code{fun}.}
}
\value{
A pairwise phylogenetic dissimilarity matrix of class \code{dist}.
}
\description{
This function calculates pairwise phylogenetic dissimilarity between communities. It works with both binary and
quantitative community data sets. A wide range of phylogentic community dissimilarity metrics are supported,
including phylogenetic Sorensen's and Jaccard's distances, turnover and nestedness components of Sorensen's distance
(Baselga & Orme, 2012), and phylogenetic versions of all community distance indices provided through the \code{vegan} library.
The function also includes options to scale the community matrix in order to focus the analysis on endemism and/or
on proportional differences in community composition. The results from this function can be visualized using
\link{ps_rgb} or \link{ps_regions}, or used in a variety of statistical analyses.
}
\examples{
# example data set:
ps <- ps_simulate(n_tips = 50)

# The default arguments give Sorensen's quantitative dissimilarity index
# (a.k.a. Bray-Curtis distance):
d <- ps_dissim(ps)

# Specifying a custom formula explicitly via `designdist`;
# (this is the Bray-Curtis formula, so it's equivalent to the prior example)
d <- ps_dissim(ps, method = "(b+c)/(2*a+b+c)",
      fun = "designdist", terms = "minimum", abcd = TRUE)

# Alternative arguments can specify a wide range of dissimilarity measures;
# here's endemism-weighted Jaccard's dissimilarity:
d <- ps_dissim(ps, method = "jaccard", endemism = TRUE)

}
\references{
Graham, C. H., & Fine, P. V. (2008). Phylogenetic beta diversity: linking ecological and evolutionary
processes across space in time. Ecology Letters, 11(12), 1265-1277.

Baselga, A., & Orme, C. D. L. (2012). betapart: an R package for the study of beta diversity. Methods in
Ecology and Evolution, 3(5), 808-812.

Pavoine, S. (2016). A guide through a family of phylogenetic dissimilarity measures among sites.
Oikos, 125(12), 1719-1732.
}
\seealso{
\code{\link[=ps_add_dissim]{ps_add_dissim()}}
}
