#' plot_phenology plots the phenology.
#' @title Plot the phenology from a result.
#' @author Marc Girondot
#' @return Return a list of outputs. For each site:\cr
#' $site=name of the site\cr
#' estimate1=Estimation of counts not taking into account the observations\cr
#' sd1=The SD of estimation1\cr
#' estimate2=Estimation of counts taking into account the observations\cr
#' sd2=The SD of estimation2
#' CI_min=The lower limit for confidence interval
#' CI_max=The upper limit for confidence interval
#' @param result A result file generated by fit_phenology
#' @param data A dataset generated by add_format
#' @param pdf TRUE or FALSE, indicates if a pdf file is generated.
#' @param parametersfixed Set of fixed parameters
#' @param parameters Set of parameters to be changed
#' @param series Number of series to be analyzed or 'all'
#' @param ... Parameters used for graphics (xlab, ylab, ylim, cex and pch)
#' @param moon If TRUE, the moon phase is ploted. Default is FALSE
#' @param replicate.CI Number of replicates for estimation of confidence interval.
#' @param help If TRUE, an help is displayed
#' @description The function "plot_phenology" plots the phenology graph from a result.
#' @examples
#' library(phenology)
#' # Read a file with data
#' \dontrun{
#' Gratiot<-read.delim("http://max2.ese.u-psud.fr/epc/conservation/BI/Complete.txt", header=FALSE)
#' }
#' data(Gratiot)
#' # Generate a formatted list nammed data_Gratiot 
#' data_Gratiot<-add_format(origin=NULL, add=Gratiot, name="Complete", reference=as.Date("2001-01-01"), format="%d/%m/%Y")
#' # Generate initial points for the optimisation
#' parg<-par_init(data_Gratiot, parametersfixed=NULL)
#' # Run the optimisation
#' \dontrun{
#' result_Gratiot<-fit_phenology(data=data_Gratiot, parametersfit=parg, parametersfixed=NULL, trace=1)
#' }
#' data(result_Gratiot)
#' # Plot the phenology and get some stats
#' output<-plot_phenology(result=result_Gratiot, pdf=FALSE)
#' @export



plot_phenology <-
function(result=NULL, ..., data=NULL, parameters=NULL, parametersfixed=NA, pdf=FALSE, 
	series="all", moon=FALSE, replicate.CI=1000, help=FALSE) {

if(help || (is.null(data) && is.null(result))) {
	cat("The syntaxe of this function is :\n")
	cat("plot_phenology(result) or \n")
	cat("plot_phenology(result=res, series=x, pdf=TRUE, ...)\n")
	cat("with res being the output of fit_phenology.\n")
	cat("series is used to indicate the series to plot.\n")
	cat("If only the data must be printed, use :\n")
	cat("plot_phenology(data=dataset, series=x, pdf=TRUE, ...)\n")
	cat("x can be a vector with all the series to be printed, for example\n")
	cat("plot_phenology(result=res, series=c(1,3,4), pdf=TRUE, ...)\n")
	cat("x can be 'all' to print all the series at one time. In this\n")
	cat("case pdf should be TRUE if graphs are necessary.\n")
	cat("Optional parameters ... are:\n")
	cat("xlab: Label for X axis [default Months]\n")
	cat("ylab: Label for Y axis [default Number]\n")
	cat("ylim: maximum value for Y axis [default estimated from data]\n")
	cat("cex: size of the points [default 0.5]\n")
	cat("pch: symbol of the points [default 16]\n")
	
} else {

out <- list()

pnp<-as.list(c(...))

if (is.null(data))  {data<-result$data}

if (series[1]=="all") {series <- c(1:length(data))}

series <- as.numeric(series)

for(kseries in 1:length(series)) {

reference<-data[[series[kseries]]]$Date[1]-(data[[series[kseries]]]$ordinal[1]+1)

if (!is.null(result)) {
	# si il y en a pas estimable, je les mets à 0
	res_se<-result$se

	## je stocke les données ajustées dans parres
	parres<-result$par
	parametersfixed<-result$parametersfixed
	
} else {
	# je n ai pas de result, je lis parameters
	parres<-parameters
	res_se<-0
}


res_se[is.na(res_se)]<-0


# length(result$par) donne le nombre de paramètres pour lesquels on a result
# length(parametersfixed) donne le nombre de paramètres fixés
# je passe les parametersfixed en revue pour savoir si le sd# est fourni pour ce paramètre

if (!all(is.na(parametersfixed))) {

for(i in 1:length(parametersfixed)) {
	nm<-names(parametersfixed[i])
# est-ce que j ai un sd pour ce paramètre et le paramètre
# si oui je stocke les deux
# sinon, je mets le sd à 0

	if (substr(nm,1,3)!="sd#")	{
		parres<-c(parres, parametersfixed[i])
		if (is.na(parametersfixed[paste("sd#", nm, sep="")])) {
# je n'ai pas de sd pour ce paramètre
			res_se<-c(res_se, 0)
		} else {
# j'ai un sd pour ce paramètre		
			res_se<-c(res_se, parametersfixed[paste("sd#", nm, sep="")])
		}
	}
}

}

## crée un tableau dans lequel on met ce qu'on va utiliser pour le graph
val=matrix(c(0:364, rep(NA, 365), rep(NA, 365), rep(NA, 365), rep(NA, 365), rep(NA, 365), rep(NA, 365)), ncol=7)

## On le nomme, c'est plus simple
colnames(val)=c("days", "Obs", "Theor", "Theor-2SE", "Theor+2SE", "Theor-2SD", "Theor+2SD")

if (is.null(data)) {nmser=""} else {
	nmser<-names(data[series[kseries]])
# 2012-06-03 Je rentre les valeurs avec un nombre
	xxx <- data[series[kseries]][[1]][,"ordinal"]
	val[xxx,"Obs"] <- data[series[kseries]][[1]][,"nombre"]
	
	# 2012-06-03 Je rentre les valeurs avec une incertitude
	for (xjkk in 1:length(data[series[kseries]][[1]][,"ordinal2"])) {
		if (!is.na(data[series[kseries]][[1]][xjkk,"ordinal2"])) {
			xxx <- seq(from=data[series[kseries]][[1]][xjkk,"ordinal"], to=data[series[kseries]][[1]][xjkk,"ordinal2"])
			val[xxx,"Obs"] <-data[series[kseries]][[1]][xjkk,"nombre"]/length(xxx)
		}
	}

	
}

if (!is.null(parres)) {

## crée un tableau avec des réplicats des modèles
par2=matrix(rep(NA, length(parres)*replicate.CI), ncol=length(parres))

## On nomme les colonnes comme les noms des paramètres
colnames(par2)=names(parres)


# on générère replicate.CI jeux de paramètres à partir des paramètres obtenus en vraisemblance
# ici il faut que je prenne aussi ceux transmis en parmetersfixed
for(i in 1:length(parres)) par2[,i]=rnorm(replicate.CI, mean=as.numeric(parres[i]), sd=res_se[i])

## je génère une matrice de replicate.CI saisons de pontes
## note que je ne gère pas les années bissextiles
ponte2=matrix(rep(NA, 365*replicate.CI), ncol=365)

# affiche le nom de la série
cat("\n", nmser, "\n", sep="")

cat("\n")

pb<-txtProgressBar(min=1, max=replicate.CI, style=3)



## je remplis les replicate.CI saisons de ponte
for(j in 1:replicate.CI) {

setTxtProgressBar(pb, j)

# j'ai tous les paramètres dans xpar
# maintenant tous les paramètre fixés appraissent dans resfit
	xparec <- .format_par(par2[j,], nmser)
	
	
	ponte2[j,1:365]=.daily_count(1:365, xparec, print=FALSE)
	
	
# je viens de générer les pontes du jour j
}

## je calcule les écart-types des nb de pontes générées chaque jour
sd2<-apply(ponte2, 2, sd)

mnponte<-mean(apply(ponte2, 1, sum))
sdponte<-sd(apply(ponte2, 1, sum))
out1<-c(estimate1=mnponte, sd1=sdponte)


# dans ponte2[nbsimul 1:replicate.CI, jour 1:365] j'ai la donnée théorique
for(i in 1:dim(data[[series[kseries]]])[1]) {
		if (!is.na(data[[series[kseries]]]$ordinal2[i])) {
			for(j in (1+data[[series[kseries]]]$ordinal[i]):data[[series[kseries]]]$ordinal2[i]) {
					ponte2[1:replicate.CI, j]<-0
			}
		}
}

for(i in 1:dim(data[[series[kseries]]])[1]) {
	ponte2[1:replicate.CI, data[[series[kseries]]]$ordinal[i]]<-data[[series[kseries]]]$nombre[i]
}



mnponte<-mean(apply(ponte2, 1, sum))
sdponte<-sd(apply(ponte2, 1, sum))
out1<-c(out1, estimate2=mnponte, sd2=sdponte)

# 20 mai 2012
out1 <- c(out1, CI_Min=max(mnponte-2*sdponte, sum(data[[series[kseries]]]$nombre)), CI_Max=mnponte+2*sdponte)

## je remplis le tableau val avec les nb théoriques

xparec <- .format_par(parres, nmser)

val[1:365, "Theor"]=.daily_count(1:365, xparec, print=FALSE)

## je remplis le tableau val avec les nb théoriques +/- 2 SD
for(i in 1:365) {val[i, "Theor-2SE"]=max(0, val[i, "Theor"]-2*sd2[i])}
val[1:365, "Theor+2SE"]=val[1:365, "Theor"]+2*sd2[1:365]


## je calcule la distribution théorique des points minimaux
for(i in 1:365) {val[i, "Theor-2SD"]=max(c(subset(0:trunc(3*val[i, "Theor"]), pnbinom(0:trunc(3*val[i, "Theor"]), size=abs(as.numeric(xparec["Theta"])), mu=val[i, "Theor"])<=0.05), 0))}

## je calcule la distribution théoriques des points maximaux
for(i in 1:365) {val[i, "Theor+2SD"]=min(c(subset(0:trunc(3*val[i, "Theor"]), pnbinom(0:trunc(3*val[i, "Theor"]), size=abs(as.numeric(xparec["Theta"])), mu=val[i, "Theor"])>=0.95), trunc(3*val[i, "Theor"])+1))}


}

vmaxx<-c(reference, reference+364)



if (is.null(pnp$ylim)) {

	if ((!is.null(data)) && (!is.null(parres))) {
		vmaxy<-c(0, max(val[, "Theor+2SD"], data[[series[kseries]]]$nombre[(is.na(data[[series[kseries]]]$ordinal2)) & (!is.na(data[[series[kseries]]]$nombre))]))
	} else {
		if (!is.null(data)) {
			vmaxy<-c(0, max(data[[series[kseries]]]$nombre[(is.na(data[[series[kseries]]]$ordinal2)) & (!is.na(data[[series[kseries]]]$nombre))]))
		} else {
			vmaxy<-c(0, max(val[, "Theor+2SD"]))
		}
	}
} else {

	vmaxy<-ifelse(length(pnp$ylim)==1, c(0, pnp$ylim), pnp$ylim)
}

if (vmaxy[2]==0) vmaxy[2] <- 0.1

x<-seq(from=reference, to=reference+364, by="1 day")


xlab<-ifelse(is.null(pnp$xlab), "Months", pnp$xlab)
ylab<-ifelse(is.null(pnp$ylab), "Number", pnp$ylab)
pch<-ifelse(is.null(pnp$pch), 16, pnp$pch)
cex<-ifelse(is.null(pnp$cex), 0.5, pnp$cex)

if (moon) {
	moony<-vmaxy[2]*1.06
	mp<-moon_phase(x, phase=TRUE)
	mpT1<-ifelse((mp!="FM") | (is.na(mp)), FALSE, TRUE)
	mpT2<-ifelse((mp!="NM") | (is.na(mp)), FALSE, TRUE)
#	mpT3<-ifelse((mp!="FQ") | (is.na(mp)), FALSE, TRUE)
#	mpT4<-ifelse((mp!="LQ") | (is.na(mp)), FALSE, TRUE)
}

cpt <- 1
if (pdf) cpt <- 2

for (ncpt in 1:cpt) {

if (ncpt==2) {pdf(paste(names(data[series[kseries]]),".pdf", sep=""))}


## je fais les graphiques
## Pour les dates seules
par(new=FALSE);

plot(x, rep(0, 365) , type="n", xlim=vmaxx, ylim=vmaxy, bty="n", xlab=xlab, ylab=ylab)

if (moon) {
	points(x[mpT1], rep(moony, length(x[mpT1])), cex=1, bg="black", col="black", pch=21, xpd=TRUE)
	points(x[mpT2], rep(moony, length(x[mpT2])), cex=1, bg="white", col="black", pch=21, xpd=TRUE)
#	points(x[mpT3], rep(moony, length(x[mpT3])), cex=3, bg="black", col="black", pch=21)	
#	points(x[mpT3]+8, rep(moony, length(x[mpT3])), cex=3, bg="white", col="white", pch=21)
#	points(x[mpT4], rep(moony, length(x[mpT4])), cex=3, bg="black", col="black", pch=21)	
#	points(x[mpT4]-8, rep(moony, length(x[mpT4])), cex=3, bg="white", col="white", pch=21)
}

par(new=TRUE);


if (!is.null(data)) {
plot(data[[series[kseries]]]$Date[is.na(data[[series[kseries]]]$Date2)], data[[series[kseries]]]$nombre[is.na(data[[series[kseries]]]$Date2)] , 
	type="p", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes=FALSE, bty="n", cex=cex, col="black", pch=pch)


## Pour les dates avec incertitudes
par(new=TRUE);
for(i in 1:dim(data[[series[kseries]]])[1]) {
	if (!is.na(data[[series[kseries]]]$ordinal2[i])) {
		x0<-data[[series[kseries]]]$Date[i]
		x1<-data[[series[kseries]]]$Date2[i]
		lgt01<-as.numeric(data[[series[kseries]]]$Date2[i]-data[[series[kseries]]]$Date[i]+1)
		y0<-data[[series[kseries]]]$nombre[i]/lgt01
		y1<-y0
		segments(x0, y0, x1=x1, y1=y1, col="green", lwd=2)
	}
}



par(new=TRUE);
}

if (!is.null(parres)) {

plot((reference+val[, "days"]),val[, "Theor"] , type="l", xlim=vmaxx, ylim=vmaxy,  xlab="", ylab="", axes = FALSE, bty="n");
par(new=TRUE);
plot((reference+val[, "days"]),val[, "Theor-2SE"] , type="l", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes = FALSE, lty=2, bty="n");
par(new=TRUE);
plot((reference+val[, "days"]),val[, "Theor+2SE"],  type="l", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes = FALSE, lty=2, bty="n");
par(new=TRUE);
plot((reference+val[, "days"]),val[, "Theor-2SD"],  type="l", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes = FALSE, lty=2, bty="n", col="red");
par(new=TRUE);
plot((reference+val[, "days"]), val[, "Theor+2SD"], type="l", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes = FALSE, lty=2, bty="n", col="red");
}

mtext(names(data[series[kseries]]), side=3, line=1)

if (ncpt==2) dev.off()

}




if (!is.null(parres)) {

out2 <- list(list(estimates=out1, values=val))
names(out2) <- nmser

out <- c(out, out2)

}
mtext(names(data[series[kseries]]), side=3, line=1)


}

}

	growlnotify('Plot is done!')
	
	class(out) <- "phenologyout"
	
	return(out)

}
