% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit-ns.r
\name{fit.ns}
\alias{fit.ns}
\title{Fitting a Neyman-Scott point process model}
\usage{
fit.ns(points, lims, R, disp = "gaussian", child.dist = "pois",
  child.info = NULL, sibling.list = NULL, edge.correction = "pbc",
  start = NULL, bounds = NULL, use.bobyqa = FALSE, trace = FALSE)
}
\arguments{
\item{points}{A matrix or list of matrices containing locations of
observed points, where each row corresponds to a point and each
column corresponds to a dimension. If a list, then the patterns
are assumed to be independent and a single process is fitted to
all.}

\item{lims}{A matrix or list of matrices with two columns,
corresponding to the upper and lower limits of each dimension,
respectively. If a list, then each matrix provides the limits
for the corresponding pattern in \code{points}.}

\item{R}{Truncation distance for the difference process.}

\item{disp}{A character string indicating the distribution of
children around their parents. Use \code{"gaussian"} for
multivariate normal dispersion with standard deviation
\code{sigma}, or \code{"uniform"} for uniform dispersion within
distance \code{tau} of the parent.}

\item{child.dist}{The distribution of the number of children
generated by a randomly selected parent. For a Poisson
distribution, use \code{"pois"}; for a binomial distribution,
use \code{"binomx"}, where \code{"x"} is replaced by the fixed
value of the number of independent trials (e.g.,
\code{"binom5"} for a Binomial(5, p) distribution, and
\code{"binom50"} for a Binomial(50, p) distribution); and
\code{"twocamera"} for a child distribution appropriate for a
two-camera aerial survey.}

\item{child.info}{A list of further information that is required
about the distribution for the number of children generated by
parents. See `Details'.}

\item{sibling.list}{An optional list that comprises (i) a component
named \code{sibling.mat}, containing a matrix such that the jth
entry in the ith row is \code{TRUE} if the ith and jth points
are known siblings, \code{FALSE} if they are known nonsiblings,
and \code{NA} if their sibling status is not known; (ii) alpha,
providing the probability that a sibling is successfully
identified as a sibling; and (iii) beta, providing the
probability that a nonsibling is successfully identified as a
nonsibling. For multi-pattern fitting, this object must be a
list of such lists, one for each pattern.}

\item{edge.correction}{The method used for the correction of edge
effects. Either \code{"pbc"} for periodic boundary conditions,
or \code{"buffer"} for a buffer-zone correction.}

\item{start}{A named vector of starting values for the model
parameters.}

\item{bounds}{A list with named components. Each component should
be a vector of length two, giving the upper and lower bounds
for the named parameter.}

\item{use.bobyqa}{Logical; if \code{TRUE} the \link{bobyqa} function
is used for optimisation. Otherwise the \link{nlminb} function
is used. Note that \link{bobyqa} seems to be less stable than
\link{nlminb}, but does not require calculation of the Palm
likelihood's partial derivatives.}

\item{trace}{Logical; if \code{TRUE}, parameter values are printed
to the screen for each iteration of the optimisation procedure.}
}
\value{
An R6 reference class object.
}
\description{
Estimates parameters for a Neyman-Scott point process by maximising
the Palm likelihood. This approach was first proposed by Tanaka et
al. (2008) for two-dimensional Thomas processes. Further
generalisations were made by Stevenson, Borchers, and Fewster (in
press) and Jones-Todd et al. (in press).
}
\details{
The parameter \code{D} is the density of parent points, which is
always estimated. Possible additional parameters are
\itemize{
  \item \code{lambda}, the expected number of children generated per
        parent (when \code{child.dist = "pois"}).

\item \code{p}, the proportion of the \code{x} possible children
        that are generated (when \code{child.dist = "binomx"}).

\item \code{kappa}, the average length of the surface phase of a
        diving cetacean (when \code{child.dist = "twocamera"}; see
        Stevenson, Borchers, and Fewster, in press).

\item \code{sigma}, the standard deviation of dispersion along
        each dimension (when \code{disp} = "gaussian").

\item \code{tau}, the maximum distance a child can be from its
        parent (when \code{disp} = "uniform").

}

The \code{"child.info"} argument is required when \code{child.dist}
is set to \code{"twocamera"}. It must be a list that comprises (i) a
component named \code{w}, providing the halfwidth of the detection
zone; (ii) a component named \code{b}, providing the halfwidth of
the survey area; (iii) a component named \code{l}, providing the
time lag between cameras (in seconds); and (iv) a component named
\code{tau}, providing the mean dive-cycle duration. See Stevenson,
Borchers, and Fewster (in press) for details.
}
\examples{
## Fitting model to example data.
fit <- fit.ns(example.2D, lims = rbind(c(0, 1), c(0, 1)), R = 0.5)
## Printing estimates.
coef(fit)
## Plotting the estimated Palm intensity.
plot(fit)
\dontrun{
## Simulating data and fitting additional models.
set.seed(1234)
## One-dimensional Thomas process.
data.thomas <- sim.ns(c(D = 10, lambda = 5, sigma = 0.025), lims = rbind(c(0, 1)))
## Fitting a model to these data.
fit.thomas <- fit.ns(data.thomas$points, lims = rbind(c(0, 1)), R = 0.5)
## Three-dimensional Matern process.
data.matern <- sim.ns(c(D = 10, lambda = 10, tau = 0.1), disp = "uniform",
                      lims = rbind(c(0, 1), c(0, 2), c(0, 3)))
## Fitting a model to these data.
fit.matern <- fit.ns(data.matern$points, lims = rbind(c(0, 1), c(0, 2), c(0, 3)),
                     R = 0.5, disp = "uniform")
}

}
\references{
Jones-Todd, C. M., Caie, P., Illian, J. B., Stevenson,
    B. C., Savage, A., Harrison, D. J., and Bown, J. L. (in
    press). Identifying prognostic structural features in tissue
    sections of colon cancer patients using point pattern
    analysis. \emph{Statistics in Medicine}.

Stevenson, B. C., Borchers, D. L., and Fewster,
    R. M. (in press) Cluster capture-recapture to account for
    identification uncertainty on aerial surveys of animal
    populations. \emph{Biometrics}.

Tanaka, U., Ogata, Y., and Stoyan, D. (2008) Parameter
    estimation and model selection for Neyman-Scott point
    processes. \emph{Biometrical Journal}, \strong{50}: 43--57.
}
\seealso{
Use \link{coef.palm} to extract estimated parameters, and
    \link{plot.palm} to plot the estimated Palm intensity
    function. Use \link{boot.palm} to run a parametric bootstrap,
    allowing calculation of standard errors and confidence
    intervals.

See \link{sim.ns} to simulate from a Neyman-Scott point
    process.
}
