% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parentChild2TaxonTree.R
\name{parentChild2taxonTree}
\alias{parentChild2taxonTree}
\title{Create a Taxonomy-Based Phylogeny ('Taxon Tree') from a Table of Parent-Child Taxon Relationships}
\usage{
parentChild2taxonTree(parentChild, tipSet = "nonParents", cleanTree = TRUE)
}
\arguments{
\item{parentChild}{A two-column matrix of type \code{character} where
each element is a taxon name. Each row represents a parent-child relationship
with first the parent (column 1) taxon name and then the child (column 2).}

\item{tipSet}{This argument controls which taxa are selected as tip taxa for the
output tree. The default \code{tipSet = "nonParents"} selects all child taxa which
are not listed as parents in \code{parentChild}. Alternatively, \code{tipSet = "all"}
will add a tip to every internal node with the parent-taxon name encapsulated in
parentheses.}

\item{cleanTree}{When \code{TRUE} (the default), the tree is run through a series of
post-processing, including having singles collapsed,
nodes reordered and being written out as a Newick string and read
back in, to ensure functionality with ape functions
and ape-derived functions. 
If \code{FALSE}, none of this post-processing is done and
users should beware, as such trees can lead to hard-crashes of R.}
}
\value{
A phylogeny of class \code{phylo}, with tip taxa as
controlled by argument \code{tipSet}.
The output tree is returned with no edge lengths.

The names of higher taxa than the tips should be appended
as the element \code{$node.label} for the internal nodes.
}
\description{
This function takes a two-column matrix of taxon names,
indicating a set of binary parent-taxon:child-taxon 
paired relationships with a common root, and returns
a 'taxonomy-tree' phylogeny object of class \code{phylo}.
}
\details{
All taxa listed must be traceable via their parent-child relationships to a single,
common ancestor which will act as the root node for output phylogeny. Additionally,
the root used will be the parent taxon to all tip taxa closest in terms of parent-child
relationships to the tip taxa: i.e., the most recent common ancestor. Ancestral taxa which
are singular internal nodes that trace to this root are removed, and a message
is printed.
}
\examples{

#let's create a small, really cheesy example
pokexample <- rbind(
    cbind("Squirtadae", c("Squirtle","Blastoise","Wartortle")),
    c("Shelloidea","Lapras"), c("Shelloidea","Squirtadae"),
    c("Pokezooa","Shelloidea"), c("Pokezooa","Parasect"),
    c("Rodentapokemorpha","Linoone"), c("Rodentapokemorpha","Sandshrew"),
    c("Rodentapokemorpha","Pikachu"), c("Hirsutamona","Ursaring"),
    c("Hirsutamona","Rodentapokemorpha"), c("Pokezooa","Hirsutamona")
    )

#Default: tipSet = 'nonParents'
pokeTree <- parentChild2taxonTree(
    parentChild = pokexample,
    tipSet = "nonParents")
plot(pokeTree)
nodelabels(pokeTree$node.label)

#Get ALL taxa as tips with tipSet = 'all'
pokeTree <- parentChild2taxonTree(
    parentChild = pokexample,
    tipSet = "all")
plot(pokeTree)
nodelabels(pokeTree$node.label)


\dontrun{

# let's try a dataset where not all the
    # taxon relationships lead to a common root

pokexample_bad <- rbind(
    cbind("Squirtadae", c("Squirtle","Blastoise","Wartortle")),
    c("Shelloidea","Lapras"), c("Shelloidea","Squirtadae"),
    c("Pokezooa","Shelloidea"), c("Pokezooa","Parasect"),
    c("Rodentapokemorpha","Linoone"), c("Rodentapokemorpha","Sandshrew"),
    c("Rodentapokemorpha","Pikachu"), c("Hirsutamona","Ursaring"),
    c("Hirsutamona","Rodentapokemorpha"), c("Pokezooa","Hirsutamona"),
    c("Umbrarcheota","Gengar")
    )

# this should return an error
    # as Gengar doesn't share common root
pokeTree <- parentChild2taxonTree(parentChild = pokexample_bad)


# another example, where a taxon is listed as both parent and child
pokexample_bad2 <- rbind(
    cbind("Squirtadae", c("Squirtle","Blastoise","Wartortle")),
    c("Shelloidea", c("Lapras","Squirtadae","Shelloidea")),
    c("Pokezooa","Shelloidea"), c("Pokezooa","Parasect"),
    c("Rodentapokemorpha","Linoone"), c("Rodentapokemorpha","Sandshrew"),
    c("Rodentapokemorpha","Pikachu"), c("Hirsutamona","Ursaring"),
    c("Hirsutamona","Rodentapokemorpha"), c("Pokezooa","Hirsutamona"),
    c("Umbrarcheota","Gengar")
    )

#this should return an error, as Shelloidea is its own parent
pokeTree <- parentChild2taxonTree(parentChild = pokexample_bad2)

}



# note that we should even be able to do this
    # with ancestor-descendent pairs from
    # simulated datasets from simFossilRecord, like so:
set.seed(444)
record <- simFossilRecord(
    p = 0.1, q = 0.1, nruns = 1,
    nTotalTaxa = c(30, 40), 
    nExtant = 0
    )
taxa <- fossilRecord2fossilTaxa(record)
# need to reorder the columns so parents
    # (ancestors) first, then children 
parentChild2taxonTree(parentChild = taxa[,2:1])
# now note that it issues a warning that
    # the input wasn't type character
    # and it will be coerced to be such

}
\seealso{
\code{\link{makePBDBtaxonTree}}, \code{\link{taxonTable2taxonTree}}
}
\author{
David W. Bapst
}
