% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pact.R
\name{eval.pact.cv}
\alias{eval.pact.cv}
\title{Evaluation functions for cross-validated predictions}
\usage{
eval.pact.cv(out.cv, method = c("discrete", "continuous"), g = log(1),
  plot.score = TRUE, plot.time = NULL, perm.test = FALSE, nperm = 100)
}
\arguments{
\item{out.cv}{The object from \code{pact.cv}}

\item{method}{The evaluation method. Currently two options, \code{method='discrete'} or
\code{method='continuous'}, are available. See 'Details'.}

\item{g}{The cut-point for grouping scores into subsets 'benefit' and 'no benefit' from 
new treatment. Ignored for \code{method='continuous'}.}

\item{plot.score}{Used only for plots if \code{method='continuous'} is chosen for survival response. 
Logical representing whether survival 
curves at specific quantiles of cross-validated scores are to be drawn. See 'Details'.}

\item{plot.time}{Used only for plots if \code{method='continuous'} is chosen for survival response. 
Probability of survival greater than \code{plot.time} is plotted as a function of cross-validated 
score and Treatment. See 'Details'.}

\item{perm.test}{Logical. If \code{perm.test=TRUE}, a permutation based test of significance
is conducted for statistics computed from the cross-validated scores. See 'Value' and the package 
vignette and for more details on the permutation tests.}

\item{nperm}{The number of permutations for the permutation test. Ignored if \code{perm.test=FALSE}}
}
\value{
The return object is of class \code{eval.cv} and is a list whose components depend on the family ('cox' or 'binomial') and the 
chosen evaluation method ('continuous' or 'discrete')

\item{LR.Benefit}{For \code{family='cox'} and \code{method='discrete'}. The log-rank statistic for the 
survival difference between E and C for the 'benefit' from E group.}

\item{LR.NoBenefit}{For \code{family='cox'} and \code{method='discrete'}. The log-rank statistic for the 
survival difference between E and C for the 'do not benefit' from E group.}

\item{RR.T.Benefit}{For \code{family='binomial'} and \code{method='discrete'}. 
The response rate for subjects getting E in the 'benefit' from E group.}

\item{RR.C.Benefit}{For \code{family='binomial'} and \code{method='discrete'}. 
The response rate for subjects getting C in the 'benefit' from E group}

\item{RR.T.NoBenefit}{For \code{family='binomial'} and \code{method='discrete'}. 
The response rate for subjects getting E in the 'do not benefit' from E group}

\item{RR.C.NoBenefit}{For \code{family='binomial'} and \code{method='discrete'}. 
The response rate for subjects getting C in the 'do not benefit' from E group}.

\item{pval.Benefit}{If \code{perm.test=TRUE}, p-value from permutation test. 
For \code{family='cox'} and \code{method='discrete'}, permutation based p-value for LR.Benefit. 
For \code{family='binomial'} and \code{method='discrete'}, permutation based p-value for 
difference in response rates for E and C for the subset predicted 'benefit' from E.}

\item{pval.NoBenefit}{If \code{perm.test=TRUE}, p-value from permutation test. 
For \code{family='cox'} and \code{method='discrete'}, permutation based p-value for LR.NoBenefit. 
For \code{family='binomial'} and \code{method='discrete'}, permutation based p-value for 
difference in response rates for E and C for the subset predicted 'no benefit' from E.}

\item{reg}{For \code{method='continuous'}, the regression model with treatment, predictive score and
treatment x predictive score interaction}

\item{pval.twosided}{For \code{method='continuous'}. Two-sided (non-directional) permutation based p-value for the treatment x predictive score
interaction coefficient}

\item{pval.onesided}{For \code{method='continuous'}. One-sided (directional, greater) permutation based p-value for the treatment x predictive score
interaction coefficient}

\item{call}{The function call}

Additional plots for both \code{method='discrete'} as well as \code{method='continuous'}. 
\code{print} method is available for a nice display of objects of class \code{eval.cv}. See package vignette.
}
\description{
Methods for the evaluation of the cross-validated predictive scores obtained from \code{\link{pact.cv}}
}
\details{
Currently two methods are defined for the evaluation of the scores obtained from \code{pact.cv}. In 
\code{method='discrete'} a user specified cut-off score is used to classify the subjects into groups
'benefit' or 'do not benefit' from new treatment. In each of the 'benefit' and 'do not benefit' groups 
the actual responses in the control (C) and the experimental (E) groups are compared.   
For the 'cox' family, the 'score' for a subject represents the predicted change in the log hazard when 
the subject is treated with E as against C (with lower values denoting benefit with E). In the case of the 
'binomial' family, the 'score' represents the predicted change in the log odds of a response when the 
subject is treated with E as against C (with higher values denoting benefit with E). 
For the 'cox' family, examples of the cut-point \code{g} could be \code{g=log(1)} with score < g
meaning benefit with E. Or one could be more stringent and have \code{g} correspond to 
a 30\% reduction in hazard (\code{g=log(0.70)}). 
For the 'binomial' family, \code{g=log(1.20)} with score > g meaning sensitive to E would mean that
subjects predicted to receive at least 20\% increase in odds of response with E are 
classified as benefitting from E.  
\cr\cr In \code{method='continuous'} no cut-off is applied to the cross-validated scores. 
A Cox proportional hazards (PH) regression or a logistic regression 
model (respectively for 'survival' and 'binary' response) is then developed that includes 
the main effect of treatment, main effect of cross-validated score, and treatment*score interaction.
For survival response, this model is used to generate the Kaplan Meier survival curves for each treatment 
at the at 20th, 40th, 60th and 80th percentiles of predictive scores (\code{plot.score = TRUE}). 
The model is also used to compute the estimated probability of surviving beyond a landmark time 
specified in \code{plot.time} as a function of treatment and (cross-validated) score (if 
\code{plot.time = NULL}, this plot is not produced). For binary response, the output from evaluation
is a plot of the probability of response as a functions of the predictive score and Treatment.
\cr\cr If \code{perm.test=TRUE}, permutation based significance tests are performed on appropriate 
test statistics and p-values are computed. See 'Value' and the package vignette and for more details 
on the permutation tests.
}
\examples{
### Survival response
data(prostateCancer)
Y <- prostateCancer[,3:4]
Xf <- prostateCancer[,7:8]
Xv <- prostateCancer[,c(5:6,9)]
Treatment <- as.factor(prostateCancer[,2])
p <- pact.fit(Y=Y, Xf=Xf, Xv=Xv, Treatment=Treatment, family="cox", varSelect="univar")
cv <- pact.cv(p, nfold=5)
\dontrun{eval.pact.cv(cv, method="discrete", g=log(0.80), perm.test=TRUE, nperm=500)}  ## At least 20\% predicted reduction in HR classified as 'sensitive'
eval.pact.cv(cv, method="continuous", plot.score=TRUE, perm.test=FALSE)

### Binary response
data(EORTC10994)
Y <- as.factor(EORTC10994[,4])
Xv <- EORTC10994[,c(2,5:7)]
Treatment <- as.factor(EORTC10994[,3])
p <- pact.fit(Y=Y,Xv=Xv,Treatment=Treatment,family="binomial", varSelect="univar")
cv <- pact.cv(p, nfold=5)
\dontrun{eval.pact.cv(cv, method="discrete", g=log(1), perm.test=TRUE, nperm=500)}


}
\author{
Jyothi Subramanian and Richard Simon
\cr Maintainer: Jyothi Subramanian <\email{subramanianj01@gmail.com}>
}
\keyword{pact,}
\keyword{pact.cv}

