% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get-key-values.R
\name{oe_get_keys}
\alias{oe_get_keys}
\alias{oe_get_keys.default}
\alias{oe_get_keys.character}
\alias{oe_get_keys.sf}
\alias{print.oe_key_values_list}
\title{Return keys and (optionally) values stored in "other_tags" column}
\usage{
oe_get_keys(
  zone,
  layer = "lines",
  values = FALSE,
  which_keys = NULL,
  download_directory = oe_download_directory()
)

\method{oe_get_keys}{default}(
  zone,
  layer = "lines",
  values = FALSE,
  which_keys = NULL,
  download_directory = oe_download_directory()
)

\method{oe_get_keys}{character}(
  zone,
  layer = "lines",
  values = FALSE,
  which_keys = NULL,
  download_directory = oe_download_directory()
)

\method{oe_get_keys}{sf}(
  zone,
  layer = "lines",
  values = FALSE,
  which_keys = NULL,
  download_directory = oe_download_directory()
)

\method{print}{oe_key_values_list}(x, n = getOption("oe_max_print_keys", 10L), ...)
}
\arguments{
\item{zone}{An \code{sf} object with an \code{other_tags} field or a character vector
(of length 1) that can be linked to or pointing to a \code{.osm.pbf} or \code{.gpkg}
file with an \code{other_tags} field. Character vectors are linked to \code{.osm.pbf}
files using \code{oe_find()}.}

\item{layer}{Which \code{layer} should be read in? Typically \code{points}, \code{lines}
(the default), \code{multilinestrings}, \code{multipolygons} or \code{other_relations}. If
you specify an ad-hoc query using the argument \code{query} (see introductory
vignette and examples), then \code{oe_get()} and \code{oe_read()} will read the layer
specified in the query and ignore \code{layer}. See also
\href{https://github.com/ropensci/osmextract/issues/122}{#122}.}

\item{values}{Logical. If \code{TRUE}, then function returns the keys and the
corresponding values, otherwise only the keys. Defaults to \code{FALSE. }}

\item{which_keys}{Character vector used to subset only some keys and
corresponding values. Ignored if \code{values} is \code{FALSE}. See examples.}

\item{download_directory}{Path of the directory that stores the \code{.osm.pbf}
files. Only relevant when \code{zone} is as a character vector that must be
matched to a file via \code{oe_find()}. Ignored unless \code{zone} is a character
vector.}

\item{x}{object of class \code{oe_key_values_list}}

\item{n}{Maximum number of keys (and corresponding values) to print; can be
set globally by \code{options(oe_max_print_keys=...)}. Default value is 10.}

\item{...}{Ignored.}
}
\value{
If the argument \code{values} is \code{FALSE} (the default), then the function
returns a character vector with the names of all keys stored in the
\code{other_tags} field. If \code{values} is \code{TRUE}, then the function returns named
list which stores all keys and the corresponding values. In the latter
case, the returned object has class \code{oe_key_values_list} and we defined an
ad-hoc printing method. See Details.
}
\description{
This function returns the OSM keys and (optionally) the values stored in the
\code{other_tags} field. See Details. In both cases, the keys are sorted according
to the number of occurrences, which means that the most common keys are
stored first.
}
\details{
OSM data are typically documented using several
\href{https://wiki.openstreetmap.org/wiki/Tags}{\code{tags}}, i.e. pairs of two
items, namely a \code{key} and a \code{value}. The conversion between \code{.osm.pbf} and
\code{.gpkg} formats is governed by a \code{CONFIG} file that lists which tags must
be explicitly added to the \code{.gpkg} file. All the other keys are
automatically stored using an \code{other_tags} field with a syntax compatible
with the PostgreSQL HSTORE type. See
\href{https://gdal.org/drivers/vector/osm.html#driver-capabilities}{here} for
more details.

When the argument \code{values} is \code{TRUE}, then the function returns a named
list of class \code{oe_key_values_list} that, for each key, summarises the
corresponding values. The key-value pairs are stored using the following
format: \verb{list(key1 = c("value1", "value1", "value2", ...), key2 = c("value1", ...) ...)}. We decided to implement an ad-hoc method for
printing objects of class \code{oe_key_values_list} using the following
structure:\preformatted{key1 = {#value1 = n1; #value2 = n2; #value3 = n3,
  ...} key2 = {#value1 = n1; #value2 = n2; ...} key3 = {#value1 = n1} ...}
where \code{n1} denotes the number of times that value1 is repeated, \code{n2}
denotes the number of times that value2 is repeated and so on. Also the
values are listed according to the number of occurrences in decreasing
order. By default, the function prints only the ten most common keys, but
the number can be adjusted using the option \code{oe_max_print_keys}.

Finally, the \code{hstore_get_value()} function can be used inside the \code{query}
argument in \code{oe_get()} to extract one particular tag from an existing file.
Check the introductory vignette and see examples.
}
\examples{
# Copy the ITS file to tempdir() to make sure that the examples do not
# require internet connection. You can skip the next 4 lines (and start
# directly with oe_get_keys) when running the examples locally.

its_pbf = file.path(tempdir(), "test_its-example.osm.pbf")
file.copy(
  from = system.file("its-example.osm.pbf", package = "osmextract"),
  to = its_pbf,
  overwrite = TRUE
)

# Get keys
oe_get_keys("ITS Leeds", download_directory = tempdir())

# Get keys and values
oe_get_keys("ITS Leeds", values = TRUE, download_directory = tempdir())

# Subset some keys
oe_get_keys(
  "ITS Leeds", values = TRUE, which_keys = c("surface", "lanes"),
  download_directory = tempdir()
)

# Print all (non-NA) values for a given set of keys
res = oe_get_keys("ITS Leeds", values = TRUE, download_directory = tempdir())
res["surface"]

# Get keys from an existing sf object
its = oe_get("ITS Leeds", download_directory = tempdir())
oe_get_keys(its, values = TRUE)

# Get keys from a character vector pointing to a file (might be faster than
# reading the complete file and then filter it)
its_path = oe_get(
  "ITS Leeds", download_only = TRUE,
  download_directory = tempdir(), quiet = TRUE
)
oe_get_keys(its_path, values = TRUE)

# Add a key to an existing .gpkg file without repeating the
# vectortranslate operations
its = oe_get("ITS Leeds", download_directory = tempdir())
colnames(its)
its_extra = oe_read(
  its_path,
  query = "SELECT *, hstore_get_value(other_tags, 'oneway') AS oneway FROM lines",
  quiet = TRUE
)
colnames(its_extra)

# The following fails since there is no points layer in the .gpkg file
\dontrun{
oe_get_keys(its_path, layer = "points")}

# Add layer and read keys
its_path = oe_get(
  "ITS Leeds", layer = "points", download_only = TRUE,
  download_directory = tempdir(), quiet = TRUE
)
oe_get_keys(its_path, layer = "points")

# Remove .pbf and .gpkg files in tempdir
rm(its_pbf, res, its_path, its, its_extra)
oe_clean(tempdir())
}
\seealso{
\code{oe_vectortranslate()}
}
