% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ortho_optim.r
\name{ortho_optim}
\alias{ortho_optim}
\title{Orthogonality constrained optimization}
\usage{
ortho_optim(B, fn, grad = NULL, ..., maximize = FALSE, control = list(),
  maxitr = 500, verbose = FALSE)
}
\arguments{
\item{B}{Initial \code{B} values. Must be a matrix, and the columns are subject to the orthogonality constrains. Will be processed by Gram-Schmidt if not orthogonal}

\item{fn}{A function that calculate the objective function value. The first argument should be \code{B}. Returns a single value.}

\item{grad}{A function that calculate the gradient. The first argument should be \code{B}. Returns a matrix with the same dimension as \code{B}. If not specified, then numerical approximation is used.}

\item{...}{Arguments passed to \code{fn} and \code{grad}}

\item{maximize}{By default, the solver will try to minimize the objective function unless \code{maximize = TRUE}}

\item{control}{A list of tuning varaibles for optimization. \code{epsilon} is the size for numerically appriximating the gradient. For others, see Wen and Yin (2013).}

\item{maxitr}{Maximum number of iterations}

\item{verbose}{Should information be displayed}
}
\value{
A \code{orthoDr} object; a list consisting of
\item{B}{The optimal \code{B} value}
\item{fn}{The final funtional value}
\item{itr}{The number of iterations}
\item{converge}{convergence code}
}
\description{
A general purpose optimization solver with orthogonality constraint. The orthogonality constrained optimization method is a nearly direct translation from Wen and Yin (2010)'s matlab code.
}
\examples{
library(pracma)
# an example of searching for the first principal component
set.seed(1)
P = 100
N = 400
X = scale(matrix(rnorm(N*P), N, P), scale = FALSE)
w = gramSchmidt(matrix(rnorm(P), P, 1))$Q
fx <- function(w, X) t(w) \%*\% t(X) \%*\% X \%*\% w

# fit the
fit = ortho_optim(w, fx, X = X, maximize = TRUE, verbose = 0)

# compare the result with prcomp
distance(fit$B, as.matrix(prcomp(X)$rotation[, 1]), type = "dist")

# if we have the derivative

gx <- function(w, X) 2*t(X) \%*\% X \%*\% w
fit = ortho_optim(w, fx, gx, X = X, maximize = TRUE, verbose = 0)
distance(fit$B, as.matrix(prcomp(X)$rotation[, 1]), type = "dist")
}
\references{
Wen, Z. and Yin, W., "A feasible method for optimization with orthogonality constraints." Mathematical Programming 142.1-2 (2013): 397-434.
DOI: \url{https://doi.org/10.1007/s10107-012-0584-1}
}
