% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sw.R
\name{swTheta}
\alias{swTheta}
\title{Seawater potential temperature}
\usage{
swTheta(
  salinity,
  temperature = NULL,
  pressure = NULL,
  referencePressure = 0,
  longitude = NULL,
  latitude = NULL,
  eos = getOption("oceEOS", default = "gsw")
)
}
\arguments{
\item{salinity}{either salinity (PSU) (in which case \code{temperature} and
\code{pressure} must be provided) \emph{or} an \code{oce} object (in which
case \code{salinity}, etc. are inferred from the object).}

\item{temperature}{\emph{in-situ} temperature (\eqn{^\circ}{deg}C), defined
on the ITS-90 scale; see \dQuote{Temperature units} in the documentation for
\code{\link[=swRho]{swRho()}}, and the examples below.}

\item{pressure}{pressure (dbar)}

\item{referencePressure}{reference pressure (dbar)}

\item{longitude}{longitude of observation (only used if \code{eos="gsw"};
see \sQuote{Details}).}

\item{latitude}{latitude of observation (only used if \code{eos="gsw"}; see
\sQuote{Details}).}

\item{eos}{equation of state, either \code{"unesco"} (references 1 and 2) or \code{"gsw"}
(references 3 and 4).}
}
\value{
Potential temperature (\eqn{^\circ}{deg}C) of seawater, referenced
to pressure \code{referencePressure}.
}
\description{
Compute the potential temperature of seawater, denoted \eqn{\theta}{theta}
in the UNESCO system, and \code{pt} in the GSW system.
}
\details{
Different formulae are used depending on the equation of state. If \code{eos}
is \code{"unesco"}, the method of Fofonoff \emph{et al.} (1983) is used
(see references 1 and 2).
Otherwise, \code{swTheta} uses \code{\link[gsw:gsw_pt_from_t]{gsw::gsw_pt_from_t()}} from
the \CRANpkg{gsw} package.

If the first argument is a \code{ctd} or \code{section} object, then values
for salinity, etc., are extracted from it, and used for the calculation, and
the corresponding arguments to the present function are ignored.
}
\examples{
library(oce)
# Example 1: test value from Fofonoff et al., 1983
stopifnot(abs(36.8818748026 - swTheta(40, T90fromT68(40), 10000, 0, eos="unesco")) < 0.0000000001)

# Example 2: a deep-water station. Note that theta and CT are
# visually identical on this scale.
data(section)
stn <- section[["station", 70]]
plotProfile(stn, "temperature", ylim=c(6000, 1000))
lines(stn[["theta"]], stn[["pressure"]], col=2)
lines(stn[["CT"]], stn[["pressure"]], col=4, lty=2)
legend("bottomright", lwd=1, col=c(1, 2, 4), lty=c(1, 1, 2),
    legend=c("in-situ", "theta", "CT"),
    title=sprintf("MAD(theta-CT)=\%.4f", mean(abs(stn[["theta"]] - stn[["CT"]]))))

}
\references{
\enumerate{
\item Fofonoff, P. and R. C. Millard Jr, 1983. Algorithms for computation of
fundamental properties of seawater.
\emph{Unesco Technical Papers in Marine Science}, \emph{44}, 53 pp
\item Gill, A.E., 1982. \emph{Atmosphere-ocean Dynamics}, Academic Press, New
York, 662 pp.
\item IOC, SCOR, and IAPSO (2010). The international thermodynamic equation of
seawater-2010: Calculation and use of thermodynamic properties.  Technical
Report 56, Intergovernmental Oceanographic Commission, Manuals and Guide.
\item McDougall, T.J. and P.M. Barker, 2011: Getting started with TEOS-10 and
the Gibbs Seawater (GSW) Oceanographic Toolbox, 28pp., SCOR/IAPSO WG127,
ISBN 978-0-646-55621-5.
}
}
\seealso{
Other functions that calculate seawater properties: 
\code{\link{T68fromT90}()},
\code{\link{T90fromT48}()},
\code{\link{T90fromT68}()},
\code{\link{computableWaterProperties}()},
\code{\link{locationForGsw}()},
\code{\link{swAbsoluteSalinity}()},
\code{\link{swAlphaOverBeta}()},
\code{\link{swAlpha}()},
\code{\link{swBeta}()},
\code{\link{swCSTp}()},
\code{\link{swConservativeTemperature}()},
\code{\link{swDepth}()},
\code{\link{swDynamicHeight}()},
\code{\link{swLapseRate}()},
\code{\link{swN2}()},
\code{\link{swPressure}()},
\code{\link{swRho}()},
\code{\link{swRrho}()},
\code{\link{swSCTp}()},
\code{\link{swSR}()},
\code{\link{swSTrho}()},
\code{\link{swSigma0}()},
\code{\link{swSigma1}()},
\code{\link{swSigma2}()},
\code{\link{swSigma3}()},
\code{\link{swSigma4}()},
\code{\link{swSigmaTheta}()},
\code{\link{swSigmaT}()},
\code{\link{swSigma}()},
\code{\link{swSoundAbsorption}()},
\code{\link{swSoundSpeed}()},
\code{\link{swSpecificHeat}()},
\code{\link{swSpice}()},
\code{\link{swSstar}()},
\code{\link{swTFreeze}()},
\code{\link{swTSrho}()},
\code{\link{swThermalConductivity}()},
\code{\link{swViscosity}()},
\code{\link{swZ}()}
}
\author{
Dan Kelley
}
\concept{functions that calculate seawater properties}
