% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ctd.R
\name{as.ctd}
\alias{as.ctd}
\title{Coerce data into CTD object}
\usage{
as.ctd(salinity, temperature = NULL, pressure = NULL,
  conductivity = NULL, scan = NULL, time = NULL, other = NULL,
  units = NULL, flags = NULL, missingValue = NULL, type = "",
  serialNumber = "", ship = NULL, cruise = NULL, station = NULL,
  startTime = NULL, longitude = NULL, latitude = NULL,
  deploymentType = "unknown", pressureAtmospheric = 0,
  sampleInterval = NA, profile = NULL, debug = getOption("oceDebug"))
}
\arguments{
\item{salinity}{There are several distinct choices for \code{salinity}.
(1) It can be a
vector indicating the practical salinity through the water column. In that case,
\code{as.ctd} employs the other arguments listed below. (2)
it can be something (a data frame, a list or an \code{oce} object)
from which practical salinity, temperature, pressure, and conductivity
can be inferred. In this case, the relevant information
is extracted  and the other arguments to \code{as.ctd} are ignored, except for
\code{pressureAtmospheric}. If the first argument has salinity, etc., in
matrix form (as can happen with some objects of \code{\link{argo-class}}),
then only the first column is used, and a warning to that effect is given,
unless the \code{profile} argument is specified and then that specific
profile is extracted. (3) It can be an object of \code{\link{rsk-class}},
(see \dQuote{Converting rsk objects} for details). (4)
It can be unspecified, in which case \code{conductivity} becomes a mandatory
argument, because it will be needed for computing actual salinity,
using \code{\link{swSCTp}}.}

\item{temperature}{\emph{in-situ} temperature [\eqn{^\circ deg}C], defined on
the ITS-90 scale; see \dQuote{Temperature units} in the documentation for
\code{\link{swRho}}.}

\item{pressure}{Vector of pressure values, one for each \code{salinity} and
\code{temperature} pair, or just a single pressure, which is repeated to match
the length of \code{salinity}.}

\item{conductivity}{electrical conductivity ratio through the water column
(optional). To convert from raw conductivity in milliSeimens per centimeter
divide by 42.914 to get conductivity ratio (see Culkin and Smith, 1980).}

\item{scan}{optional scan number.  If not provided, this will be set to
\code{seq_along(salinity)}.}

\item{time}{optional vector of times of observation}

\item{other}{optional list of other data columns that are not in the standard
list}

\item{units}{an optional list containing units.  If not supplied,
defaults are set for \code{pressure}, \code{temperature}, \code{salinity},
and \code{conductivity}. Since these are simply guesses, users
are advised strongly to supply \code{units}. See \dQuote{Examples}.}

\item{flags}{if supplied, this is a \code{\link{list}} containing data-quality
flags. The elements of this list must have names that match the data
provided to the object.}

\item{missingValue}{optional missing value, indicating data that should be
taken as \code{NA}. Set to \code{NULL} to turn off this feature.}

\item{type}{optional type of CTD, e.g. "SBE"}

\item{serialNumber}{optional serial number of instrument}

\item{ship}{optional string containing the ship from which the observations were made.}

\item{cruise}{optional string containing a cruise identifier.}

\item{station}{optional string containing a station identifier.}

\item{startTime}{optional indication of the start time for the profile,
which is used in some several plotting functions.  This is best given as a
\code{\link{POSIXt}} time, but it may also be a character string
that can be converted to a time with \code{\link{as.POSIXct}},
using \code{UTC} as the timezone.}

\item{longitude}{optional numerical value containing longitude in decimal
degrees, positive in the eastern hemisphere. If this is a single number,
then it is stored in the \code{metadata} slot of the returned value; if it
is a vector of numbers, then they are stored in the \code{data} slot.}

\item{latitude}{optional numerical value containing the latitude in decimal
degrees, positive in the northern hemisphere. See the note on length, for
the \code{longitude} argument.}

\item{deploymentType}{character string indicating the type of deployment. Use
\code{"unknown"} if this is not known, \code{"profile"} for a profile (in
which the data were acquired during a downcast, while the device was lowered
into the water column, perhaps also including an upcast; \code{"moored"} if
the device is installed on a fixed mooring, \code{"thermosalinograph"} (or
\code{"tsg"}) if the device is mounted on a moving vessel, to record
near-surface properties, or \code{"towyo"} if the device is repeatedly
lowered and raised.}

\item{pressureAtmospheric}{A numerical value (a constant or a vector),
that is subtracted from pressure before storing it in the return value.
(This altered pressure is also used in calculating \code{salinity}, if
that is to be computed from \code{conductivity}, etc., using
\code{\link{swSCTp}} (see \code{salinity} above).}

\item{sampleInterval}{optional numerical value indicating the time between
samples in the profile.}

\item{profile}{optional positive integer specifying the number of the profile
to extract from an object that has data in matrices, such as for some
\code{argo} objects. Currently the \code{profile} argument is only utilized for
\code{\link{argo-class}} objects.}

\item{debug}{an integer specifying whether debugging information is
to be printed during the processing. This is a general parameter that
is used by many \code{oce} functions. Generally, setting \code{debug=0}
turns off the printing, while higher values suggest that more information
be printed. If one function calls another, it usually reduces the value of
\code{debug} first, so that a user can often obtain deeper debugging
by specifying higher \code{debug} values.}
}
\value{
An object of \code{\link{ctd-class}}.
}
\description{
Assemble data into a \code{\link{ctd-class}} dataset.
}
\section{Converting rsk objects}{

If the \code{salinity} argument is an object of \code{\link{rsk-class}},
then \code{as.ctd} passes it,
\code{pressureAtmospheric},
\code{longitude},
\code{latitude}
\code{ship},
\code{cruise},
\code{station} and
\code{deploymentType}
to \code{\link{rsk2ctd}}, which builds the ctd object that is
returned by \code{as.ctd}. The other arguments to \code{as.ctd}
are ignored in this instance, because \code{rsk} objects already
contain their information. If required, any data or metadata
element can be added to the value returned by \code{as.ctd}
using \code{\link{oceSetData}} or \code{\link{oceSetMetadata}},
respectively.

The returned \code{\link{rsk-class}} object contains pressure in a form that
may need to be adjusted, because \code{rsk} objects may contain
either absolute pressure or sea pressure. This adjustment is handled
automatically by \code{as.ctd}, by examination of the metadata item
named \code{pressureType} (described in the documentation for
\code{\link{read.rsk}}).  Once the sea pressure is determined,
adjustments may be made with the \code{pressureAtmospheric} argument,
although in that case it is better considered a pressure adjustment
than the atmospheric pressure.

\code{\link{rsk-class}} objects may store sea pressure or absolute pressure (the
sum of sea pressure and atmospheric pressure), depending on how the object was
created with \code{\link{as.rsk}} or \code{\link{read.rsk}}.  However,
\code{\link{ctd-class}} objects store sea pressure, which is needed for
plotting, calculating density, etc. This poses no difficulties, however,
because \code{as.ctd} automatically converts absolute pressure to sea pressure,
if the metadata in the \code{\link{rsk-class}} object indicates that this is
appropriate. Further alteration of the pressure can be accomplished with the
\code{pressureAtmospheric} argument, as noted above.
}

\examples{
library(oce)
## 1. fake data, with default units
pressure <- 1:50
temperature <- 10 - tanh((pressure - 20) / 5) + 0.02*rnorm(50)
salinity <- 34 + 0.5*tanh((pressure - 20) / 5) + 0.01*rnorm(50)
ctd <- as.ctd(salinity, temperature, pressure)
# Add a new column
fluo <- 5 * exp(-pressure / 20)
ctd <- oceSetData(ctd, name="fluorescence", value=fluo,
                  unit=list(unit=expression(mg/m^3), scale=""))
summary(ctd)

## 2. fake data, with supplied units (which are the defaults, actually)
ctd <- as.ctd(salinity, temperature, pressure,
    units=list(salinity=list(unit=expression(), scale="PSS-78"),
    temperature=list(unit=expression(degree*C), scale="ITS-90"),
    pressure=list(unit=expression(dbar), scale="")))

}
\references{
Culkin, F., and Norman D. Smith, 1980. Determination of the
concentration of potassium chloride solution having the same electrical
conductivity, at 15 C and infinite frequency, as standard seawater of salinity
35.0000 ppt (Chlorinity 19.37394 ppt). \emph{IEEE Journal of Oceanic
Engineering}, \bold{5}, pp 22-23.
}
\seealso{
Other things related to \code{ctd} data: \code{\link{[[,ctd-method}},
  \code{\link{[[<-,ctd-method}},
  \code{\link{cnvName2oceName}}, \code{\link{ctd-class}},
  \code{\link{ctdDecimate}}, \code{\link{ctdFindProfiles}},
  \code{\link{ctdRaw}}, \code{\link{ctdTrim}},
  \code{\link{ctd}}, \code{\link{handleFlags,ctd-method}},
  \code{\link{initialize,ctd-method}},
  \code{\link{initializeFlagScheme,ctd-method}},
  \code{\link{oceNames2whpNames}},
  \code{\link{oceUnits2whpUnits}},
  \code{\link{plot,ctd-method}}, \code{\link{plotProfile}},
  \code{\link{plotScan}}, \code{\link{plotTS}},
  \code{\link{read.ctd.itp}}, \code{\link{read.ctd.odf}},
  \code{\link{read.ctd.sbe}},
  \code{\link{read.ctd.woce.other}},
  \code{\link{read.ctd.woce}}, \code{\link{read.ctd}},
  \code{\link{setFlags,ctd-method}},
  \code{\link{subset,ctd-method}},
  \code{\link{summary,ctd-method}},
  \code{\link{woceNames2oceNames}},
  \code{\link{woceUnit2oceUnit}}, \code{\link{write.ctd}}
}
\author{
Dan Kelley
}
\concept{things related to \code{ctd} data}
