% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ctd.R
\name{ctdFindProfiles}
\alias{ctdFindProfiles}
\title{Find Profiles within a Tow-Yow CTD Record}
\usage{
ctdFindProfiles(x, cutoff = 0.5, minLength = 10, minHeight = 0.1 *
  diff(range(x[["pressure"]])), smoother = smooth.spline,
  direction = c("descending", "ascending"), arr.ind = FALSE,
  debug = getOption("oceDebug"), ...)
}
\arguments{
\item{x}{A \code{ctd} object, i.e. one inheriting from \code{\link{ctd-class}}.}

\item{cutoff}{criterion on pressure difference; see \dQuote{Details}.}

\item{minLength}{lower limit on number of points in candidate profiles.}

\item{minHeight}{lower limit on height of candidate profiles.}

\item{smoother}{The smoothing function to use for identifying down/up
casts. The default is \code{smooth.spline}, which performs well for 
a small number of cycles; see \dQuote{Examples} for a method that is
better for a long tow-yo. The return value from \code{smoother} must 
be either a list containing an element named \code{y} or something
that can be coerced to a vector with \code{\link{as.vector}}. To
turn smoothing off, so that cycles in pressure are determined by
simple first difference, set \code{smoother} to \code{NULL}.}

\item{direction}{String indicating the travel direction to be selected.}

\item{arr.ind}{Logical indicating whether the array indices should be returned;
the alternative is to return a vector of ctd objects.}

\item{debug}{An integer specifying whether debugging information is
to be printed during the processing. This is a general parameter that
is used by many \code{oce} functions. Generally, setting \code{debug=0}
turns off the printing, while higher values suggest that more information
be printed.}

\item{...}{Optional extra arguments that are passed to the smoothing function, \code{smoother}.}
}
\value{
If \code{arr.ind=TRUE}, a data frame with columns \code{start} and \code{end}, the indices
of the downcasts.  Otherwise, a vector of \code{ctd} objects.
}
\description{
Examine the pressure record looking for extended periods of either ascent or descent, and return
either indices to these events or a vector of CTD records containing the events.
}
\details{
The method works by examining the pressure record.  First, this is smoothed using
\code{smoother()} (see \dQuote{Arguments}), and then the result is first-differenced
using \code{\link{diff}}.  Median values of the positive and
negative first-difference values are then multiplied by \code{cutoff}.  This establishes criteria
for any given point to be in an ascending profile, a descending profile, or a non-profile.
Contiguous regions are then found, and those that have fewer than \code{minLength} points are
discarded.  Then, those that have pressure ranges less than \code{minHeight} are discarded.

Caution: this method is not well-suited to all datasets. For example, the default
value of \code{smoother} is \code{\link{smooth.spline}}, and this works well for just a few
profiles, but poorly for a tow-yo with a long sequence of profiles; in the latter case,
it can be preferable to use simpler smoothers (see \dQuote{Examples}). Also, depending
on the sampling protocol, it is often necessary to pass the resultant profiles through
\code{\link{ctdTrim}}, to remove artifacts such as an equilibration phase, etc.
Generally, one is well-advised to use the present function for a quick look at the data,
relying on e.g. \code{\link{plotScan}} to identify profiles visually, for a final product.
}
\examples{
\dontrun{
library(oce)
d <- read.csv("towyow.csv", header=TRUE)
towyow <- as.ctd(d$salinity, d$temperature, d$pressure)

casts <- ctdFindProfiles(towyow)
par(mfrow=c(length(casts), 3))
for (cast in casts) {
  plotProfile(cast, "salinity")
  plotProfile(cast, "temperature")
  plotTS(cast, type='o')
}

## Using a moving average to smooth pressure, instead of the default
## smooth.spline() method. This avoids a tendency of smooth.spline()
## to smooth out the profiles in a tow-yo with many (dozens or more) cycles.
movingAverage <- function(x, n = 11, ...) 
{
   f <- rep(1/n, n)
   stats::filter(x, f, ...)
}
casts <- ctdFindProfiles(towyo, smoother=movingAverage)
}

}
\author{
Dan Kelley and Clark Richards
}
\seealso{
The documentation for \code{\link{ctd-class}} explains the structure
of CTD objects, and also outlines the other functions dealing with them.

Other things related to \code{ctd} data: \code{\link{[[,ctd-method}},
  \code{\link{[[<-,ctd-method}}, \code{\link{as.ctd}},
  \code{\link{cnvName2oceName}}, \code{\link{ctd-class}},
  \code{\link{ctdAddColumn}}, \code{\link{ctdDecimate}},
  \code{\link{ctdRaw}}, \code{\link{ctdTrim}},
  \code{\link{ctdUpdateHeader}}, \code{\link{ctd}},
  \code{\link{gps-class}},
  \code{\link{handleFlags,ctd-method}},
  \code{\link{plot,ctd-method}}, \code{\link{plotProfile}},
  \code{\link{plotScan}}, \code{\link{plotTS}},
  \code{\link{read.ctd.itp}}, \code{\link{read.ctd.odf}},
  \code{\link{read.ctd.sbe}},
  \code{\link{read.ctd.woce.other}},
  \code{\link{read.ctd.woce}}, \code{\link{read.ctd}},
  \code{\link{subset,ctd-method}},
  \code{\link{summary,ctd-method}},
  \code{\link{woceNames2oceNames}}, \code{\link{write.ctd}}
}

