#' Generate a List of Available Cell Blocks
#'
#' This function creates and returns a named list of cell blocks,
#' where each name corresponds to a descriptive label of the cell block,
#' and the value is the function name associated with that cell block.
#'
#' @return A named list where each name is a string describing a cell block
#' (e.g., "T2 Cell Block 1.1") and each value is a string corresponding
#' to the function name (e.g., "replext_t2_c1.1") that is associated with
#' the simulation process for that particular cell block.
#'
#' @keywords internal
getCellBlocks <- function(){
  list("T2 Cell Block 1.1" = "replext_t2_c1.1",
       "T2 Cell Block 1.2" = "replext_t2_c1.2",
       "T2 Cell Block 2.1" = "replext_t2_c2.1",
       "T2 Cell Block 2.2" = "replext_t2_c2.2",
       "T2 Cell Block 3.1" = "replext_t2_c3.1",
       "T2 Cell Block 3.2" = "replext_t2_c3.2",
       "T2 Cell Block 4.1" = "replext_t2_c4.1",
       "T2 Cell Block 4.2" = "replext_t2_c4.2",
       "T3 Cell Block 1.1" = "replext_t3_c1.1",
       "T3 Cell Block 1.2" = "replext_t3_c1.2",
       "T3 Cell Block 2.1" = "replext_t3_c2.1",
       "T3 Cell Block 2.2" = "replext_t3_c2.2",
       "T3 Cell Block 3.1" = "replext_t3_c3.1",
       "T3 Cell Block 3.2" = "replext_t3_c3.2",
       "T3 Cell Block 4.1" = "replext_t3_c4.1",
       "T3 Cell Block 4.2" = "replext_t3_c4.2",
       "T4 Cell Block 1.1" = "replext_t4_c1.1",
       "T4 Cell Block 2.1" = "replext_t4_c2.1",
       "T4 Cell Block 3.1" = "replext_t4_c3.1",
       "T4 Cell Block 4.1" = "replext_t4_c4.1",
       "T4 Cell Block 5.1" = "replext_t4_c5.1",
       "T4 Cell Block 6.1" = "replext_t4_c6.1",
       "T4 Cell Block 7.1" = "replext_t4_c7.1",
       "T5 Cell Block 1.1" = "replext_t5_c1.1",
       "T5 Cell Block 1.2" = "replext_t5_c1.2",
       "T5 Cell Block 1.3" = "replext_t5_c1.3",
       "T5 Cell Block 2.1" = "replext_t5_c2.1",
       "T5 Cell Block 2.2" = "replext_t5_c2.2",
       "T5 Cell Block 2.3" = "replext_t5_c2.3",
       "T6 Cell Block 1.1" = "replext_t6_c1.1",
       "T6 Cell Block 1.2" = "replext_t6_c1.2",
       "T6 Cell Block 1.3" = "replext_t6_c1.3",
       "T6 Cell Block 2.1" = "replext_t6_c2.1",
       "T6 Cell Block 2.2" = "replext_t6_c2.2",
       "T6 Cell Block 2.3" = "replext_t6_c2.3",
       "TS1 Cell Block 1.1" = "replext_ts1_c1.1",
       "TS1 Cell Block 2.1" = "replext_ts1_c2.1",
       "TS1 Cell Block 3.1" = "replext_ts1_c3.1",
       "TS2 Cell Block 1.1" = "replext_ts2_c1.1",
       "TS2 Cell Block 1.2" = "replext_ts2_c1.2",
       "TS2 Cell Block 2.1" = "replext_ts2_c2.1",
       "TS2 Cell Block 2.2" = "replext_ts2_c2.2",
       "TS2 Cell Block 3.1" = "replext_ts2_c3.1",
       "TS2 Cell Block 3.2" = "replext_ts2_c3.2",
       "TS2 Cell Block 4.1" = "replext_ts2_c4.1",
       "TS2 Cell Block 4.2" = "replext_ts2_c4.2",
       "TS3 Cell Block 1.1" = "replext_ts3_c1.1",
       "TS3 Cell Block 1.2" = "replext_ts3_c1.2",
       "TS3 Cell Block 2.1" = "replext_ts3_c2.1",
       "TS3 Cell Block 2.2" = "replext_ts3_c2.2",
       "TS3 Cell Block 3.1" = "replext_ts3_c3.1",
       "TS3 Cell Block 3.2" = "replext_ts3_c3.2",
       "TS3 Cell Block 4.1" = "replext_ts3_c4.1",
       "TS3 Cell Block 4.2" = "replext_ts3_c4.2")
}

#' Generate UI Elements for Selected Cell Block
#'
#' This function generates a dynamic user interface (UI) for the Shiny app based on the selected cell block.
#' It creates a list of Shiny UI elements, such as numeric inputs and text inputs,
#' tailored to the requirements of the chosen cell block.
#'
#' @param cellBlock A character string identifying the selected cell block.
#'                  The function uses this parameter to determine which set of UI elements to generate.
#'
#' @return A list of Shiny UI elements specific to the selected cell block.
#'         These UI elements include numeric inputs, text inputs, and other relevant controls
#'         required to capture user inputs for simulation parameters.
#'
#' @keywords internal
getUIParams <- function(cellBlock) {
  switch(cellBlock,
         "replext_t2_c1.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = NA),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = NA),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t2_c1.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 3),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = NA),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = NA),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t2_c2.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t2_c2.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 3),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t2_c3.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 1),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t2_c3.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 3),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 1),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t2_c4.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "4,3,5,4,6,4,3,4,5,6"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "2,4,3,5,3,6,7,11,10,9"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t2_c4.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 3),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "4,3,5,4,6,4,3,4,5,6"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "2,4,3,5,3,6,7,11,10,9"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t3_c1.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 7),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = NA),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = NA),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t3_c1.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 7),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 3),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = NA),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = NA),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t3_c2.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 7),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t3_c2.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 7),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 3),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t3_c3.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 7),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 1),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t3_c3.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 7),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 3),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 1),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t3_c4.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 7),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "4,3,5,4,6,4"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "2,4,3,5,3,6"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t3_c4.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 7),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 3),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "4,3,5,4,6,4"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "2,4,3,5,3,6"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t4_c1.1" = list(shiny::textInput("rdist", "Distribution type:", "rlnorm"),
                                  shiny::numericInput("par1_1", "meanlog Group 1:", 1),
                                  shiny::numericInput("par2_1", "sdlog Group1:", 0.6),
                                  shiny::numericInput("par1_2", "meanlog Group 2:", 2),
                                  shiny::numericInput("par2_2", "sdlog Group2:", 1),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "5,5,10"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "5,10,10"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t4_c2.1" = list(shiny::textInput("rdist", "Distribution type:", "rpois"),
                                  shiny::numericInput("par1_1", "lambda Group 1:", 5),
                                  #shiny::numericInput("par2_1", "not applicable", NA),
                                  shiny::numericInput("par1_2", "lambda Group 2:", 10),
                                  #shiny::numericInput("par2_2", "not applicable", NA),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "5,5,10"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "5,10,10"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t4_c3.1" = list(shiny::textInput("rdist", "Distribution type:", "rchisq"),
                                  shiny::numericInput("par1_1", "dof Group 1:", 1),
                                  shiny::numericInput("par2_1", "ncp Group 1", 0),
                                  shiny::numericInput("par1_2", "dof Group 2:", 6),
                                  shiny::numericInput("par2_2", "ncp Group 2", 0),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "5,5,10"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "5,10,10"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t4_c4.1" = list(shiny::textInput("rdist", "Distribution type:", "rlnorm"),
                                  shiny::numericInput("par1_1", "meanlog Group 1:", 1),
                                  shiny::numericInput("par2_1", "sdlog Group1:", 0.6),
                                  shiny::numericInput("par1_2", "meanlog Group 2:", 3),
                                  shiny::numericInput("par2_2", "sdlog Group2:", 4),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "5,5,10"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "5,10,10"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t4_c5.1" = list(shiny::textInput("rdist", "Distribution type:", "rcauchy"),
                                  shiny::numericInput("par1_1", "location Group 1:", 5),
                                  shiny::numericInput("par2_1", "scale Group1:", 2),
                                  shiny::numericInput("par1_2", "location Group 2:", 10),
                                  shiny::numericInput("par2_2", "scale Group2:", 4),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "5,5,10"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "5,10,10"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t4_c6.1" = list(shiny::textInput("rdist", "Distribution type:", "rchisq,rpois"),
                                  shiny::numericInput("par1_1", "dof Group 1:", 6),
                                  shiny::numericInput("par2_1", "ncp Group1:", 0),
                                  shiny::numericInput("par1_2", "lambda Group 2:", 10),
                                  #shiny::numericInput("par2_2", "scale Group2:", NA),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "5,5,10"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "5,10,10"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t4_c7.1" = list(shiny::textInput("rdist", "Distribution type:", "rlnorm,rchisq"),
                                  shiny::numericInput("par1_1", "meanlog Group 1:", 1),
                                  shiny::numericInput("par2_1", "sdlog Group1:", 0.6),
                                  shiny::numericInput("par1_2", "dof Group 2:", 6),
                                  shiny::numericInput("par2_2", "ncp Group2:", 0),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "5,5,10"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "5,10,10"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t5_c1.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0),
                                  shiny::numericInput("correl", "Correlation between the groups:", value = 0.8),
                                  shiny::textInput("n", "Sample size:", "3,4,5,6,7,8,9,10,15,25"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t5_c1.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 3),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0),
                                  shiny::numericInput("correl", "Correlation between the groups:", value = 0.8),
                                  shiny::textInput("n", "Sample size:", "3,4,5,6,7,8,9,10,15,25"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t5_c1.3" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.5),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.5),
                                  shiny::numericInput("correl", "Correlation between the groups:", value = 0.8),
                                  shiny::textInput("n", "Sample size:", "3,4,5,6,7,8,9,10,15,25"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t5_c2.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 3),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.5),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.5),
                                  shiny::numericInput("correl", "Correlation between the groups:", value = 0.8),
                                  shiny::textInput("n", "Sample size:", "3,4,5,6,7,8,9,10,15,25"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t5_c2.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.2),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                  shiny::numericInput("correl", "Correlation between the groups:", value = 0.8),
                                  shiny::textInput("n", "Sample size:", "3,4,5,6,7,8,9,10,15,25"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t5_c2.3" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 3),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.2),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                  shiny::numericInput("correl", "Correlation between the groups:", value = 0.8),
                                  shiny::textInput("n", "Sample size:", "3,4,5,6,7,8,9,10,15,25"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t6_c1.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 7),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0),
                                  shiny::numericInput("correl", "Correlation between the groups:", value = 0.8),
                                  shiny::textInput("n", "Sample size:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t6_c1.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 7),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 3),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0),
                                  shiny::numericInput("correl", "Correlation between the groups:", value = 0.8),
                                  shiny::textInput("n", "Sample size:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t6_c1.3" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 7),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.5),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.5),
                                  shiny::numericInput("correl", "Correlation between the groups:", value = 0.8),
                                  shiny::textInput("n", "Sample size:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t6_c2.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 7),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 3),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.5),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.5),
                                  shiny::numericInput("correl", "Correlation between the groups:", value = 0.8),
                                  shiny::textInput("n", "Sample size:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t6_c2.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 7),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.2),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                  shiny::numericInput("correl", "Correlation between the groups:", value = 0.8),
                                  shiny::textInput("n", "Sample size:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_t6_c2.3" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 7),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 3),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = 0.2),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                  shiny::numericInput("correl", "Correlation between the groups:", value = 0.8),
                                  shiny::textInput("n", "Sample size:", "3,4,5,6,7,8,9,10,15,25,50,100"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts1_c1.1" = list(shiny::textInput("rdist", "Distribution type:", "rlnorm"),
                                  shiny::numericInput("par1_1", "meanlog Group 1:", 1),
                                  shiny::numericInput("par2_1", "sdlog Group1:", 0.6),
                                  shiny::numericInput("par1_2", "meanlog Group 2:", 1),
                                  shiny::numericInput("par2_2", "sdlog Group2:", 0.6),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "5,5,10"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "5,10,10"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts1_c2.1" = list(shiny::textInput("rdist", "Distribution type:", "rpois"),
                                  shiny::numericInput("par1_1", "lambda Group 1:", 5),
                                  #shiny::numericInput("par2_1", "not applicable", NA),
                                  shiny::numericInput("par1_2", "lambda Group 2:", 5),
                                  #shiny::numericInput("par2_2", "not applicable", NA),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "5,5,10"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "5,10,10"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts1_c3.1" = list(shiny::textInput("rdist", "Distribution type:", "rchisq"),
                                  shiny::numericInput("par1_1", "dof Group 1:", 3),
                                  shiny::numericInput("par2_1", "ncp Group 1", 0),
                                  shiny::numericInput("par1_2", "dof Group 2:", 3),
                                  shiny::numericInput("par2_2", "ncp Group 2", 0),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "5,5,10"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "5,10,10"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts2_c1.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                  shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                  shiny::numericInput("M2", "Mean for the second group:", 5),
                                  shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                  shiny::numericInput("M3", "Mean for the third group:", 5),
                                  shiny::numericInput("S3", "Standard deviation for the third group:", 1),
                                  shiny::numericInput("Sk1", "Skew for the first group:", value = NA),
                                  shiny::numericInput("Sk2", "Skew for the second group:", value = NA),
                                  shiny::numericInput("Sk3", "Skew for the third group:", value = NA),
                                  shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::textInput("n3", "Sample sizes for the third group:", "3,4,5,6,7,8,9,10,15"),
                                  shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                  shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                  shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts2_c1.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 5),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 2),
                                   shiny::numericInput("M3", "Mean for the third group:", 5),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 4),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = NA),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = NA),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = NA),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts2_c2.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 5),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                   shiny::numericInput("M3", "Mean for the third group:", 5),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 1),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = 0.8),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts2_c2.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 5),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 2),
                                   shiny::numericInput("M3", "Mean for the third group:", 5),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 4),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = 0.8),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts2_c3.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 5),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                   shiny::numericInput("M3", "Mean for the third group:", 5),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 1),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = 1),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts2_c3.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 5),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 2),
                                   shiny::numericInput("M3", "Mean for the third group:", 5),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 4),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = 1),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "3,4,5,6,7,8,9,10,15"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts2_c4.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 5),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                   shiny::numericInput("M3", "Mean for the third group:", 5),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 1),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = 0.8),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "2,2,2,3,2,2,3,2,3,2"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "2,3,3,4,2,3,4,2,4,2"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "3,3,4,3,6,6,4,7,5,8"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts2_c4.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 5),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 2),
                                   shiny::numericInput("M3", "Mean for the third group:", 5),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 4),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = 0.8),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "2,2,2,3,2,2,3,2,3,2"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "2,3,3,4,2,3,4,2,4,2"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "3,3,4,3,6,6,4,7,5,8"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts3_c1.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 6),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                   shiny::numericInput("M3", "Mean for the third group:", 7),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 1),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = NA),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = NA),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = NA),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts3_c1.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 6),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 2),
                                   shiny::numericInput("M3", "Mean for the third group:", 7),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 4),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = NA),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = NA),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = NA),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts3_c2.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 6),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                   shiny::numericInput("M3", "Mean for the third group:", 7),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 1),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = 0.8),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts3_c2.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 6),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 2),
                                   shiny::numericInput("M3", "Mean for the third group:", 7),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 4),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = 0.8),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts3_c3.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 6),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                   shiny::numericInput("M3", "Mean for the third group:", 7),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 1),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = 1),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts3_c3.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 6),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 2),
                                   shiny::numericInput("M3", "Mean for the third group:", 7),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 4),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = 1),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "2,3,4,5,6,7,8,9,10,15,25,50,100"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts3_c4.1" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 6),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 1),
                                   shiny::numericInput("M3", "Mean for the third group:", 7),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 1),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = 0.8),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "2,2,2,3,2,2,3,2,3,2"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "2,3,3,4,2,3,4,2,4,2"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "3,3,4,3,6,6,4,7,5,8"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95)),
         "replext_ts3_c4.2" = list(shiny::numericInput("M1", "Mean for the first group:", 5),
                                   shiny::numericInput("S1", "Standard deviation for the first group:", 1),
                                   shiny::numericInput("M2", "Mean for the second group:", 6),
                                   shiny::numericInput("S2", "Standard deviation for the second group:", 2),
                                   shiny::numericInput("M3", "Mean for the third group:", 7),
                                   shiny::numericInput("S3", "Standard deviation for the third group:", 4),
                                   shiny::numericInput("Sk1", "Skew for the first group:", value = 0.8),
                                   shiny::numericInput("Sk2", "Skew for the second group:", value = 0.8),
                                   shiny::numericInput("Sk3", "Skew for the third group:", value = 0.8),
                                   shiny::textInput("n1", "Sample sizes for the first group:", "2,2,2,3,2,2,3,2,3,2"),
                                   shiny::textInput("n2", "Sample sizes for the second group:", "2,3,3,4,2,3,4,2,4,2"),
                                   shiny::textInput("n3", "Sample sizes for the third group:", "3,3,4,3,6,6,4,7,5,8"),
                                   shiny::numericInput("n_simulations", "Number of simulation iterations:", 10000),
                                   shiny::numericInput("nboot", "Number of bootstrap iterations:", 1000),
                                   shiny::numericInput("conf.level", "Confidence level:", 0.95))

         )
}

#' Append Input Parameters to Data Frame
#'
#' This internal function appends the input parameters and a unique run code to the data frame of simulation results.
#' It creates a comprehensive data frame that includes both the results and the parameters used for the simulation,
#' facilitating easier tracking and analysis of the simulation runs.
#'
#' @param df A data frame containing the simulation results.
#' @param input A list of input parameters used in the simulation, typically sourced from the Shiny app's user inputs.
#'
#' @return A data frame that combines the original simulation results with the input parameters used in the simulation.
#'         Additionally, a unique run code is generated and appended to each row for identification purposes.
#'
#' @keywords internal
appendInputParams <- function(df, input) {
  # Generate a unique code for the simulation run
  run_code <- paste(sample(letters, 10, replace = TRUE), collapse = "")

  # Create a data frame of input parameters
  if (grepl("^replext_t2_", input$cellBlock) || grepl("^replext_t3_", input$cellBlock)) {
    params_df <- data.frame(
      M1 = input$M1, S1 = input$S1, M2 = input$M2, S2 = input$S2,
      Sk1 = input$Sk1, Sk2 = input$Sk2, n1 = input$n1, n2 = input$n2,
      n_simulations = input$n_simulations, nboot = input$nboot,
      conf.level = input$conf.level, Seed = input$seed, RunCode = run_code,
      stringsAsFactors = FALSE
    )
  } else if (grepl("^replext_t4_", input$cellBlock) || grepl("^replext_ts1_", input$cellBlock)) {
    params_df <- data.frame(
      rdist = text_to_char_vector(input$rdist), par1_1 = input$par1_1, par2_1 = handle_null(input$par2_1),
      par1_2 = input$par1_2, par2_2 = handle_null(input$par2_2),
      n1 = text_to_vector(input$n1),
      n2 = text_to_vector(input$n2),
      n_simulations = input$n_simulations, nboot = input$nboot,
      conf.level = input$conf.level, Seed = input$seed, RunCode = run_code,
      stringsAsFactors = FALSE
    )
  } else if (grepl("^replext_t5_", input$cellBlock) || grepl("^replext_t6_", input$cellBlock)) {
    params_df <- data.frame(
      M1 = input$M1, S1 = input$S1, M2 = input$M2, S2 = input$S2,
      Sk1 = handle_null(input$Sk1), Sk2 = handle_null(input$Sk2),
      correl = input$correl, n = text_to_vector(input$n),
      n_simulations = input$n_simulations, nboot = input$nboot,
      conf.level = input$conf.level, Seed = input$seed, RunCode = run_code,
      stringsAsFactors = FALSE
    )
  } else if (grepl("^replext_ts2_", input$cellBlock) || grepl("^replext_ts3_", input$cellBlock)) {
    params_df <- data.frame(
      M1 = input$M1, S1 = input$S1, M2 = input$M2, S2 = input$S2,
      M3 = input$M3, S3 = input$S3,
      Sk1 = handle_null(input$Sk1), Sk2 = handle_null(input$Sk2),
      Sk3 = handle_null(input$Sk3),
      n1 = text_to_vector(input$n1), n2 = text_to_vector(input$n2),
      n3 = text_to_vector(input$n3),
      n_simulations = input$n_simulations, nboot = input$nboot,
      conf.level = input$conf.level, Seed = input$seed, RunCode = run_code,
      stringsAsFactors = FALSE
    )
  } else {
    stop("Must select a supported cell block")
  }

  # Repeat the parameters data frame to match the number of rows in df
  params_df <- params_df[rep(1, nrow(df)), ]

  # Combine with the simulation results
  cbind(df, params_df)
}

#' Execute Simulation Based on User Inputs
#'
#' This internal function manages the simulation process in a Shiny app environment.
#' It dynamically selects the appropriate simulation function based on the selected cell block
#' and passes user inputs to this function. The function also handles the setting of a random
#' number seed, if provided, to ensure reproducibility of results.
#'
#' @param input A list of inputs gathered from the Shiny app's UI, including the selected
#'        cell block and other parameters necessary for the simulation.
#'
#' @return The result of the simulation function that corresponds to the selected cell block.
#'         This result is typically a data frame containing the outcomes of the simulation.
#'
#' @keywords internal
runSimulation <- function(input) {

  # Set the seed if provided
  if (!is.na(input$seed) && input$seed > 0) {
    set.seed(input$seed)
  }

  # Dynamically call the appropriate function based on the cell block prefix
  if (grepl("^replext_t2_", input$cellBlock) || grepl("^replext_t3_", input$cellBlock)) {
    return(replext_t2_c1.1(M1 = input$M1, S1 = input$S1, M2 = input$M2, S2 = input$S2,
                           Sk1 = handle_null(input$Sk1), Sk2 = handle_null(input$Sk2),
                           n1 = text_to_vector(input$n1), n2 = text_to_vector(input$n2),
                           n_simulations = input$n_simulations, nboot = input$nboot,
                           conf.level = input$conf.level))
  } else if (grepl("^replext_t4_", input$cellBlock) || grepl("^replext_ts1_", input$cellBlock)) {
    return(replext_t4_c1.1(rdist = text_to_char_vector(input$rdist), par1_1 = input$par1_1, par2_1 = handle_null(input$par2_1),
                           par1_2 = input$par1_2, par2_2 = handle_null(input$par2_2),
                           n1 = text_to_vector(input$n1),
                           n2 = text_to_vector(input$n2),
                           n_simulations = input$n_simulations, nboot = input$nboot,
                           conf.level = input$conf.level))
  } else if (grepl("^replext_t5_", input$cellBlock) || grepl("^replext_t6_", input$cellBlock)) {
    return(replext_t5_c1.1(M1 = input$M1, S1 = input$S1, M2 = input$M2, S2 = input$S2,
                           Sk1 = handle_null(input$Sk1), Sk2 = handle_null(input$Sk2),
                           correl = input$correl, n = text_to_vector(input$n),
                           n_simulations = input$n_simulations, nboot = input$nboot,
                           conf.level = input$conf.level))
  } else if (grepl("^replext_ts2_", input$cellBlock) || grepl("^replext_ts3_", input$cellBlock)) {
    return(replext_ts2_c1.1(M1 = input$M1, S1 = input$S1, M2 = input$M2, S2 = input$S2,
                            M3 = input$M3, S3 = input$S3,
                            Sk1 = handle_null(input$Sk1), Sk2 = handle_null(input$Sk2),
                            Sk3 = handle_null(input$Sk3),
                            n1 = text_to_vector(input$n1), n2 = text_to_vector(input$n2),
                            n3 = text_to_vector(input$n3),
                            n_simulations = input$n_simulations, nboot = input$nboot,
                            conf.level = input$conf.level))
  } else {
    stop("Must select a supported cell block")
  }

}


#' Handle NULL or Empty Input Parameters
#'
#' This internal function is used to process input parameters in a Shiny app.
#' It checks if the provided parameter is `NA` or an empty string and accordingly
#' returns `NULL` or converts it to a numeric value. This function ensures that
#' simulation functions receive properly formatted parameters.
#'
#' @param par_input An input parameter that could be `NA` or an empty string,
#'        typically a user input from the Shiny app's UI. Default is an empty string.
#'
#' @return `NULL` if the input parameter is `NA` or an empty string;
#'         otherwise, the numeric value of the input parameter.
#'
#' @keywords internal
handle_null <- function(par_input = "") {
  if (is.na(par_input) || par_input == "") {
    return(NULL)
  } else {
    return(as.numeric(par_input))
  }
}

#' Convert Comma-Separated String to Numeric Vector
#'
#' This internal function takes a string of comma-separated values and
#' converts it into a numeric vector. It is typically used to process
#' user inputs from the Shiny app's UI where multiple values can be
#' entered as a single string.
#'
#' @param text_input A string containing comma-separated values,
#'        typically a user input from the Shiny app's UI.
#'
#' @return A numeric vector converted from the comma-separated string.
#'         If the input is an empty string, returns an empty numeric vector.
#'
#' @keywords internal
text_to_vector <- function(text_input) {
  as.numeric(unlist(strsplit(text_input, ",")))
}

#' Convert Comma-Separated String to Character Vector
#'
#' This internal function takes a string of comma-separated values and
#' converts it into a character vector. It is used to process user inputs
#' from the Shiny app's UI, particularly when these inputs need to be
#' retained as character data.
#'
#' @param text_input A string containing comma-separated values,
#'        typically user input from the Shiny app's UI. The function
#'        trims leading and trailing whitespace before processing.
#'
#' @return A character vector converted from the comma-separated string.
#'         If the input is an empty string or consists only of whitespace,
#'         returns an empty character vector.
#'
#' @keywords internal
text_to_char_vector <- function(text_input) {
  strsplit(trimws(text_input), ",")[[1]]
}
