% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/egonets.R
\name{egonet_attrs}
\alias{egonet_attrs}
\title{Retrieve alter's attributes (network effects)}
\usage{
egonet_attrs(graph, attrs, V = NULL, direction = "outgoing",
  fun = function(x) x, as.df = FALSE, self = getOption("diffnet.self"),
  valued = getOption("diffnet.valued"), ...)
}
\arguments{
\item{graph}{Any class of accepted graph format (see \code{\link{netdiffuseR-graphs}}).}

\item{attrs}{If \code{graph} is static, Numeric matrix with \eqn{n} rows, otherwise a list of numeric matrices with \eqn{n} rows.}

\item{V}{Integer vector. Set of vertices from which the attributes will be retrieved.}

\item{direction}{Character scalar. Either \code{"outgoing"}, \code{"incoming"}.}

\item{fun}{Function. Applied to each}

\item{as.df}{Logical scalar. When TRUE returns a data.frame instead of a list (see details).}

\item{self}{Logical scalar. When \code{TRUE} autolinks (loops, self edges) are allowed (see details).}

\item{valued}{Logical scalar. When \code{TRUE} weights will be considered. Otherwise non-zero values will be replaced by ones.}

\item{...}{Further arguments to be passed to \code{fun}.}
}
\value{
A list with ego alters's attributes. By default, if the graph is static, the
output is a list of length \code{length(V)} with matrices having the following
columns:

\item{value}{Either the corresponding value of the tie.}
\item{id}{Alter's id}
\item{...}{Further attributes contained in \code{attrs}}

On the other hand, if \code{graph} is dynamic, the output is list of length
\eqn{T} of lists of length \code{length(V)} with data frames having the following
columns:

\item{value}{The corresponding value of the adjacency matrix.}
\item{id}{Alter's id}
\item{per}{Time id}
\item{...}{Further attributes contained in \code{attrs}}
}
\description{
For a given set of vertices V, retrieves each vertex's alter's
attributes. This function enables users to calculate exposure on variables
other than the attribute that is diffusing.  Further, it enables the
specification of alternative functions to use to characterize ego's
personal network including calculating the mean, maximum, minimum, median, or
sum of the alters' attributes. These measures may be static or dynamic over
the interval of diffusion and they may be binary or valued.
}
\details{
By indexing inner/outer edges, this function retrieves ego network attributes
for all \eqn{v \in V}{v in V}, which by default is the complete set
of vertices in the graph.

When \code{as.df=TRUE} the function returns a data.frame of size
\eqn{(|V|\times T)\times k}{(|V| * T)*k} where \eqn{T} is the number of time
periods and \eqn{k} is the number of columns generated by the function.

The function can be used to create network effects as those in the \pkg{RSiena}
package. The difference here is that the definition of the statistic directly
relies on the user. For example, in the \pkg{RSiena} package, the dyadic covariate
effect \emph{37. covariate (centered) main effect (X)}

\deqn{%
s_{i37}(x) = \sum_j x_{ij}(w_{ij}-\bar w)
}{%
s_i37(x) = sum(x[ij] * (w[ij] - mean(w)))
}

Which, having a diffnet object with attributes named \code{x} and \code{w},
can be calculated as

\preformatted{
    egonet_attrs(diffnet, as.df=TRUE, fun=function(dat) {
     sum(dat[, "x"]*(dat[, "w"] - mean(dat[, "w"])))
    })
    }

Furthermore, we could use the \emph{median} centered instead, for example

\preformatted{
    egonet_attrs(diffnet, as.df=TRUE, fun=function(dat) {
     sum(dat[, "x"]*(dat[, "w"] - median(dat[, "w"])))
    })
    }

Where for each \eqn{i}, \code{dat} will be a matrix with as many rows
as individuals in his egonetwork. Such matrix holds the column names of the
attributes in the network.

When \code{self = TRUE}, it will include ego's attributes, regardless
the network has loops or not.
}
\examples{
# Simple example with diffnet -----------------------------------------------
set.seed(1001)
diffnet <- rdiffnet(150, 5, seed.graph="small-world")

# Adding attributes
indeg <- dgr(diffnet, cmode="indegree")
head(indeg)
diffnet[["indegree"]] <- indeg

# Retrieving egonet's attributes (vertices 1 and 20)
egonet_attrs(diffnet, V=c(1,20))

# Example with a static network ---------------------------------------------

set.seed(1231)
n <- 20
net <- rgraph_ws(n = n, k = 4, p = .5)
someattr <- matrix(rnorm(n * 2), ncol= 2, dimnames = list(NULL, c("a", "b")))

# Maximum of -a- in ego network
ans <- egonet_attrs(net, someattr, fun = function(x) max(x[,"a"]))
ans

# checking it worked, taking a look at node 1, 2, and 3
max(someattr[which(net[1,] == 1),"a"]) == ans[1] # TRUE
max(someattr[which(net[2,] == 1),"a"]) == ans[2] # TRUE
max(someattr[which(net[3,] == 1),"a"]) == ans[3] # TRUE


}
\seealso{
Other data management functions: \code{\link{diffnet-class}},
  \code{\link{edgelist_to_adjmat}}, \code{\link{isolated}},
  \code{\link{survey_to_diffnet}}
}
\author{
George G. Vega Yon
}
