// This file is part of UniLib <http://github.com/ufal/unilib/>.
//
// Copyright 2014 Institute of Formal and Applied Linguistics, Faculty of
// Mathematics and Physics, Charles University in Prague, Czech Republic.
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
//
// UniLib version: 3.1.1
// Unicode version: 8.0.0

#include "uninorms.h"

namespace ufal {
namespace nametag {
namespace unilib {

void uninorms::nfc(std::u32string& str) {
  decompose(str, false);
  compose(str);
}

void uninorms::nfd(std::u32string& str) {
  decompose(str, false);
}

void uninorms::nfkc(std::u32string& str) {
  decompose(str, true);
  compose(str);
}

void uninorms::nfkd(std::u32string& str) {
  decompose(str, true);
}

void uninorms::compose(std::u32string& str) {
  size_t old, com;
  for (old = 0, com = 0; old < str.size(); old++, com++) {
    str[com] = str[old];
    if (str[old] >= Hangul::LBase && str[old] < Hangul::LBase + Hangul::LCount) {
      // Check Hangul composition L + V [+ T].
      if (old + 1 < str.size() && str[old + 1] >= Hangul::VBase && str[old + 1] < Hangul::VBase + Hangul::VCount) {
        str[com] = Hangul::SBase + ((str[old] - Hangul::LBase) * Hangul::VCount + str[old + 1] - Hangul::VBase) * Hangul::TCount;
        old++;
        if (old + 1 < str.size() && str[old + 1] > Hangul::TBase && str[old + 1] < Hangul::TBase + Hangul::TCount)
          str[com] += str[++old] - Hangul::TBase;
      }
    } else if (str[old] >= Hangul::SBase && str[old] < Hangul::SBase + Hangul::SCount) {
      // Check Hangul composition LV + T
      if ((str[old] - Hangul::SBase) % Hangul::TCount && old + 1 < str.size() && str[old + 1] > Hangul::TBase && str[old + 1] < Hangul::TBase + Hangul::TCount)
        str[com] += str[++old] - Hangul::TBase;
    } else if (str[old] < CHARS) {
      // Check composition_data
      auto composition = &composition_block[composition_index[str[old] >> 8]][str[old] & 0xFF];
      auto starter = com;
      for (int last_ccc = -1; old + 1 < str.size(); old++) {
        int ccc = str[old + 1] < CHARS ? ccc_block[ccc_index[str[old + 1] >> 8]][str[old + 1] & 0xFF] : 0;
        if (composition[1] - composition[0] && last_ccc < ccc) {
          // Try finding a composition.
          auto l = composition[0], r = composition[1];
          while (l + 2 < r) {
            auto m = l + (((r - l) >> 1) & ~1);
            if (composition_data[m] <= str[old + 1]) l = m;
            if (composition_data[m] >= str[old + 1]) r = m;
          }
          if (composition_data[l] == str[old + 1]) {
            // Found a composition.
            str[starter] = composition_data[l + 1];
            composition = &composition_block[composition_index[composition_data[l + 1] >> 8]][composition_data[l + 1] & 0xFF];
            continue;
          }
        }

        if (!ccc) break;
        last_ccc = ccc;
        str[++com] = str[old + 1];
      }
    }
  }

  if (com < old) str.resize(com);
}

void uninorms::decompose(std::u32string& str, bool kompatibility) {
  // Count how much additional space do we need.
  bool any_decomposition = false;
  size_t additional = 0;
  for (auto&& chr : str) {
    int decomposition_len = 0;

    if (chr >= Hangul::SBase && chr < Hangul::SBase + Hangul::SCount) {
      // Hangul decomposition.
      decomposition_len = 2 + ((chr - Hangul::SBase) % Hangul::TCount ? 1 : 0);
    } else if (chr < CHARS) {
      // Check decomposition_data.
      auto decomposition = &decomposition_block[decomposition_index[chr >> 8]][chr & 0xFF];
      decomposition_len = (decomposition[1] >> 2) - (decomposition[0] >> 2);
      if (decomposition_len && !kompatibility && (decomposition[0] & 1)) decomposition_len = 0;
      if (decomposition_len && kompatibility && (decomposition[0] & 2))
        // Further kompatibility decomposition.
        for (auto i = decomposition[0] >> 2; i < decomposition[1] >> 2; i++) {
          auto further_decomposition = &decomposition_block[decomposition_index[decomposition_data[i] >> 8]][decomposition_data[i] & 0xFF];
          if (further_decomposition[0] & 1) decomposition_len += (further_decomposition[1] >> 2) - (further_decomposition[0] >> 2) - 1;
        }
    }
    // Do we decompose current character?
    if (!decomposition_len) continue;
    any_decomposition = true;
    additional += decomposition_len - 1;
  }

  // If needed, allocate enough space and perform the decomposition.
  if (any_decomposition) {
    str.resize(str.size() + additional);
    for (size_t dec = str.size(), old = dec - additional; old--; )
      if (str[old] >= Hangul::SBase && str[old] < Hangul::SBase + Hangul::SCount) {
        // Hangul decomposition.
        char32_t s_index = str[old] - Hangul::SBase;
        if (s_index % Hangul::TCount) str[--dec] = Hangul::TBase + s_index % Hangul::TCount;
        str[--dec] = Hangul::VBase + (s_index % Hangul::NCount) / Hangul::TCount;
        str[--dec] = Hangul::LBase + s_index / Hangul::NCount;
      } else if (str[old] < CHARS) {
        // Check decomposition_data.
        auto decomposition = &decomposition_block[decomposition_index[str[old] >> 8]][str[old] & 0xFF];
        int decomposition_len = (decomposition[1] >> 2) - (decomposition[0] >> 2);
        if (decomposition_len && !kompatibility && (decomposition[0] & 1)) decomposition_len = 0;
        if (decomposition_len && kompatibility && (decomposition[0] & 2)) {
          // Further kompatibility decomposition.
          while (decomposition_len--) {
            auto chr = decomposition_data[(decomposition[0] >> 2) + decomposition_len];
            auto further_decomposition = &decomposition_block[decomposition_index[chr >> 8]][chr & 0xFF];
            if (further_decomposition[0] & 1) {
              for (int further_decomposition_len = (further_decomposition[1] >> 2) - (further_decomposition[0] >> 2); further_decomposition_len--; )
                str[--dec] = decomposition_data[(further_decomposition[0] >> 2) + further_decomposition_len];
            } else {
              str[--dec] = chr;
            }
          }
        } else if (decomposition_len) {
          // Non-recursive decomposition.
          while (decomposition_len--)
            str[--dec] = decomposition_data[(decomposition[0] >> 2) + decomposition_len];
        } else {
          // No decomposition.
          str[--dec] = str[old];
        }
      } else {
        // Non-Unicode character.
        str[--dec] = str[old];
      }
  }

  // Sort combining marks.
  for (size_t i = 1; i < str.size(); i++) {
    unsigned ccc = str[i] < CHARS ? ccc_block[ccc_index[str[i] >> 8]][str[i] & 0xFF] : 0;
    if (!ccc) continue;

    auto chr = str[i];
    size_t j;
    for (j = i; j && (str[j-1] < CHARS ? ccc_block[ccc_index[str[j-1] >> 8]][str[j-1] & 0xFF] : 0) > ccc; j--) str[j] = str[j-1];
    str[j] = chr;
  }
}

// Data fields
const char32_t uninorms::CHARS;

const char32_t uninorms::Hangul::SBase;
const char32_t uninorms::Hangul::LBase;
const char32_t uninorms::Hangul::VBase;
const char32_t uninorms::Hangul::TBase;
const char32_t uninorms::Hangul::LCount;
const char32_t uninorms::Hangul::VCount;
const char32_t uninorms::Hangul::TCount;
const char32_t uninorms::Hangul::NCount;
const char32_t uninorms::Hangul::SCount;

const uint8_t uninorms::ccc_index[uninorms::CHARS >> 8] = {
  0,0,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,0,0,15,0,0,0,16,17,18,19,20,21,22,0,0,23,0,0,0,0,0,0,0,0,0,0,0,24,25,0,0,26,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,27,0,28,29,30,31,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,32,0,0,33,0,0,34,35,36,0,0,0,0,0,0,37,0,0,0,0,0,38,39,40,41,42,43,44,45,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,46,47,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,49,50,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,51,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};
const uint8_t uninorms::ccc_block[][256] = {
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,232,220,220,220,220,232,216,220,220,220,220,220,202,202,220,220,220,220,202,202,220,220,220,220,220,220,220,220,220,220,220,1,1,1,1,1,220,220,220,220,230,230,230,230,230,230,230,230,240,230,220,220,220,230,230,230,220,220,0,230,230,230,220,220,220,220,230,232,220,220,230,233,234,234,233,234,234,233,230,230,230,230,230,230,230,230,230,230,230,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,230,230,230,230,220,230,230,230,222,220,230,230,230,230,230,230,220,220,220,220,220,220,230,230,220,230,230,222,228,230,10,11,12,13,14,15,16,17,18,19,19,20,21,22,0,23,0,24,25,0,230,220,0,18,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,230,230,230,230,230,30,31,32,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,27,28,29,30,31,32,33,34,230,230,220,220,230,230,230,230,230,220,230,230,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,35,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,230,230,230,230,0,0,230,230,230,230,220,230,0,0,230,230,0,220,230,230,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,220,230,230,220,230,230,220,220,220,230,220,220,230,220,230,230,230,220,230,220,230,220,230,220,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,230,230,230,230,220,230,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,230,0,230,230,230,230,230,230,230,230,230,0,230,230,230,0,230,230,230,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,220,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,230,230,220,230,230,220,230,230,230,220,220,220,27,28,29,230,230,230,220,230,230,220,220,230,230,230,230,230},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,230,220,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,84,91,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,103,103,9,0,0,0,0,0,0,0,0,0,0,0,0,0,107,107,107,107,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,118,118,0,0,0,0,0,0,0,0,0,0,0,0,0,0,122,122,122,122,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,0,220,0,216,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,129,130,0,132,0,0,0,0,0,130,130,130,130,0,0,130,0,230,230,9,0,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,228,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,222,230,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,230,230,230,230,230,0,0,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,230,230,220,220,220,220,220,220,230,230,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,220,230,230,230,230,230,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,0,1,220,220,220,220,220,230,230,220,220,220,220,230,0,1,1,1,1,1,1,1,0,0,0,0,220,0,0,0,0,0,0,230,0,0,0,230,230,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,220,230,230,230,230,230,230,230,220,230,230,234,214,220,202,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,0,0,0,0,0,0,233,220,230,220},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,1,1,230,230,230,230,1,1,1,230,230,0,0,0,0,230,0,0,0,1,1,230,220,230,1,1,220,220,220,220,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,218,228,232,222,224,224,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,0,0,0,0,230,230,230,230,230,230,230,230,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,220,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,0,230,230,220,0,0,230,230,0,0,0,0,0,230,230,0,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,26,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,230,230,230,230,220,220,220,220,220,220,220,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,220,0,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,1,220,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {230,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,230,230,230,230,0,0,0,230,230,230,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,230,230,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,216,216,1,1,1,0,0,0,226,216,216,216,216,216,0,0,0,0,0,0,0,0,220,220,220,220,220,220,220,220,0,0,230,230,230,230,230,220,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,230,230,230,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,220,220,220,220,220,220,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}
};

const uint8_t uninorms::composition_index[uninorms::CHARS >> 8] = {
  0,1,2,3,4,5,6,5,5,7,5,8,9,10,5,5,11,5,5,5,5,5,5,5,5,5,5,12,5,5,13,14,5,15,16,5,5,5,5,5,5,5,5,5,5,5,5,5,17,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,18,19,5,20,21,22,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
    5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
    5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
    5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
    5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5
};
const uint16_t uninorms::composition_block[][257] = {
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,3,5,7,7,7,39,45,55,67,101,103,117,131,161,163,173,185,191,209,241,245,245,261,275,289,327,331,343,347,365,377,377,377,377,377,377,377,409,415,425,437,471,473,487,503,531,535,545,557,563,581,613,617,617,633,647,663,701,705,719,723,743,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,769,769,771,773,777,779,779,779,787,787,787,787,787,789,789,789,789,789,797,803,805,805,807,807,807,807,815,815,815,815,815,815,823,823,825,827,831,833,833,833,841,841,841,841,841,843,843,843,843,843,851,857,859,859,861,861,861,861,869,869,869,869},
  {869,869,869,877,885,885,885,885,885,885,885,885,885,885,885,885,885,885,885,889,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,897,901,901,901,901,901,901,901,901,901,901,901,901,901,903,905,905,905,905,905,907,909,909,909,909,909,909,909,911,913,915,917,917,917,917,917,917,917,917,917,917,917,917,917,917,917,917,917,917,917,917,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,929,939,939,939,939,939,939,939,939,939,939,939,939,939,939,949,959,959,959,959,959,959,959,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,963,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965},
  {965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,967,969,971,973,973,973,973,973,975,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979},
  {979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,993,993,993,993,1001,1001,1011,1011,1025,1025,1025,1025,1025,1025,1033,1033,1035,1035,1035,1035,1047,1047,1047,1047,1057,1057,1057,1059,1059,1061,1061,1061,1077,1077,1077,1077,1085,1085,1097,1097,1113,1113,1113,1113,1113,1113,1121,1121,1125,1125,1125,1125,1141,1141,1141,1141,1153,1159,1165,1165,1165,1167,1167,1167,1167,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171},
  {1171,1171,1171,1171,1171,1171,1171,1173,1173,1173,1173,1173,1173,1173,1173,1173,1173,1177,1177,1177,1179,1179,1185,1189,1191,1199,1199,1201,1201,1201,1201,1203,1203,1203,1203,1203,1211,1211,1211,1211,1213,1213,1213,1213,1215,1215,1217,1217,1217,1221,1221,1221,1223,1223,1229,1233,1235,1243,1243,1245,1245,1245,1245,1247,1247,1247,1247,1247,1255,1255,1255,1255,1257,1257,1257,1257,1259,1259,1261,1261,1261,1261,1261,1261,1261,1261,1261,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1265,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1269,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1273,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1283,1283,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1289,1289,1289,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291},
  {1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1293,1293,1293,1293,1293,1293,1293,1293,1295,1295,1295,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301},
  {1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1313,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315},
  {1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1319,1319,1319,1319,1319,1319,1319,1325,1325,1325,1325,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327},
  {1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1331,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1339,1339,1339,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341},
  {1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343},
  {1343,1343,1343,1343,1343,1343,1345,1345,1347,1347,1349,1349,1351,1351,1353,1353,1353,1353,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1357,1357,1359,1359,1361,1363,1363,1363,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365},
  {1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1367,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1371,1373,1373,1373,1373,1373,1373,1373,1375,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1381,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1387,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1391,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393},
  {1393,1401,1409,1411,1413,1415,1417,1419,1421,1429,1437,1439,1441,1443,1445,1447,1449,1453,1457,1457,1457,1457,1457,1457,1457,1461,1465,1465,1465,1465,1465,1465,1465,1473,1481,1483,1485,1487,1489,1491,1493,1501,1509,1511,1513,1515,1517,1519,1521,1527,1533,1533,1533,1533,1533,1533,1533,1539,1545,1545,1545,1545,1545,1545,1545,1549,1553,1553,1553,1553,1553,1553,1553,1557,1561,1561,1561,1561,1561,1561,1561,1567,1573,1573,1573,1573,1573,1573,1573,1573,1579,1579,1579,1579,1579,1579,1579,1587,1595,1597,1599,1601,1603,1605,1607,1615,1623,1625,1627,1629,1631,1633,1635,1637,1637,1637,1637,1639,1639,1639,1639,1639,1639,1639,1639,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1643,1643,1643,1643,1643,1643,1643,1643,1643,1649,1649,1649,1649,1649,1649,1649,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1653,1653,1653,1653,1653,1653,1653,1653,1659,1659},
  {1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1661,1661,1663,1663,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1667,1667,1669,1669,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671},
  {1671,1671,1671,1671,1673,1673,1673,1673,1673,1675,1675,1675,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1679,1679,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1683,1683,1683,1683,1683,1683,1683,1685,1685,1687,1687,1687,1689,1689,1689,1689,1689,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1693,1693,1693,1695,1697,1697,1697,1697,1697,1697,1697,1697,1697,1697,1697,1697,1697,1699,1701,1701,1701,1703,1705,1705,1705,1707,1709,1711,1713,1713,1713,1713,1713,1715,1717,1717,1717,1719,1721,1721,1721,1721,1721,1721,1721,1721,1721,1721,1723,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1727,1727,1727,1727,1727,1727,1729,1731,1731,1733,1733,1733,1733,1733,1733,1733,1735,1737,1739,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741},
  {1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1743,1743,1743,1743,1743,1745,1745,1747,1747,1749,1749,1751,1751,1753,1753,1755,1755,1757,1757,1759,1759,1761,1761,1763,1763,1765,1765,1767,1767,1767,1769,1769,1771,1771,1773,1773,1773,1773,1773,1773,1773,1777,1777,1777,1781,1781,1781,1785,1785,1785,1789,1789,1789,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1795,1795,1795,1795,1795,1795,1795,1795,1795,1797,1797,1797,1797,1797,1799,1799,1801,1801,1803,1803,1805,1805,1807,1807,1809,1809,1811,1811,1813,1813,1815,1815,1817,1817,1819,1819,1821,1821,1821,1823,1823,1825,1825,1827,1827,1827,1827,1827,1827,1827,1831,1831,1831,1835,1835,1835,1839,1839,1839,1843,1843,1843,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1849,1851,1853,1855,1855,1855,1855,1855,1855,1855,1855,1855,1855,1855,1857,1857,1857},
  {1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1859,1859,1861,1861,1861,1861,1861,1861,1861,1861,1861,1861,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863},
  {1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1865,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867},
  {1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871},
  {1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877},
  {1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1879,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881}
};
const char32_t uninorms::composition_data[] = {
  0,824,8814,824,8800,824,8815,768,192,769,193,770,194,771,195,772,256,774,258,775,550,776,196,777,7842,778,197,780,461,783,512,785,514,803,7840,805,7680,808,260,775,7682,803,7684,817,7686,769,262,770,264,775,266,780,268,807,199,775,7690,780,270,803,7692,807,7696,813,7698,817,7694,768,200,769,201,770,202,771,7868,772,274,774,276,775,278,776,203,777,7866,780,282,783,516,785,518,803,7864,807,552,808,280,813,7704,816,7706,775,7710,769,500,770,284,772,7712,774,286,775,288,780,486,807,290,770,292,775,7714,776,7718,780,542,803,7716,807,7720,814,7722,768,204,769,205,770,206,771,296,772,298,774,300,775,304,776,207,777,7880,780,463,783,520,785,522,803,7882,808,302,816,7724,770,308,769,7728,780,488,803,7730,807,310,817,7732,769,313,780,317,803,7734,807,315,813,7740,817,7738,769,7742,775,7744,803,7746,768,504,769,323,771,209,775,7748,780,327,803,7750,807,325,813,7754,817,7752,768,210,769,211,770,212,771,213,772,332,774,334,775,558,776,214,777,7886,779,336,780,465,783,524,785,526,795,416,803,7884,808,490,769,7764,775,7766,769,340,775,7768,780,344,783,528,785,530,803,7770,807,342,817,7774,769,346,770,348,775,7776,780,352,803,7778,806,536,807,350,775,7786,780,356,803,7788,806,538,807,354,813,7792,817,7790,768,217,769,218,770,219,771,360,772,362,774,364,776,220,777,7910,778,366,779,368,780,467,783,532,785,534,795,431,803,7908,804,7794,808,370,813,7798,816,7796,771,7804,803,7806,768,7808,769,7810,770,372,775,7814,776,7812,803,7816,775,7818,776,7820,768,7922,769,221,770,374,771,7928,772,562,775,7822,776,376,777,7926,803,7924,769,377,770,7824,775,379,780,381,803,7826,817,7828,768,224,769,225,770,226,771,227,772,257,774,259,775,551,776,228,777,7843,778,229,780,462,783,513,785,515,803,7841,805,7681,808,261,775,7683,803,7685,817,7687,769,263,770,265,775,267,780,269,807,231,775,7691,780,271,803,7693,807,7697,813,7699,817,7695,768,232,769,233,770,234,771,7869,772,275,774,277,775,279,776,235,777,7867,780,283,783,517,785,519,803,7865,807,553,808,281,813,7705,816,7707,775,7711,769,501,770,
    285,772,7713,774,287,775,289,780,487,807,291,770,293,775,7715,776,7719,780,543,803,7717,807,7721,814,7723,817,7830,768,236,769,237,770,238,771,297,772,299,774,301,776,239,777,7881,780,464,783,521,785,523,803,7883,808,303,816,7725,770,309,780,496,769,7729,780,489,803,7731,807,311,817,7733,769,314,780,318,803,7735,807,316,813,7741,817,7739,769,7743,775,7745,803,7747,768,505,769,324,771,241,775,7749,780,328,803,7751,807,326,813,7755,817,7753,768,242,769,243,770,244,771,245,772,333,774,335,775,559,776,246,777,7887,779,337,780,466,783,525,785,527,795,417,803,7885,808,491,769,7765,775,7767,769,341,775,7769,780,345,783,529,785,531,803,7771,807,343,817,7775,769,347,770,349,775,7777,780,353,803,7779,806,537,807,351,775,7787,776,7831,780,357,803,7789,806,539,807,355,813,7793,817,7791,768,249,769,250,770,251,771,361,772,363,774,365,776,252,777,7911,778,367,779,369,780,468,783,533,785,535,795,432,803,7909,804,7795,808,371,813,7799,816,7797,771,7805,803,7807,768,7809,769,7811,770,373,775,7815,776,7813,778,7832,803,7817,775,7819,776,7821,768,7923,769,253,770,375,771,7929,772,563,775,7823,776,255,777,7927,778,7833,803,7925,769,378,770,7825,775,380,780,382,803,7827,817,7829,768,8173,769,901,834,8129,768,7846,769,7844,771,7850,777,7848,772,478,769,506,769,508,772,482,769,7688,768,7872,769,7870,771,7876,777,7874,769,7726,768,7890,769,7888,771,7894,777,7892,769,7756,772,556,776,7758,772,554,769,510,768,475,769,471,772,469,780,473,768,7847,769,7845,771,7851,777,7849,772,479,769,507,769,509,772,483,769,7689,768,7873,769,7871,771,7877,777,7875,769,7727,768,7891,769,7889,771,7895,777,7893,769,7757,772,557,776,7759,772,555,769,511,768,476,769,472,772,470,780,474,768,7856,769,7854,771,7860,777,7858,768,7857,769,7855,771,7861,777,7859,768,7700,769,7702,768,7701,769,7703,768,7760,769,7762,768,7761,769,7763,775,7780,775,7781,775,7782,775,7783,769,7800,769,7801,776,7802,776,7803,775,7835,768,7900,769,7898,771,7904,777,7902,803,7906,768,7901,769,7899,771,7905,777,7903,803,7907,768,7914,769,7912,
    771,7918,777,7916,803,7920,768,7915,769,7913,771,7919,777,7917,803,7921,780,494,772,492,772,493,772,480,772,481,774,7708,774,7709,772,560,772,561,780,495,768,8122,769,902,772,8121,774,8120,787,7944,788,7945,837,8124,768,8136,769,904,787,7960,788,7961,768,8138,769,905,787,7976,788,7977,837,8140,768,8154,769,906,772,8153,774,8152,776,938,787,7992,788,7993,768,8184,769,908,787,8008,788,8009,788,8172,768,8170,769,910,772,8169,774,8168,776,939,788,8025,768,8186,769,911,787,8040,788,8041,837,8188,837,8116,837,8132,768,8048,769,940,772,8113,774,8112,787,7936,788,7937,834,8118,837,8115,768,8050,769,941,787,7952,788,7953,768,8052,769,942,787,7968,788,7969,834,8134,837,8131,768,8054,769,943,772,8145,774,8144,776,970,787,7984,788,7985,834,8150,768,8056,769,972,787,8000,788,8001,787,8164,788,8165,768,8058,769,973,772,8161,774,8160,776,971,787,8016,788,8017,834,8166,768,8060,769,974,787,8032,788,8033,834,8182,837,8179,768,8146,769,912,834,8151,768,8162,769,944,834,8167,837,8180,769,979,776,980,776,1031,774,1232,776,1234,769,1027,768,1024,774,1238,776,1025,774,1217,776,1244,776,1246,768,1037,772,1250,774,1049,776,1252,769,1036,776,1254,772,1262,774,1038,776,1264,779,1266,776,1268,776,1272,776,1260,774,1233,776,1235,769,1107,768,1104,774,1239,776,1105,774,1218,776,1245,776,1247,768,1117,772,1251,774,1081,776,1253,769,1116,776,1255,772,1263,774,1118,776,1265,779,1267,776,1269,776,1273,776,1261,776,1111,783,1142,783,1143,776,1242,776,1243,776,1258,776,1259,1619,1570,1620,1571,1621,1573,1620,1572,1620,1574,1620,1730,1620,1747,1620,1728,2364,2345,2364,2353,2364,2356,2494,2507,2519,2508,2878,2891,2902,2888,2903,2892,3031,2964,3006,3018,3031,3020,3006,3019,3158,3144,3285,3264,3266,3274,3285,3271,3286,3272,3285,3275,3390,3402,3415,3404,3390,3403,3530,3546,3535,3548,3551,3550,3530,3549,4142,4134,6965,6918,6965,6920,6965,6922,6965,6924,6965,6926,6965,6930,6965,6971,6965,6973,6965,6976,6965,6977,6965,6979,772,7736,772,7737,772,7772,772,7773,775,7784,775,7785,770,7852,774,7862,770,7853,774,
    7863,770,7878,770,7879,770,7896,770,7897,768,7938,769,7940,834,7942,837,8064,768,7939,769,7941,834,7943,837,8065,837,8066,837,8067,837,8068,837,8069,837,8070,837,8071,768,7946,769,7948,834,7950,837,8072,768,7947,769,7949,834,7951,837,8073,837,8074,837,8075,837,8076,837,8077,837,8078,837,8079,768,7954,769,7956,768,7955,769,7957,768,7962,769,7964,768,7963,769,7965,768,7970,769,7972,834,7974,837,8080,768,7971,769,7973,834,7975,837,8081,837,8082,837,8083,837,8084,837,8085,837,8086,837,8087,768,7978,769,7980,834,7982,837,8088,768,7979,769,7981,834,7983,837,8089,837,8090,837,8091,837,8092,837,8093,837,8094,837,8095,768,7986,769,7988,834,7990,768,7987,769,7989,834,7991,768,7994,769,7996,834,7998,768,7995,769,7997,834,7999,768,8002,769,8004,768,8003,769,8005,768,8010,769,8012,768,8011,769,8013,768,8018,769,8020,834,8022,768,8019,769,8021,834,8023,768,8027,769,8029,834,8031,768,8034,769,8036,834,8038,837,8096,768,8035,769,8037,834,8039,837,8097,837,8098,837,8099,837,8100,837,8101,837,8102,837,8103,768,8042,769,8044,834,8046,837,8104,768,8043,769,8045,834,8047,837,8105,837,8106,837,8107,837,8108,837,8109,837,8110,837,8111,837,8114,837,8130,837,8178,837,8119,768,8141,769,8142,834,8143,837,8135,837,8183,768,8157,769,8158,834,8159,824,8602,824,8603,824,8622,824,8653,824,8655,824,8654,824,8708,824,8713,824,8716,824,8740,824,8742,824,8769,824,8772,824,8775,824,8777,824,8813,824,8802,824,8816,824,8817,824,8820,824,8821,824,8824,824,8825,824,8832,824,8833,824,8928,824,8929,824,8836,824,8837,824,8840,824,8841,824,8930,824,8931,824,8876,824,8877,824,8878,824,8879,824,8938,824,8939,824,8940,824,8941,12441,12436,12441,12364,12441,12366,12441,12368,12441,12370,12441,12372,12441,12374,12441,12376,12441,12378,12441,12380,12441,12382,12441,12384,12441,12386,12441,12389,12441,12391,12441,12393,12441,12400,12442,12401,12441,12403,12442,12404,12441,12406,12442,12407,12441,12409,12442,12410,12441,12412,12442,12413,12441,12446,12441,12532,12441,12460,12441,12462,12441,12464,12441,12466,12441,
    12468,12441,12470,12441,12472,12441,12474,12441,12476,12441,12478,12441,12480,12441,12482,12441,12485,12441,12487,12441,12489,12441,12496,12442,12497,12441,12499,12442,12500,12441,12502,12442,12503,12441,12505,12442,12506,12441,12508,12442,12509,12441,12535,12441,12536,12441,12537,12441,12538,12441,12542,69818,69786,69818,69788,69818,69803,69927,69934,69927,69935,70462,70475,70487,70476,70832,70844,70842,70843,70845,70846,71087,71098,71087,71099
};

const uint8_t uninorms::decomposition_index[uninorms::CHARS >> 8] = {
  0,1,2,3,4,5,6,7,7,8,9,10,11,12,13,14,15,7,7,7,7,7,7,7,7,7,7,16,7,17,18,19,20,21,22,23,24,7,7,7,7,7,25,7,26,27,28,29,30,31,32,33,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,34,35,7,7,7,36,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,37,38,39,40,41,42,43,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,44,45,7,46,47,48,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,49,7,50,51,52,53,54,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,55,56,7,57,58,59,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,60,61,62,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
    7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
    7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
    7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
    7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7
};
const uint16_t uninorms::decomposition_block[][257] = {
  {4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,5,8,8,8,8,8,8,8,9,16,17,20,20,20,20,21,28,28,29,33,37,45,48,48,49,57,61,64,65,77,89,100,100,108,116,124,132,140,148,148,156,164,172,180,188,196,204,212,220,220,228,236,244,252,260,268,268,268,276,284,292,300,308,308,308,316,324,332,340,348,356,356,364,372,380,388,396,404,412,420,428,428,436,444,452,460,468,476,476,476,484,492,500,508,516,516,524},
  {524,532,540,548,556,564,572,580,588,596,604,612,620,628,636,644,652,652,652,660,668,676,684,692,700,708,716,724,732,740,748,756,764,772,780,788,796,804,812,812,812,820,828,836,844,852,860,868,876,884,885,893,900,908,916,924,932,932,940,948,956,964,972,981,989,996,996,996,1004,1012,1020,1028,1036,1045,1052,1052,1052,1060,1068,1076,1084,1092,1100,1100,1100,1108,1116,1124,1132,1140,1148,1156,1164,1172,1180,1188,1196,1204,1212,1220,1228,1236,1244,1244,1244,1252,1260,1268,1276,1284,1292,1300,1308,1316,1324,1332,1340,1348,1356,1364,1372,1380,1388,1396,1404,1412,1420,1429,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1440,1448,1448,1448,1448,1448,1448,1448,1448,1448,1448,1448,1448,1448,1448,1456,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1465,1477,1489,1501,1509,1517,1525,1533,1541,1548,1556,1564,1572,1580,1588,1596,1604,1612,1624,1636,1648,1660,1672,1684,1696,1708,1708,1720,1732,1744,1756,1764,1772,1772,1772,1780,1788,1796,1804,1812,1820,1832,1844,1852,1860,1869,1877,1885,1892,1900,1908,1908,1908,1916,1924,1936,1948,1956,1964,1972,1980},
  {1980,1988,1996,2004,2012,2020,2028,2036,2044,2052,2060,2068,2076,2084,2092,2100,2108,2116,2124,2132,2140,2148,2156,2164,2172,2180,2188,2196,2204,2204,2204,2212,2220,2220,2220,2220,2220,2220,2220,2228,2236,2244,2252,2264,2276,2288,2300,2308,2316,2328,2340,2348,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2357,2361,2365,2369,2373,2377,2381,2385,2389,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2393,2401,2409,2417,2425,2433,2440,2440,2441,2445,2449,2453,2457,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460},
  {2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2464,2468,2468,2472,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2484,2484,2484,2484,2484,2485,2492,2492,2492,2492,2496,2496,2496,2496,2496,2497,2506,2512,2520,2524,2532,2540,2548,2548,2556,2556,2564,2572,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2592,2600,2608,2616,2624,2632,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2652,2660,2668,2676,2684,2685,2689,2693,2698,2706,2713,2717,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2721,2725,2729,2732,2733,2737,2740,2740,2740,2741,2744,2744,2744,2744,2744,2744,2744},
  {2744,2752,2760,2760,2768,2768,2768,2768,2776,2776,2776,2776,2776,2784,2792,2800,2800,2800,2800,2800,2800,2800,2800,2800,2800,2800,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2824,2832,2832,2840,2840,2840,2840,2848,2848,2848,2848,2848,2856,2864,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2880,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2896,2904,2904,2904,2904,2904,2904,2904,2904,2904,2904,2904,2904,2904,2904,2912,2920,2928,2936,2936,2936,2944,2952,2952,2952,2960,2968,2976,2984,2992,3000,3000,3000,3008,3016,3024,3032,3040,3048,3048,3048,3056,3064,3072,3080,3088,3096,3104,3112,3120,3128,3136,3144,3144,3144,3152,3160,3160,3160,3160,3160,3160,3160},
  {3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3161,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168},
  {3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3176,3184,3192,3200,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3209,3217,3225,3233,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3248,3248,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3272,3272,3272,3272,3272,3272,3272,3272,3280,3280,3280,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3296,3304,3312,3320,3328,3336,3344,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3352,3360,3368,3368,3368,3368,3368,3368,3368,3368,3368,3368,3368,3368,3368,3368,3368,3368,3376,3384,3384,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392},
  {3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3392,3400,3400,3400,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3408,3416,3424,3432,3432,3432,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440},
  {3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3440,3448,3448,3448,3456,3464,3464,3464,3464,3464,3464,3464,3464,3464,3464,3464,3464,3464,3464,3464,3464,3472,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3480,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3488,3496,3504,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512},
  {3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3512,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3520,3528,3528,3528,3528,3528,3528,3528,3536,3544,3544,3552,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564},
  {3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3564,3572,3580,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3588,3596,3596,3604,3616,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624},
  {3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3624,3625,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3632,3633,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3640,3641,3649,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656},
  {3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3656,3657,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3660,3668,3668,3668,3668,3668,3668,3668,3668,3668,3668,3676,3676,3676,3676,3676,3684,3684,3684,3684,3684,3692,3692,3692,3692,3692,3700,3700,3700,3700,3700,3700,3700,3700,3700,3700,3700,3700,3700,3708,3708,3708,3708,3708,3708,3708,3708,3708,3708,3716,3716,3724,3733,3744,3753,3764,3764,3764,3764,3764,3764,3764,3764,3772,3772,3772,3772,3772,3772,3772,3772,3772,3772,3772,3772,3772,3772,3772,3772,3772,3772,3780,3780,3780,3780,3780,3780,3780,3780,3780,3780,3788,3788,3788,3788,3788,3796,3796,3796,3796,3796,3804,3804,3804,3804,3804,3812,3812,3812,3812,3812,3812,3812,3812,3812,3812,3812,3812,3812,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820},
  {3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3820,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3828,3829,3832,3832,3832,3832},
  {3832,3832,3832,3832,3832,3832,3832,3840,3840,3848,3848,3856,3856,3864,3864,3872,3872,3872,3872,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3880,3888,3888,3896,3896,3896,3904,3912,3912,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920},
  {3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3920,3921,3925,3929,3932,3933,3937,3941,3945,3949,3953,3957,3961,3965,3969,3973,3976,3977,3981,3985,3989,3993,3997,4001,4005,4009,4013,4017,4021,4025,4029,4033,4037,4041,4045,4048,4049,4053,4057,4061,4065,4069,4073,4077,4081,4085,4089,4093,4097,4101,4105,4109,4113,4117,4121,4125,4129,4133,4137,4141,4145,4149,4153,4157,4160,4160,4160,4160,4160,4160,4160,4160,4160,4160,4160,4160,4160,4161,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4164,4165,4169,4173,4177,4181,4185,4189,4193,4197,4201,4205,4209,4213,4217,4221,4225,4229,4233,4237,4241,4245,4249,4253,4257,4261,4265,4269,4273,4277,4281,4285,4289,4293,4297,4301,4305,4309,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312,4312},
  {4312,4320,4328,4336,4344,4352,4360,4368,4376,4388,4400,4408,4416,4424,4432,4440,4448,4456,4464,4472,4480,4492,4504,4516,4528,4536,4544,4552,4560,4572,4584,4592,4600,4608,4616,4624,4632,4640,4648,4656,4664,4672,4680,4688,4696,4704,4712,4724,4736,4744,4752,4760,4768,4776,4784,4792,4800,4812,4824,4832,4840,4848,4856,4864,4872,4880,4888,4896,4904,4912,4920,4928,4936,4944,4952,4960,4968,4980,4992,5004,5016,5028,5040,5052,5064,5072,5080,5088,5096,5104,5112,5120,5128,5140,5152,5160,5168,5176,5184,5192,5200,5212,5224,5236,5248,5260,5272,5280,5288,5296,5304,5312,5320,5328,5336,5344,5352,5360,5368,5376,5384,5396,5408,5420,5432,5440,5448,5456,5464,5472,5480,5488,5496,5504,5512,5520,5528,5536,5544,5552,5560,5568,5576,5584,5592,5600,5608,5616,5624,5632,5640,5648,5656,5664,5673,5682,5688,5688,5688,5688,5688,5696,5704,5712,5720,5732,5744,5756,5768,5780,5792,5804,5816,5828,5840,5852,5864,5876,5888,5900,5912,5924,5936,5948,5960,5968,5976,5984,5992,6000,6008,6020,6032,6044,6056,6068,6080,6092,6104,6116,6128,6136,6144,6152,6160,6168,6176,6184,6192,6204,6216,6228,6240,6252,6264,6276,6288,6300,6312,6324,6336,6348,6360,6372,6384,6396,6408,6420,6432,6440,6448,6456,6464,6476,6488,6500,6512,6524,6536,6548,6560,6572,6584,6592,6600,6608,6616,6624,6632,6640,6648,6648,6648,6648,6648,6648,6648},
  {6648,6656,6664,6676,6688,6700,6712,6724,6736,6744,6752,6764,6776,6788,6800,6812,6824,6832,6840,6852,6864,6876,6888,6888,6888,6896,6904,6916,6928,6940,6952,6952,6952,6960,6968,6980,6992,7004,7016,7028,7040,7048,7056,7068,7080,7092,7104,7116,7128,7136,7144,7156,7168,7180,7192,7204,7216,7224,7232,7244,7256,7268,7280,7292,7304,7312,7320,7332,7344,7356,7368,7368,7368,7376,7384,7396,7408,7420,7432,7432,7432,7440,7448,7460,7472,7484,7496,7508,7520,7520,7528,7528,7540,7540,7552,7552,7564,7572,7580,7592,7604,7616,7628,7640,7652,7660,7668,7680,7692,7704,7716,7728,7740,7748,7756,7764,7772,7780,7788,7796,7804,7812,7820,7828,7836,7844,7852,7852,7852,7864,7876,7892,7908,7924,7940,7956,7972,7984,7996,8012,8028,8044,8060,8076,8092,8104,8116,8132,8148,8164,8180,8196,8212,8224,8236,8252,8268,8284,8300,8316,8332,8344,8356,8372,8388,8404,8420,8436,8452,8464,8476,8492,8508,8524,8540,8556,8572,8580,8588,8600,8608,8620,8620,8628,8640,8648,8656,8664,8672,8681,8688,8693,8701,8710,8716,8728,8736,8748,8748,8756,8768,8776,8784,8792,8800,8810,8818,8826,8832,8840,8848,8860,8872,8872,8872,8880,8892,8900,8908,8916,8924,8926,8934,8942,8948,8956,8964,8976,8988,8996,9004,9012,9024,9032,9040,9048,9056,9066,9074,9080,9084,9084,9084,9096,9104,9116,9116,9124,9136,9144,9152,9160,9168,9178,9181,9188,9190},
  {9190,9194,9197,9201,9205,9209,9213,9217,9221,9225,9229,9232,9232,9232,9232,9232,9232,9233,9236,9236,9236,9236,9236,9237,9244,9244,9244,9244,9244,9244,9244,9244,9244,9244,9244,9244,9245,9249,9257,9268,9268,9268,9268,9268,9268,9268,9268,9269,9272,9272,9272,9273,9281,9292,9293,9301,9312,9312,9312,9312,9313,9320,9321,9328,9328,9328,9328,9328,9328,9328,9328,9329,9337,9345,9352,9352,9352,9352,9352,9352,9352,9352,9352,9352,9352,9352,9352,9353,9368,9368,9368,9368,9368,9368,9368,9369,9372,9372,9372,9372,9372,9372,9372,9372,9372,9372,9372,9372,9372,9372,9372,9372,9373,9377,9380,9380,9381,9385,9389,9393,9397,9401,9405,9409,9413,9417,9421,9425,9429,9433,9437,9441,9445,9449,9453,9457,9461,9465,9469,9473,9477,9481,9485,9488,9489,9493,9497,9501,9505,9509,9513,9517,9521,9525,9529,9533,9537,9540,9540,9540,9540,9540,9540,9540,9540,9540,9540,9540,9541,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9548,9549},
  {9549,9561,9573,9577,9584,9585,9597,9609,9612,9613,9621,9625,9629,9633,9637,9641,9645,9649,9653,9657,9660,9661,9665,9672,9672,9673,9677,9681,9685,9689,9692,9692,9693,9701,9713,9720,9721,9724,9724,9728,9729,9732,9732,9736,9745,9749,9752,9753,9757,9761,9764,9765,9769,9773,9777,9781,9785,9789,9792,9793,9805,9809,9813,9817,9821,9824,9824,9824,9824,9825,9829,9833,9837,9841,9844,9844,9844,9844,9844,9844,9845,9857,9869,9885,9897,9909,9921,9933,9945,9957,9969,9981,9993,10005,10017,10029,10037,10041,10049,10061,10069,10073,10081,10093,10109,10117,10121,10129,10141,10145,10149,10153,10157,10161,10169,10181,10189,10193,10201,10213,10229,10237,10241,10249,10261,10265,10269,10273,10276,10276,10276,10276,10276,10276,10276,10276,10276,10277,10288,10288,10288,10288,10288,10288,10288,10288,10288,10288,10288,10288,10288,10288,10288,10288,10288,10296,10304,10304,10304,10304,10304,10304,10304,10304,10304,10304,10304,10304,10304,10304,10304,10304,10304,10304,10304,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10312,10320,10328,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336,10336},
  {10336,10336,10336,10336,10336,10344,10344,10344,10344,10344,10352,10352,10352,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10360,10368,10368,10376,10376,10376,10376,10376,10377,10385,10396,10397,10405,10416,10416,10416,10416,10416,10416,10416,10416,10416,10416,10416,10416,10416,10416,10416,10416,10416,10424,10424,10424,10432,10432,10432,10440,10440,10448,10448,10448,10448,10448,10448,10448,10448,10448,10448,10448,10448,10448,10448,10448,10448,10448,10448,10448,10448,10448,10448,10448,10456,10456,10464,10464,10464,10464,10464,10464,10464,10464,10464,10464,10464,10472,10480,10488,10496,10504,10504,10504,10512,10520,10520,10520,10528,10536,10536,10536,10536,10536,10536,10536,10544,10552,10552,10552,10560,10568,10568,10568,10576,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10584,10592,10600,10608,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10616,10624,10632,10640,10648,10648,10648,10648,10648,10648,10648,10656,10664,10672,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680},
  {10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10680,10684,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688},
  {10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10688,10689,10693,10697,10701,10705,10709,10713,10717,10721,10725,10733,10741,10749,10757,10765,10773,10781,10789,10797,10805,10813,10825,10837,10849,10861,10873,10885,10897,10909,10921,10937,10953,10969,10985,11001,11017,11033,11049,11065,11081,11097,11105,11113,11121,11129,11137,11145,11153,11161,11169,11181,11193,11205,11217,11229,11241,11253,11265,11277,11289,11301,11313,11325,11337,11349,11361,11373,11385,11397,11409,11421,11433,11445,11457,11469,11481,11493,11505,11517,11529,11541,11553,11565,11577,11589,11601,11613,11617,11621,11625,11629,11633,11637,11641,11645,11649,11653,11657,11661,11665,11669,11673,11677,11681,11685,11689,11693,11697,11701,11705,11709,11713,11717,11721,11725,11729,11733,11737,11741,11745,11749,11753,11757,11761,11765,11769,11773,11777,11781,11785,11789,11793,11797,11801,11805,11809,11813,11817,11821,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824},
  {11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11824,11825,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11840,11841,11853,11861,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11872,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880},
  {11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11880,11881,11885,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888},
  {11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11888,11889,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892},
  {11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11892,11893,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11896,11897,11900,11900,11900,11900,11900,11900,11900,11900,11900,11900,11900,11900,11901},
  {11901,11905,11909,11913,11917,11921,11925,11929,11933,11937,11941,11945,11949,11953,11957,11961,11965,11969,11973,11977,11981,11985,11989,11993,11997,12001,12005,12009,12013,12017,12021,12025,12029,12033,12037,12041,12045,12049,12053,12057,12061,12065,12069,12073,12077,12081,12085,12089,12093,12097,12101,12105,12109,12113,12117,12121,12125,12129,12133,12137,12141,12145,12149,12153,12157,12161,12165,12169,12173,12177,12181,12185,12189,12193,12197,12201,12205,12209,12213,12217,12221,12225,12229,12233,12237,12241,12245,12249,12253,12257,12261,12265,12269,12273,12277,12281,12285,12289,12293,12297,12301,12305,12309,12313,12317,12321,12325,12329,12333,12337,12341,12345,12349,12353,12357,12361,12365,12369,12373,12377,12381,12385,12389,12393,12397,12401,12405,12409,12413,12417,12421,12425,12429,12433,12437,12441,12445,12449,12453,12457,12461,12465,12469,12473,12477,12481,12485,12489,12493,12497,12501,12505,12509,12513,12517,12521,12525,12529,12533,12537,12541,12545,12549,12553,12557,12561,12565,12569,12573,12577,12581,12585,12589,12593,12597,12601,12605,12609,12613,12617,12621,12625,12629,12633,12637,12641,12645,12649,12653,12657,12661,12665,12669,12673,12677,12681,12685,12689,12693,12697,12701,12705,12709,12713,12717,12721,12725,12729,12733,12737,12741,12745,12749,12753,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12756,12757},
  {12757,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12760,12761,12764,12765,12769,12773,12776,12776,12776,12776,12776,12776,12776,12776,12776,12776,12776,12776,12776,12776,12776,12776,12776,12776,12784,12784,12792,12792,12800,12800,12808,12808,12816,12816,12824,12824,12832,12832,12840,12840,12848,12848,12856,12856,12864,12864,12872,12872,12872,12880,12880,12888,12888,12896,12896,12896,12896,12896,12896,12896,12904,12912,12912,12920,12928,12928,12936,12944,12944,12952,12960,12960,12968,12976,12976,12976,12976,12976,12976,12976,12976,12976,12976,12976,12976,12976,12976,12976,12976,12976,12976,12976,12976,12976,12976,12976,12984,12984,12984,12984,12984,12984,12985,12993,13000,13000,13009,13016,13016,13016,13016,13016,13016,13016,13016,13016,13016,13016,13016,13016,13024,13024,13032,13032,13040,13040,13048,13048,13056,13056,13064,13064,13072,13072,13080,13080,13088,13088,13096,13096,13104,13104,13112,13112,13112,13120,13120,13128,13128,13136,13136,13136,13136,13136,13136,13136,13144,13152,13152,13160,13168,13168,13176,13184,13184,13192,13200,13200,13208,13216,13216,13216,13216,13216,13216,13216,13216,13216,13216,13216,13216,13216,13216,13216,13216,13216,13216,13216,13216,13216,13216,13216,13224,13224,13224,13232,13240,13248,13256,13256,13256,13256,13265,13272},
  {13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13272,13273,13277,13281,13285,13289,13293,13297,13301,13305,13309,13313,13317,13321,13325,13329,13333,13337,13341,13345,13349,13353,13357,13361,13365,13369,13373,13377,13381,13385,13389,13393,13397,13401,13405,13409,13413,13417,13421,13425,13429,13433,13437,13441,13445,13449,13453,13457,13461,13465,13469,13473,13477,13481,13485,13489,13493,13497,13501,13505,13509,13513,13517,13521,13525,13529,13533,13537,13541,13545,13549,13553,13557,13561,13565,13569,13573,13577,13581,13585,13589,13593,13597,13601,13605,13609,13613,13617,13621,13625,13629,13633,13637,13641,13645,13648,13648,13648,13649,13653,13657,13661,13665,13669,13673,13677,13681,13685,13689,13693,13697,13701,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13704,13705},
  {13705,13717,13729,13741,13753,13765,13777,13789,13801,13813,13825,13837,13849,13861,13873,13889,13905,13921,13937,13953,13969,13985,14001,14017,14033,14049,14065,14081,14097,14113,14141,14164,14165,14177,14189,14201,14213,14225,14237,14249,14261,14273,14285,14297,14309,14321,14333,14345,14357,14369,14381,14393,14405,14417,14429,14441,14453,14465,14477,14489,14501,14513,14525,14537,14549,14561,14573,14585,14597,14601,14605,14609,14612,14612,14612,14612,14612,14612,14612,14612,14613,14625,14633,14641,14649,14657,14665,14673,14681,14689,14697,14705,14713,14721,14729,14737,14745,14749,14753,14757,14761,14765,14769,14773,14777,14781,14785,14789,14793,14797,14801,14809,14817,14825,14833,14841,14849,14857,14865,14873,14881,14889,14897,14905,14913,14933,14949,14956,14957,14961,14965,14969,14973,14977,14981,14985,14989,14993,14997,15001,15005,15009,15013,15017,15021,15025,15029,15033,15037,15041,15045,15049,15053,15057,15061,15065,15069,15073,15077,15081,15085,15089,15093,15097,15101,15105,15109,15113,15117,15121,15125,15129,15133,15137,15141,15145,15149,15153,15161,15169,15177,15185,15193,15201,15209,15217,15225,15233,15241,15249,15257,15265,15273,15281,15289,15297,15305,15313,15321,15329,15337,15345,15357,15369,15381,15389,15401,15409,15421,15425,15429,15433,15437,15441,15445,15449,15453,15457,15461,15465,15469,15473,15477,15481,15485,15489,15493,15497,15501,15505,15509,15513,15517,15521,15525,15529,15533,15537,15541,15545,15549,15553,15557,15561,15565,15569,15573,15577,15581,15585,15589,15593,15597,15601,15605,15608,15609},
  {15609,15629,15645,15665,15677,15697,15709,15721,15745,15761,15773,15785,15797,15813,15829,15845,15861,15877,15893,15909,15933,15941,15965,15989,16009,16025,16049,16073,16089,16101,16113,16129,16145,16165,16185,16197,16209,16225,16237,16249,16257,16265,16277,16289,16313,16329,16349,16373,16389,16401,16413,16437,16453,16477,16489,16509,16521,16537,16549,16565,16585,16601,16621,16637,16645,16665,16677,16689,16705,16717,16729,16741,16761,16777,16785,16809,16821,16841,16857,16873,16885,16897,16913,16921,16937,16957,16965,16989,17001,17009,17017,17025,17033,17041,17049,17057,17065,17073,17081,17093,17105,17117,17129,17141,17153,17165,17177,17189,17201,17213,17225,17237,17249,17261,17273,17281,17289,17301,17309,17317,17325,17337,17349,17357,17365,17373,17381,17389,17405,17413,17421,17429,17437,17445,17453,17461,17469,17481,17497,17505,17513,17521,17529,17537,17545,17553,17565,17577,17589,17601,17609,17617,17625,17633,17641,17649,17657,17665,17673,17681,17693,17705,17713,17725,17737,17749,17757,17769,17781,17797,17805,17817,17829,17841,17853,17873,17897,17905,17913,17921,17929,17937,17945,17953,17961,17969,17977,17985,17993,18001,18009,18017,18025,18033,18041,18057,18065,18073,18081,18097,18109,18117,18125,18133,18141,18149,18157,18165,18173,18181,18189,18201,18209,18217,18229,18241,18249,18265,18277,18285,18293,18301,18309,18321,18333,18341,18349,18357,18365,18373,18381,18389,18397,18405,18417,18429,18441,18453,18465,18477,18489,18501,18513,18525,18537,18549,18561,18573,18585,18597,18609,18621,18633,18645,18657,18669,18680},
  {18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18680,18681,18685,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688},
  {18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18688,18689,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18692,18693,18697,18700,18700,18700,18700,18700,18700,18700},
  {18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18700,18701,18705,18709,18713,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716,18716},
  {18716,18720,18724,18728,18732,18736,18740,18744,18748,18752,18756,18760,18764,18768,18772,18776,18780,18784,18788,18792,18796,18800,18804,18808,18812,18816,18820,18824,18828,18832,18836,18840,18844,18848,18852,18856,18860,18864,18868,18872,18876,18880,18884,18888,18892,18896,18900,18904,18908,18912,18916,18920,18924,18928,18932,18936,18940,18944,18948,18952,18956,18960,18964,18968,18972,18976,18980,18984,18988,18992,18996,19000,19004,19008,19012,19016,19020,19024,19028,19032,19036,19040,19044,19048,19052,19056,19060,19064,19068,19072,19076,19080,19084,19088,19092,19096,19100,19104,19108,19112,19116,19120,19124,19128,19132,19136,19140,19144,19148,19152,19156,19160,19164,19168,19172,19176,19180,19184,19188,19192,19196,19200,19204,19208,19212,19216,19220,19224,19228,19232,19236,19240,19244,19248,19252,19256,19260,19264,19268,19272,19276,19280,19284,19288,19292,19296,19300,19304,19308,19312,19316,19320,19324,19328,19332,19336,19340,19344,19348,19352,19356,19360,19364,19368,19372,19376,19380,19384,19388,19392,19396,19400,19404,19408,19412,19416,19420,19424,19428,19432,19436,19440,19444,19448,19452,19456,19460,19464,19468,19472,19476,19480,19484,19488,19492,19496,19500,19504,19508,19512,19516,19520,19524,19528,19532,19536,19540,19544,19548,19552,19556,19560,19564,19568,19572,19576,19580,19584,19588,19592,19596,19600,19604,19608,19612,19616,19620,19624,19628,19632,19636,19640,19644,19648,19652,19656,19660,19664,19668,19672,19676,19680,19684,19688,19692,19696,19700,19704,19708,19712,19716,19720,19724,19728,19732,19736,19740},
  {19740,19744,19748,19752,19756,19760,19764,19768,19772,19776,19780,19784,19788,19792,19796,19796,19796,19800,19800,19804,19804,19804,19808,19812,19816,19820,19824,19828,19832,19836,19840,19844,19844,19848,19848,19852,19852,19852,19856,19860,19860,19860,19860,19864,19868,19872,19876,19880,19884,19888,19892,19896,19900,19904,19908,19912,19916,19920,19924,19928,19932,19936,19940,19944,19948,19952,19956,19960,19964,19968,19972,19976,19980,19984,19988,19992,19996,20000,20004,20008,20012,20016,20020,20024,20028,20032,20036,20040,20044,20048,20052,20056,20060,20064,20068,20072,20076,20080,20084,20088,20092,20096,20100,20104,20108,20112,20116,20120,20124,20128,20132,20132,20132,20136,20140,20144,20148,20152,20156,20160,20164,20168,20172,20176,20180,20184,20188,20192,20196,20200,20204,20208,20212,20216,20220,20224,20228,20232,20236,20240,20244,20248,20252,20256,20260,20264,20268,20272,20276,20280,20284,20288,20292,20296,20300,20304,20308,20312,20316,20320,20324,20328,20332,20336,20340,20344,20348,20352,20356,20360,20364,20368,20372,20376,20380,20384,20388,20392,20396,20400,20404,20408,20412,20416,20420,20424,20428,20432,20436,20440,20444,20448,20452,20456,20460,20464,20468,20472,20476,20480,20484,20488,20492,20496,20500,20504,20508,20512,20516,20520,20524,20528,20532,20536,20540,20544,20548,20552,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20556,20557},
  {20557,20565,20573,20581,20593,20605,20613,20620,20620,20620,20620,20620,20620,20620,20620,20620,20620,20620,20620,20621,20629,20637,20645,20653,20660,20660,20660,20660,20660,20660,20668,20668,20677,20681,20685,20689,20693,20697,20701,20705,20709,20713,20716,20724,20732,20744,20756,20764,20772,20780,20788,20796,20804,20812,20820,20828,20828,20836,20844,20852,20860,20868,20868,20876,20876,20884,20892,20892,20900,20908,20908,20916,20924,20932,20940,20948,20956,20964,20972,20981,20989,20993,20997,21001,21005,21009,21013,21017,21021,21025,21029,21033,21037,21041,21045,21049,21053,21057,21061,21065,21069,21073,21077,21081,21085,21089,21093,21097,21101,21105,21109,21113,21117,21121,21125,21129,21133,21137,21141,21145,21149,21153,21157,21161,21165,21169,21173,21177,21181,21185,21189,21193,21197,21201,21205,21209,21213,21217,21221,21225,21229,21233,21237,21241,21245,21249,21253,21257,21261,21265,21269,21273,21277,21281,21285,21289,21293,21297,21301,21305,21309,21313,21317,21321,21325,21333,21341,21345,21349,21353,21357,21361,21365,21369,21373,21377,21381,21389,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21396,21397,21401,21405,21409,21413,21417,21421,21425,21429,21433,21437,21445,21449,21453,21457,21461,21465,21469,21473,21477,21481,21485,21489,21493,21505,21517,21529,21541,21553,21565,21577,21589,21601,21613,21625,21637,21649,21661,21673,21685,21697,21709,21713,21717,21721,21725},
  {21725,21737,21749,21761,21773,21785,21793,21801,21809,21817,21825,21833,21841,21849,21857,21865,21873,21881,21889,21897,21905,21913,21921,21929,21937,21945,21953,21961,21969,21977,21985,21993,22001,22009,22017,22025,22033,22041,22049,22057,22065,22073,22081,22089,22097,22105,22113,22121,22129,22137,22145,22153,22161,22169,22177,22185,22193,22201,22209,22217,22225,22233,22241,22249,22257,22265,22273,22281,22289,22297,22305,22313,22321,22329,22337,22345,22353,22361,22369,22377,22385,22393,22401,22409,22417,22425,22433,22441,22449,22457,22465,22473,22481,22489,22497,22509,22521,22533,22545,22557,22569,22581,22593,22605,22617,22629,22641,22649,22657,22665,22673,22681,22689,22697,22705,22713,22721,22729,22737,22745,22753,22761,22769,22777,22785,22793,22801,22809,22817,22825,22833,22841,22849,22857,22865,22873,22881,22889,22897,22905,22913,22921,22929,22937,22945,22953,22961,22969,22977,22985,22993,23001,23013,23025,23037,23049,23061,23069,23077,23085,23093,23101,23109,23117,23125,23133,23141,23149,23157,23165,23173,23181,23189,23197,23205,23213,23221,23229,23237,23245,23253,23261,23269,23277,23285,23293,23301,23309,23317,23325,23333,23341,23349,23357,23365,23373,23381,23389,23397,23405,23413,23421,23429,23437,23445,23453,23461,23469,23477,23485,23493,23501,23509,23517,23525,23533,23541,23549,23557,23565,23573,23581,23589,23597,23609,23621,23629,23637,23645,23653,23661,23669,23677,23685,23693,23701,23709,23717,23725,23733,23741,23749,23757,23769,23781,23793,23801,23809,23817,23825,23833,23841,23849,23857,23865,23873,23881},
  {23881,23889,23897,23905,23913,23921,23929,23937,23945,23953,23961,23969,23977,23985,23993,24001,24009,24017,24025,24033,24041,24049,24057,24065,24073,24081,24089,24097,24105,24113,24121,24129,24137,24145,24153,24161,24169,24177,24185,24193,24201,24209,24217,24225,24233,24241,24249,24257,24265,24273,24281,24289,24297,24305,24313,24321,24329,24337,24345,24353,24361,24369,24376,24376,24376,24376,24376,24376,24376,24376,24376,24376,24376,24376,24376,24376,24376,24376,24376,24376,24377,24389,24401,24413,24425,24437,24449,24461,24473,24485,24497,24509,24521,24533,24545,24557,24569,24581,24593,24605,24617,24629,24641,24653,24665,24677,24689,24701,24713,24725,24737,24749,24761,24773,24785,24797,24809,24821,24833,24845,24857,24869,24881,24893,24905,24917,24929,24941,24953,24965,24977,24989,25001,25013,25025,25037,25049,25061,25073,25085,25097,25109,25121,25133,25144,25144,25145,25157,25169,25181,25193,25205,25217,25229,25241,25253,25265,25277,25289,25301,25313,25325,25337,25349,25361,25373,25385,25397,25409,25421,25433,25445,25457,25469,25481,25493,25505,25517,25529,25541,25553,25565,25577,25589,25601,25613,25625,25637,25649,25661,25673,25685,25697,25709,25721,25733,25745,25757,25769,25781,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25792,25793,25805,25817,25833,25849,25865,25881,25897,25913,25929,25941,26013,26045,26060,26060,26060,26060},
  {26060,26060,26060,26060,26060,26060,26060,26060,26060,26060,26060,26060,26060,26060,26060,26060,26061,26065,26069,26073,26077,26081,26085,26089,26093,26097,26108,26108,26108,26108,26108,26108,26108,26108,26108,26108,26108,26108,26108,26108,26108,26108,26108,26108,26108,26108,26108,26108,26109,26117,26121,26125,26129,26133,26137,26141,26145,26149,26153,26157,26161,26165,26169,26173,26177,26181,26185,26189,26193,26196,26196,26197,26201,26205,26213,26221,26229,26237,26241,26245,26249,26253,26257,26260,26261,26265,26269,26273,26277,26281,26285,26289,26293,26297,26301,26305,26309,26313,26317,26321,26325,26329,26333,26336,26337,26341,26345,26349,26352,26352,26352,26352,26353,26361,26369,26376,26377,26384,26385,26393,26401,26409,26417,26425,26433,26441,26449,26457,26465,26469,26477,26485,26493,26501,26509,26517,26525,26533,26541,26549,26557,26565,26569,26573,26577,26581,26585,26589,26593,26597,26601,26605,26609,26613,26617,26621,26625,26629,26633,26637,26641,26645,26649,26653,26657,26661,26665,26669,26673,26677,26681,26685,26689,26693,26697,26701,26705,26709,26713,26717,26721,26725,26729,26733,26737,26741,26745,26749,26753,26757,26761,26765,26769,26773,26777,26781,26785,26789,26793,26797,26801,26805,26809,26813,26817,26821,26825,26829,26833,26837,26841,26845,26849,26853,26857,26861,26865,26869,26873,26877,26881,26885,26889,26893,26897,26901,26905,26909,26913,26917,26921,26925,26929,26933,26937,26941,26945,26949,26953,26957,26961,26965,26969,26973,26977,26981,26993,27005,27017,27029,27041,27053,27061,27068,27068,27068,27068},
  {27068,27069,27073,27077,27081,27085,27089,27093,27097,27101,27105,27109,27113,27117,27121,27125,27129,27133,27137,27141,27145,27149,27153,27157,27161,27165,27169,27173,27177,27181,27185,27189,27193,27197,27201,27205,27209,27213,27217,27221,27225,27229,27233,27237,27241,27245,27249,27253,27257,27261,27265,27269,27273,27277,27281,27285,27289,27293,27297,27301,27305,27309,27313,27317,27321,27325,27329,27333,27337,27341,27345,27349,27353,27357,27361,27365,27369,27373,27377,27381,27385,27389,27393,27397,27401,27405,27409,27413,27417,27421,27425,27429,27433,27437,27441,27445,27449,27453,27457,27461,27465,27469,27473,27477,27481,27485,27489,27493,27497,27501,27505,27509,27513,27517,27521,27525,27529,27533,27537,27541,27545,27549,27553,27557,27561,27565,27569,27573,27577,27581,27585,27589,27593,27597,27601,27605,27609,27613,27617,27621,27625,27629,27633,27637,27641,27645,27649,27653,27657,27661,27665,27669,27673,27677,27681,27685,27689,27693,27697,27701,27705,27709,27713,27717,27721,27725,27729,27733,27737,27741,27745,27749,27753,27757,27761,27765,27769,27773,27777,27781,27785,27789,27793,27797,27801,27805,27809,27813,27817,27821,27825,27828,27828,27828,27829,27833,27837,27841,27845,27849,27852,27852,27853,27857,27861,27865,27869,27873,27876,27876,27877,27881,27885,27889,27893,27897,27900,27900,27901,27905,27909,27912,27912,27912,27913,27917,27921,27925,27933,27937,27941,27944,27945,27949,27953,27957,27961,27965,27969,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972},
  {27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27972,27980,27980,27988,27988,27988,27988,27988,27988,27988,27988,27988,27988,27988,27988,27988,27988,27988,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996},
  {27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,27996,28004,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012},
  {28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28012,28020,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028},
  {28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28028,28036,28044,28044,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052},
  {28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28052,28060,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068},
  {28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28068,28076,28084,28096,28108,28120,28132,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28144,28152,28160,28172,28184,28196,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208},
  {28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28208,28209},
  {28209,28213,28217,28221,28225,28229,28233,28237,28241,28245,28249,28253,28257,28261,28265,28269,28273,28277,28281,28285,28289,28293,28297,28301,28305,28309,28313,28317,28321,28325,28329,28333,28337,28341,28345,28349,28353,28357,28361,28365,28369,28373,28377,28381,28385,28389,28393,28397,28401,28405,28409,28413,28417,28421,28425,28429,28433,28437,28441,28445,28449,28453,28457,28461,28465,28469,28473,28477,28481,28485,28489,28493,28497,28501,28505,28509,28513,28517,28521,28525,28529,28533,28537,28541,28545,28548,28549,28553,28557,28561,28565,28569,28573,28577,28581,28585,28589,28593,28597,28601,28605,28609,28613,28617,28621,28625,28629,28633,28637,28641,28645,28649,28653,28657,28661,28665,28669,28673,28677,28681,28685,28689,28693,28697,28701,28705,28709,28713,28717,28721,28725,28729,28733,28737,28741,28745,28749,28753,28757,28761,28765,28769,28773,28777,28781,28785,28789,28793,28797,28801,28805,28809,28813,28817,28821,28825,28829,28832,28833,28837,28840,28840,28841,28844,28844,28845,28849,28852,28852,28853,28857,28861,28865,28868,28869,28873,28877,28881,28885,28889,28893,28897,28901,28905,28909,28913,28916,28917,28920,28921,28925,28929,28933,28937,28941,28945,28948,28949,28953,28957,28961,28965,28969,28973,28977,28981,28985,28989,28993,28997,29001,29005,29009,29013,29017,29021,29025,29029,29033,29037,29041,29045,29049,29053,29057,29061,29065,29069,29073,29077,29081,29085,29089,29093,29097,29101,29105,29109,29113,29117,29121,29125,29129,29133,29137,29141,29145,29149,29153,29157,29161,29165,29169,29173,29177,29181,29185},
  {29185,29189,29193,29197,29201,29205,29208,29209,29213,29217,29221,29224,29224,29225,29229,29233,29237,29241,29245,29249,29253,29256,29257,29261,29265,29269,29273,29277,29281,29284,29285,29289,29293,29297,29301,29305,29309,29313,29317,29321,29325,29329,29333,29337,29341,29345,29349,29353,29357,29361,29365,29369,29373,29377,29381,29385,29389,29393,29396,29397,29401,29405,29409,29412,29413,29417,29421,29425,29429,29432,29433,29436,29436,29436,29437,29441,29445,29449,29453,29457,29461,29464,29465,29469,29473,29477,29481,29485,29489,29493,29497,29501,29505,29509,29513,29517,29521,29525,29529,29533,29537,29541,29545,29549,29553,29557,29561,29565,29569,29573,29577,29581,29585,29589,29593,29597,29601,29605,29609,29613,29617,29621,29625,29629,29633,29637,29641,29645,29649,29653,29657,29661,29665,29669,29673,29677,29681,29685,29689,29693,29697,29701,29705,29709,29713,29717,29721,29725,29729,29733,29737,29741,29745,29749,29753,29757,29761,29765,29769,29773,29777,29781,29785,29789,29793,29797,29801,29805,29809,29813,29817,29821,29825,29829,29833,29837,29841,29845,29849,29853,29857,29861,29865,29869,29873,29877,29881,29885,29889,29893,29897,29901,29905,29909,29913,29917,29921,29925,29929,29933,29937,29941,29945,29949,29953,29957,29961,29965,29969,29973,29977,29981,29985,29989,29993,29997,30001,30005,30009,30013,30017,30021,30025,30029,30033,30037,30041,30045,30049,30053,30057,30061,30065,30069,30073,30077,30081,30085,30089,30093,30097,30101,30105,30109,30113,30117,30121,30125,30129,30133,30137,30141,30145,30149,30153,30157,30161},
  {30161,30165,30169,30173,30177,30181,30185,30189,30193,30197,30201,30205,30209,30213,30217,30221,30225,30229,30233,30237,30241,30245,30249,30253,30257,30261,30265,30269,30273,30277,30281,30285,30289,30293,30297,30301,30305,30309,30313,30317,30321,30325,30329,30333,30337,30341,30345,30349,30353,30357,30361,30365,30369,30373,30377,30381,30385,30389,30393,30397,30401,30405,30409,30413,30417,30421,30425,30429,30433,30437,30441,30445,30449,30453,30457,30461,30465,30469,30473,30477,30481,30485,30489,30493,30497,30501,30505,30509,30513,30517,30521,30525,30529,30533,30537,30541,30545,30549,30553,30557,30561,30565,30569,30573,30577,30581,30585,30589,30593,30597,30601,30605,30609,30613,30617,30621,30625,30629,30633,30637,30641,30645,30649,30653,30657,30661,30665,30669,30673,30677,30681,30685,30689,30693,30697,30701,30705,30709,30713,30717,30721,30725,30729,30733,30737,30741,30745,30749,30753,30757,30761,30765,30769,30773,30777,30781,30785,30789,30793,30797,30801,30805,30809,30813,30817,30821,30824,30824,30825,30829,30833,30837,30841,30845,30849,30853,30857,30861,30865,30869,30873,30877,30881,30885,30889,30893,30897,30901,30905,30909,30913,30917,30921,30925,30929,30933,30937,30941,30945,30949,30953,30957,30961,30965,30969,30973,30977,30981,30985,30989,30993,30997,31001,31005,31009,31013,31017,31021,31025,31029,31033,31037,31041,31045,31049,31053,31057,31061,31065,31069,31073,31077,31081,31085,31089,31093,31097,31101,31105,31109,31113,31117,31121,31125,31129,31133,31137,31141,31145,31149,31153,31157,31161,31165,31169,31173,31177},
  {31177,31181,31185,31189,31193,31197,31201,31205,31209,31213,31217,31221,31225,31229,31233,31237,31241,31245,31249,31253,31257,31261,31265,31269,31273,31277,31281,31285,31289,31293,31297,31301,31305,31309,31313,31317,31321,31325,31329,31333,31337,31341,31345,31349,31353,31357,31361,31365,31369,31373,31377,31381,31385,31389,31393,31397,31401,31405,31409,31413,31417,31421,31425,31429,31433,31437,31441,31445,31449,31453,31457,31461,31465,31469,31473,31477,31481,31485,31489,31493,31497,31501,31505,31509,31513,31517,31521,31525,31529,31533,31537,31541,31545,31549,31553,31557,31561,31565,31569,31573,31577,31581,31585,31589,31593,31597,31601,31605,31609,31613,31617,31621,31625,31629,31633,31637,31641,31645,31649,31653,31657,31661,31665,31669,31673,31677,31681,31685,31689,31693,31697,31701,31705,31709,31713,31717,31721,31725,31729,31733,31737,31741,31745,31749,31753,31757,31761,31765,31769,31773,31777,31781,31785,31789,31793,31797,31801,31805,31809,31813,31817,31821,31825,31829,31833,31837,31841,31845,31849,31853,31857,31861,31865,31869,31873,31877,31881,31885,31889,31893,31897,31901,31905,31909,31913,31917,31921,31925,31929,31933,31937,31941,31945,31949,31953,31957,31961,31965,31969,31973,31977,31981,31985,31989,31992,31992,31993,31997,32001,32005,32009,32013,32017,32021,32025,32029,32033,32037,32041,32045,32049,32053,32057,32061,32065,32069,32073,32077,32081,32085,32089,32093,32097,32101,32105,32109,32113,32117,32121,32125,32129,32133,32137,32141,32145,32149,32153,32157,32161,32165,32169,32173,32177,32181,32185,32189,32192},
  {32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32192,32193},
  {32193,32197,32201,32205,32208,32209,32213,32217,32221,32225,32229,32233,32237,32241,32245,32249,32253,32257,32261,32265,32269,32273,32277,32281,32285,32289,32293,32297,32301,32305,32309,32313,32316,32317,32321,32324,32325,32328,32328,32329,32332,32333,32337,32341,32345,32349,32353,32357,32361,32365,32369,32372,32373,32377,32381,32385,32388,32389,32392,32393,32396,32396,32396,32396,32396,32396,32397,32400,32400,32400,32400,32401,32404,32405,32408,32409,32412,32413,32417,32421,32424,32425,32429,32432,32433,32436,32436,32437,32440,32441,32444,32445,32448,32449,32452,32453,32456,32457,32461,32464,32465,32468,32468,32469,32473,32477,32481,32484,32485,32489,32493,32497,32501,32505,32509,32512,32513,32517,32521,32525,32528,32529,32533,32537,32541,32544,32545,32548,32549,32553,32557,32561,32565,32569,32573,32577,32581,32585,32588,32589,32593,32597,32601,32605,32609,32613,32617,32621,32625,32629,32633,32637,32641,32645,32649,32653,32656,32656,32656,32656,32656,32657,32661,32665,32668,32669,32673,32677,32681,32685,32688,32689,32693,32697,32701,32705,32709,32713,32717,32721,32725,32729,32733,32737,32741,32745,32749,32753,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756},
  {32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32756,32757},
  {32757,32765,32773,32781,32789,32797,32805,32813,32821,32829,32837,32844,32844,32844,32844,32844,32845,32857,32869,32881,32893,32905,32917,32929,32941,32953,32965,32977,32989,33001,33013,33025,33037,33049,33061,33073,33085,33097,33109,33121,33133,33145,33157,33169,33173,33177,33185,33192,33193,33197,33201,33205,33209,33213,33217,33221,33225,33229,33233,33237,33241,33245,33249,33253,33257,33261,33265,33269,33273,33277,33281,33285,33289,33293,33297,33305,33313,33321,33329,33341,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33348,33349,33357,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33364,33365,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33372,33373},
  {33373,33381,33389,33392,33392,33392,33392,33392,33392,33392,33392,33392,33392,33392,33392,33392,33393,33397,33401,33405,33413,33417,33421,33425,33429,33433,33437,33441,33445,33449,33453,33457,33461,33465,33469,33473,33477,33481,33485,33489,33493,33497,33501,33505,33509,33513,33517,33521,33525,33529,33533,33537,33541,33545,33549,33553,33557,33561,33565,33568,33568,33568,33568,33568,33569,33581,33593,33605,33617,33629,33641,33653,33665,33676,33676,33676,33676,33676,33676,33676,33677,33681,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684,33684},
  {33684,33688,33692,33696,33700,33704,33708,33712,33716,33720,33724,33728,33732,33736,33740,33744,33748,33752,33756,33760,33764,33768,33772,33776,33780,33784,33788,33792,33796,33800,33804,33808,33812,33816,33820,33824,33828,33832,33836,33840,33844,33848,33852,33856,33860,33864,33868,33872,33876,33880,33884,33888,33892,33896,33900,33904,33908,33912,33916,33920,33924,33928,33932,33936,33940,33944,33948,33952,33956,33960,33964,33968,33972,33976,33980,33984,33988,33992,33996,34000,34004,34008,34012,34016,34020,34024,34028,34032,34036,34040,34044,34048,34052,34056,34060,34064,34068,34072,34076,34080,34084,34088,34092,34096,34100,34104,34108,34112,34116,34120,34124,34128,34132,34136,34140,34144,34148,34152,34156,34160,34164,34168,34172,34176,34180,34184,34188,34192,34196,34200,34204,34208,34212,34216,34220,34224,34228,34232,34236,34240,34244,34248,34252,34256,34260,34264,34268,34272,34276,34280,34284,34288,34292,34296,34300,34304,34308,34312,34316,34320,34324,34328,34332,34336,34340,34344,34348,34352,34356,34360,34364,34368,34372,34376,34380,34384,34388,34392,34396,34400,34404,34408,34412,34416,34420,34424,34428,34432,34436,34440,34444,34448,34452,34456,34460,34464,34468,34472,34476,34480,34484,34488,34492,34496,34500,34504,34508,34512,34516,34520,34524,34528,34532,34536,34540,34544,34548,34552,34556,34560,34564,34568,34572,34576,34580,34584,34588,34592,34596,34600,34604,34608,34612,34616,34620,34624,34628,34632,34636,34640,34644,34648,34652,34656,34660,34664,34668,34672,34676,34680,34684,34688,34692,34696,34700,34704,34708},
  {34708,34712,34716,34720,34724,34728,34732,34736,34740,34744,34748,34752,34756,34760,34764,34768,34772,34776,34780,34784,34788,34792,34796,34800,34804,34808,34812,34816,34820,34824,34828,34832,34836,34840,34844,34848,34852,34856,34860,34864,34868,34872,34876,34880,34884,34888,34892,34896,34900,34904,34908,34912,34916,34920,34924,34928,34932,34936,34940,34944,34948,34952,34956,34960,34964,34968,34972,34976,34980,34984,34988,34992,34996,35000,35004,35008,35012,35016,35020,35024,35028,35032,35036,35040,35044,35048,35052,35056,35060,35064,35068,35072,35076,35080,35084,35088,35092,35096,35100,35104,35108,35112,35116,35120,35124,35128,35132,35136,35140,35144,35148,35152,35156,35160,35164,35168,35172,35176,35180,35184,35188,35192,35196,35200,35204,35208,35212,35216,35220,35224,35228,35232,35236,35240,35244,35248,35252,35256,35260,35264,35268,35272,35276,35280,35284,35288,35292,35296,35300,35304,35308,35312,35316,35320,35324,35328,35332,35336,35340,35344,35348,35352,35356,35360,35364,35368,35372,35376,35380,35384,35388,35392,35396,35400,35404,35408,35412,35416,35420,35424,35428,35432,35436,35440,35444,35448,35452,35456,35460,35464,35468,35472,35476,35480,35484,35488,35492,35496,35500,35504,35508,35512,35516,35520,35524,35528,35532,35536,35540,35544,35548,35552,35556,35560,35564,35568,35572,35576,35580,35584,35588,35592,35596,35600,35604,35608,35612,35616,35620,35624,35628,35632,35636,35640,35644,35648,35652,35656,35660,35664,35668,35672,35676,35680,35684,35688,35692,35696,35700,35704,35708,35712,35716,35720,35724,35728,35732},
  {35732,35736,35740,35744,35748,35752,35756,35760,35764,35768,35772,35776,35780,35784,35788,35792,35796,35800,35804,35808,35812,35816,35820,35824,35828,35832,35836,35840,35844,35848,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852,35852}
};
const char32_t uninorms::decomposition_data[] = {
  0,32,32,776,97,32,772,50,51,32,769,956,32,807,49,111,49,8260,52,49,8260,50,51,8260,52,65,768,65,769,65,770,65,771,65,776,65,778,67,807,69,768,69,769,69,770,69,776,73,768,73,769,73,770,73,776,78,771,79,768,79,769,79,770,79,771,79,776,85,768,85,769,85,770,85,776,89,769,97,768,97,769,97,770,97,771,97,776,97,778,99,807,101,768,101,769,101,770,101,776,105,768,105,769,105,770,105,776,110,771,111,768,111,769,111,770,111,771,111,776,117,768,117,769,117,770,117,776,121,769,121,776,65,772,97,772,65,774,97,774,65,808,97,808,67,769,99,769,67,770,99,770,67,775,99,775,67,780,99,780,68,780,100,780,69,772,101,772,69,774,101,774,69,775,101,775,69,808,101,808,69,780,101,780,71,770,103,770,71,774,103,774,71,775,103,775,71,807,103,807,72,770,104,770,73,771,105,771,73,772,105,772,73,774,105,774,73,808,105,808,73,775,73,74,105,106,74,770,106,770,75,807,107,807,76,769,108,769,76,807,108,807,76,780,108,780,76,183,108,183,78,769,110,769,78,807,110,807,78,780,110,780,700,110,79,772,111,772,79,774,111,774,79,779,111,779,82,769,114,769,82,807,114,807,82,780,114,780,83,769,115,769,83,770,115,770,83,807,115,807,83,780,115,780,84,807,116,807,84,780,116,780,85,771,117,771,85,772,117,772,85,774,117,774,85,778,117,778,85,779,117,779,85,808,117,808,87,770,119,770,89,770,121,770,89,776,90,769,122,769,90,775,122,775,90,780,122,780,115,79,795,111,795,85,795,117,795,68,90,780,68,122,780,100,122,780,76,74,76,106,108,106,78,74,78,106,110,106,65,780,97,780,73,780,105,780,79,780,111,780,85,780,117,780,85,776,772,117,776,772,85,776,769,117,776,769,85,776,780,117,776,780,85,776,768,117,776,768,65,776,772,97,776,772,65,775,772,97,775,772,198,772,230,772,71,780,103,780,75,780,107,780,79,808,111,808,79,808,772,111,808,772,439,780,658,780,106,780,68,90,68,122,100,122,71,769,103,769,78,768,110,768,65,778,769,97,778,769,198,769,230,769,216,769,248,769,65,783,97,783,65,785,97,785,69,783,101,783,69,785,101,785,73,783,105,783,73,785,105,785,79,783,111,783,79,785,111,785,82,783,114,783,82,785,114,785,85,783,117,783,
    85,785,117,785,83,806,115,806,84,806,116,806,72,780,104,780,65,775,97,775,69,807,101,807,79,776,772,111,776,772,79,771,772,111,771,772,79,775,111,775,79,775,772,111,775,772,89,772,121,772,104,614,106,114,633,635,641,119,121,32,774,32,775,32,778,32,808,32,771,32,779,611,108,115,120,661,768,769,787,776,769,697,32,837,59,32,769,168,769,913,769,183,917,769,919,769,921,769,927,769,933,769,937,769,953,776,769,921,776,933,776,945,769,949,769,951,769,953,769,965,776,769,953,776,965,776,959,769,965,769,969,769,946,952,933,978,769,978,776,966,960,954,961,962,920,949,931,1045,768,1045,776,1043,769,1030,776,1050,769,1048,768,1059,774,1048,774,1080,774,1077,768,1077,776,1075,769,1110,776,1082,769,1080,768,1091,774,1140,783,1141,783,1046,774,1078,774,1040,774,1072,774,1040,776,1072,776,1045,774,1077,774,1240,776,1241,776,1046,776,1078,776,1047,776,1079,776,1048,772,1080,772,1048,776,1080,776,1054,776,1086,776,1256,776,1257,776,1069,776,1101,776,1059,772,1091,772,1059,776,1091,776,1059,779,1091,779,1063,776,1095,776,1067,776,1099,776,1381,1410,1575,1619,1575,1620,1608,1620,1575,1621,1610,1620,1575,1652,1608,1652,1735,1652,1610,1652,1749,1620,1729,1620,1746,1620,2344,2364,2352,2364,2355,2364,2325,2364,2326,2364,2327,2364,2332,2364,2337,2364,2338,2364,2347,2364,2351,2364,2503,2494,2503,2519,2465,2492,2466,2492,2479,2492,2610,2620,2616,2620,2582,2620,2583,2620,2588,2620,2603,2620,2887,2902,2887,2878,2887,2903,2849,2876,2850,2876,2962,3031,3014,3006,3015,3006,3014,3031,3142,3158,3263,3285,3270,3285,3270,3286,3270,3266,3270,3266,3285,3398,3390,3399,3390,3398,3415,3545,3530,3545,3535,3545,3535,3530,3545,3551,3661,3634,3789,3762,3755,3737,3755,3745,3851,3906,4023,3916,4023,3921,4023,3926,4023,3931,4023,3904,4021,3953,3954,3953,3956,4018,3968,4018,3953,3968,4019,3968,4019,3953,3968,3953,3968,3986,4023,3996,4023,4001,4023,4006,4023,4011,4023,3984,4021,4133,4142,4316,6917,6965,6919,6965,6921,6965,6923,6965,6925,6965,6929,6965,6970,6965,6972,6965,6974,6965,6975,6965,6978,6965,65,198,66,68,
    69,398,71,72,73,74,75,76,77,78,79,546,80,82,84,85,87,97,592,593,7426,98,100,101,601,603,604,103,107,109,331,111,596,7446,7447,112,116,117,7453,623,118,7461,946,947,948,966,967,105,114,117,118,946,947,961,966,967,1085,594,99,597,240,604,102,607,609,613,616,617,618,7547,669,621,7557,671,625,624,626,627,628,629,632,642,643,427,649,650,7452,651,652,122,656,657,658,952,65,805,97,805,66,775,98,775,66,803,98,803,66,817,98,817,67,807,769,99,807,769,68,775,100,775,68,803,100,803,68,817,100,817,68,807,100,807,68,813,100,813,69,772,768,101,772,768,69,772,769,101,772,769,69,813,101,813,69,816,101,816,69,807,774,101,807,774,70,775,102,775,71,772,103,772,72,775,104,775,72,803,104,803,72,776,104,776,72,807,104,807,72,814,104,814,73,816,105,816,73,776,769,105,776,769,75,769,107,769,75,803,107,803,75,817,107,817,76,803,108,803,76,803,772,108,803,772,76,817,108,817,76,813,108,813,77,769,109,769,77,775,109,775,77,803,109,803,78,775,110,775,78,803,110,803,78,817,110,817,78,813,110,813,79,771,769,111,771,769,79,771,776,111,771,776,79,772,768,111,772,768,79,772,769,111,772,769,80,769,112,769,80,775,112,775,82,775,114,775,82,803,114,803,82,803,772,114,803,772,82,817,114,817,83,775,115,775,83,803,115,803,83,769,775,115,769,775,83,780,775,115,780,775,83,803,775,115,803,775,84,775,116,775,84,803,116,803,84,817,116,817,84,813,116,813,85,804,117,804,85,816,117,816,85,813,117,813,85,771,769,117,771,769,85,772,776,117,772,776,86,771,118,771,86,803,118,803,87,768,119,768,87,769,119,769,87,776,119,776,87,775,119,775,87,803,119,803,88,775,120,775,88,776,120,776,89,775,121,775,90,770,122,770,90,803,122,803,90,817,122,817,104,817,116,776,119,778,121,778,97,702,383,775,65,803,97,803,65,777,97,777,65,770,769,97,770,769,65,770,768,97,770,768,65,770,777,97,770,777,65,770,771,97,770,771,65,803,770,97,803,770,65,774,769,97,774,769,65,774,768,97,774,768,65,774,777,97,774,777,65,774,771,97,774,771,65,803,774,97,803,774,69,803,101,803,69,777,101,777,69,771,101,771,69,770,769,101,770,769,69,770,768,101,770,
    768,69,770,777,101,770,777,69,770,771,101,770,771,69,803,770,101,803,770,73,777,105,777,73,803,105,803,79,803,111,803,79,777,111,777,79,770,769,111,770,769,79,770,768,111,770,768,79,770,777,111,770,777,79,770,771,111,770,771,79,803,770,111,803,770,79,795,769,111,795,769,79,795,768,111,795,768,79,795,777,111,795,777,79,795,771,111,795,771,79,795,803,111,795,803,85,803,117,803,85,777,117,777,85,795,769,117,795,769,85,795,768,117,795,768,85,795,777,117,795,777,85,795,771,117,795,771,85,795,803,117,795,803,89,768,121,768,89,803,121,803,89,777,121,777,89,771,121,771,945,787,945,788,945,787,768,945,788,768,945,787,769,945,788,769,945,787,834,945,788,834,913,787,913,788,913,787,768,913,788,768,913,787,769,913,788,769,913,787,834,913,788,834,949,787,949,788,949,787,768,949,788,768,949,787,769,949,788,769,917,787,917,788,917,787,768,917,788,768,917,787,769,917,788,769,951,787,951,788,951,787,768,951,788,768,951,787,769,951,788,769,951,787,834,951,788,834,919,787,919,788,919,787,768,919,788,768,919,787,769,919,788,769,919,787,834,919,788,834,953,787,953,788,953,787,768,953,788,768,953,787,769,953,788,769,953,787,834,953,788,834,921,787,921,788,921,787,768,921,788,768,921,787,769,921,788,769,921,787,834,921,788,834,959,787,959,788,959,787,768,959,788,768,959,787,769,959,788,769,927,787,927,788,927,787,768,927,788,768,927,787,769,927,788,769,965,787,965,788,965,787,768,965,788,768,965,787,769,965,788,769,965,787,834,965,788,834,933,788,933,788,768,933,788,769,933,788,834,969,787,969,788,969,787,768,969,788,768,969,787,769,969,788,769,969,787,834,969,788,834,937,787,937,788,937,787,768,937,788,768,937,787,769,937,788,769,937,787,834,937,788,834,945,768,945,769,949,768,949,769,951,768,951,769,953,768,953,769,959,768,959,769,965,768,965,769,969,768,969,769,945,787,837,945,788,837,945,787,768,837,945,788,768,837,945,787,769,837,945,788,769,837,945,787,834,837,945,788,834,837,913,787,837,913,788,837,913,787,768,837,913,788,768,837,913,787,769,837,913,788,769,837,913,787,834,837,913,
    788,834,837,951,787,837,951,788,837,951,787,768,837,951,788,768,837,951,787,769,837,951,788,769,837,951,787,834,837,951,788,834,837,919,787,837,919,788,837,919,787,768,837,919,788,768,837,919,787,769,837,919,788,769,837,919,787,834,837,919,788,834,837,969,787,837,969,788,837,969,787,768,837,969,788,768,837,969,787,769,837,969,788,769,837,969,787,834,837,969,788,834,837,937,787,837,937,788,837,937,787,768,837,937,788,768,837,937,787,769,837,937,788,769,837,937,787,834,837,937,788,834,837,945,774,945,772,945,768,837,945,837,945,769,837,945,834,945,834,837,913,774,913,772,913,768,913,769,913,837,32,787,953,32,787,32,834,168,834,951,768,837,951,837,951,769,837,951,834,951,834,837,917,768,917,769,919,768,919,769,919,837,8127,768,8127,769,8127,834,953,774,953,772,953,776,768,953,776,769,953,834,953,776,834,921,774,921,772,921,768,921,769,8190,768,8190,769,8190,834,965,774,965,772,965,776,768,965,776,769,961,787,961,788,965,834,965,776,834,933,774,933,772,933,768,933,769,929,788,168,768,168,769,96,969,768,837,969,837,969,769,837,969,834,969,834,837,927,768,927,769,937,768,937,769,937,837,180,32,788,8194,8195,32,32,32,32,32,32,32,32,32,8208,32,819,46,46,46,46,46,46,32,8242,8242,8242,8242,8242,8245,8245,8245,8245,8245,33,33,32,773,63,63,63,33,33,63,8242,8242,8242,8242,32,48,105,52,53,54,55,56,57,43,8722,61,40,41,110,48,49,50,51,52,53,54,55,56,57,43,8722,61,40,41,97,101,111,120,601,104,107,108,109,110,112,115,116,82,115,97,47,99,97,47,115,67,176,67,99,47,111,99,47,117,400,176,70,103,72,72,72,104,295,73,73,76,108,78,78,111,80,81,82,82,82,83,77,84,69,76,84,77,90,937,90,75,65,778,66,67,101,69,70,77,111,1488,1489,1490,1491,105,70,65,88,960,947,915,928,8721,68,100,101,105,106,49,8260,55,49,8260,57,49,8260,49,48,49,8260,51,50,8260,51,49,8260,53,50,8260,53,51,8260,53,52,8260,53,49,8260,54,53,8260,54,49,8260,56,51,8260,56,53,8260,56,55,8260,56,49,8260,73,73,73,73,73,73,73,86,86,86,73,86,73,73,86,73,73,73,73,88,88,88,73,88,73,73,76,67,68,77,105,105,105,105,105,105,105,118,118,118,105,
    118,105,105,118,105,105,105,105,120,120,120,105,120,105,105,108,99,100,109,48,8260,51,8592,824,8594,824,8596,824,8656,824,8660,824,8658,824,8707,824,8712,824,8715,824,8739,824,8741,824,8747,8747,8747,8747,8747,8750,8750,8750,8750,8750,8764,824,8771,824,8773,824,8776,824,61,824,8801,824,8781,824,60,824,62,824,8804,824,8805,824,8818,824,8819,824,8822,824,8823,824,8826,824,8827,824,8834,824,8835,824,8838,824,8839,824,8866,824,8872,824,8873,824,8875,824,8828,824,8829,824,8849,824,8850,824,8882,824,8883,824,8884,824,8885,824,12296,12297,49,50,51,52,53,54,55,56,57,49,48,49,49,49,50,49,51,49,52,49,53,49,54,49,55,49,56,49,57,50,48,40,49,41,40,50,41,40,51,41,40,52,41,40,53,41,40,54,41,40,55,41,40,56,41,40,57,41,40,49,48,41,40,49,49,41,40,49,50,41,40,49,51,41,40,49,52,41,40,49,53,41,40,49,54,41,40,49,55,41,40,49,56,41,40,49,57,41,40,50,48,41,49,46,50,46,51,46,52,46,53,46,54,46,55,46,56,46,57,46,49,48,46,49,49,46,49,50,46,49,51,46,49,52,46,49,53,46,49,54,46,49,55,46,49,56,46,49,57,46,50,48,46,40,97,41,40,98,41,40,99,41,40,100,41,40,101,41,40,102,41,40,103,41,40,104,41,40,105,41,40,106,41,40,107,41,40,108,41,40,109,41,40,110,41,40,111,41,40,112,41,40,113,41,40,114,41,40,115,41,40,116,41,40,117,41,40,118,41,40,119,41,40,120,41,40,121,41,40,122,41,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,48,8747,8747,8747,8747,58,58,61,61,61,61,61,61,10973,824,106,86,11617,27597,40863,19968,20008,20022,20031,20057,20101,20108,20128,20154,20799,20837,20843,20866,20886,20907,20960,20981,20992,21147,21241,21269,21274,21304,21313,21340,21353,21378,21430,21448,21475,22231,22303,22763,22786,22794,22805,22823,22899,23376,23424,23544,23567,23586,23608,23662,23665,24027,24037,24049,24062,24178,24186,24191,24308,24318,24331,24339,24400,24417,24435,24515,25096,25142,25163,25903,25908,25991,26007,26020,26041,26080,26085,26352,26376,26408,27424,27490,27513,27571,27595,
    27604,27611,27663,27668,27700,28779,29226,29238,29243,29247,29255,29273,29275,29356,29572,29577,29916,29926,29976,29983,29992,30000,30091,30098,30326,30333,30382,30399,30446,30683,30690,30707,31034,31160,31166,31348,31435,31481,31859,31992,32566,32593,32650,32701,32769,32780,32786,32819,32895,32905,33251,33258,33267,33276,33292,33307,33311,33390,33394,33400,34381,34411,34880,34892,34915,35198,35211,35282,35328,35895,35910,35925,35960,35997,36196,36208,36275,36523,36554,36763,36784,36789,37009,37193,37318,37324,37329,38263,38272,38428,38582,38585,38632,38737,38750,38754,38761,38859,38893,38899,38913,39080,39131,39135,39318,39321,39340,39592,39640,39647,39717,39727,39730,39740,39770,40165,40565,40575,40613,40635,40643,40653,40657,40697,40701,40718,40723,40736,40763,40778,40786,40845,40860,40864,32,12306,21313,21316,21317,12363,12441,12365,12441,12367,12441,12369,12441,12371,12441,12373,12441,12375,12441,12377,12441,12379,12441,12381,12441,12383,12441,12385,12441,12388,12441,12390,12441,12392,12441,12399,12441,12399,12442,12402,12441,12402,12442,12405,12441,12405,12442,12408,12441,12408,12442,12411,12441,12411,12442,12358,12441,32,12441,32,12442,12445,12441,12424,12426,12459,12441,12461,12441,12463,12441,12465,12441,12467,12441,12469,12441,12471,12441,12473,12441,12475,12441,12477,12441,12479,12441,12481,12441,12484,12441,12486,12441,12488,12441,12495,12441,12495,12442,12498,12441,12498,12442,12501,12441,12501,12442,12504,12441,12504,12442,12507,12441,12507,12442,12454,12441,12527,12441,12528,12441,12529,12441,12530,12441,12541,12441,12467,12488,4352,4353,4522,4354,4524,4525,4355,4356,4357,4528,4529,4530,4531,4532,4533,4378,4358,4359,4360,4385,4361,4362,4363,4364,4365,4366,4367,4368,4369,4370,4449,4450,4451,4452,4453,4454,4455,4456,4457,4458,4459,4460,4461,4462,4463,4464,4465,4466,4467,4468,4469,4448,4372,4373,4551,4552,4556,4558,4563,4567,4569,4380,4573,4575,4381,4382,4384,4386,4387,4391,4393,4395,4396,4397,4398,4399,4402,4406,4416,4423,4428,4593,4594,4439,4440,4441,
    4484,4485,4488,4497,4498,4500,4510,4513,19968,20108,19977,22235,19978,20013,19979,30002,20057,19993,19969,22825,22320,20154,40,4352,41,40,4354,41,40,4355,41,40,4357,41,40,4358,41,40,4359,41,40,4361,41,40,4363,41,40,4364,41,40,4366,41,40,4367,41,40,4368,41,40,4369,41,40,4370,41,40,4352,4449,41,40,4354,4449,41,40,4355,4449,41,40,4357,4449,41,40,4358,4449,41,40,4359,4449,41,40,4361,4449,41,40,4363,4449,41,40,4364,4449,41,40,4366,4449,41,40,4367,4449,41,40,4368,4449,41,40,4369,4449,41,40,4370,4449,41,40,4364,4462,41,40,4363,4457,4364,4453,4523,41,40,4363,4457,4370,4462,41,40,19968,41,40,20108,41,40,19977,41,40,22235,41,40,20116,41,40,20845,41,40,19971,41,40,20843,41,40,20061,41,40,21313,41,40,26376,41,40,28779,41,40,27700,41,40,26408,41,40,37329,41,40,22303,41,40,26085,41,40,26666,41,40,26377,41,40,31038,41,40,21517,41,40,29305,41,40,36001,41,40,31069,41,40,21172,41,40,20195,41,40,21628,41,40,23398,41,40,30435,41,40,20225,41,40,36039,41,40,21332,41,40,31085,41,40,20241,41,40,33258,41,40,33267,41,21839,24188,25991,31631,80,84,69,50,49,50,50,50,51,50,52,50,53,50,54,50,55,50,56,50,57,51,48,51,49,51,50,51,51,51,52,51,53,4352,4354,4355,4357,4358,4359,4361,4363,4364,4366,4367,4368,4369,4370,4352,4449,4354,4449,4355,4449,4357,4449,4358,4449,4359,4449,4361,4449,4363,4449,4364,4449,4366,4449,4367,4449,4368,4449,4369,4449,4370,4449,4366,4449,4535,4352,4457,4364,4462,4363,4468,4363,4462,19968,20108,19977,22235,20116,20845,19971,20843,20061,21313,26376,28779,27700,26408,37329,22303,26085,26666,26377,31038,21517,29305,36001,31069,21172,31192,30007,22899,36969,20778,21360,27880,38917,20241,20889,27491,19978,20013,19979,24038,21491,21307,23447,23398,30435,20225,36039,21332,22812,51,54,51,55,51,56,51,57,52,48,52,49,52,50,52,51,52,52,52,53,52,54,52,55,52,56,52,57,53,48,49,26376,50,26376,51,26376,52,26376,53,26376,54,26376,55,26376,56,26376,57,26376,49,48,26376,49,49,26376,49,50,26376,72,103,101,114,103,101,86,76,84,68,12450,12452,12454,12456,12458,12459,12461,12463,12465,12467,12469,
    12471,12473,12475,12477,12479,12481,12484,12486,12488,12490,12491,12492,12493,12494,12495,12498,12501,12504,12507,12510,12511,12512,12513,12514,12516,12518,12520,12521,12522,12523,12524,12525,12527,12528,12529,12530,12450,12495,12442,12540,12488,12450,12523,12501,12449,12450,12531,12504,12442,12450,12450,12540,12523,12452,12491,12531,12463,12441,12452,12531,12481,12454,12457,12531,12456,12473,12463,12540,12488,12441,12456,12540,12459,12540,12458,12531,12473,12458,12540,12512,12459,12452,12522,12459,12521,12483,12488,12459,12525,12522,12540,12459,12441,12525,12531,12459,12441,12531,12510,12461,12441,12459,12441,12461,12441,12491,12540,12461,12517,12522,12540,12461,12441,12523,12479,12441,12540,12461,12525,12461,12525,12463,12441,12521,12512,12461,12525,12513,12540,12488,12523,12461,12525,12527,12483,12488,12463,12441,12521,12512,12463,12441,12521,12512,12488,12531,12463,12523,12475,12441,12452,12525,12463,12525,12540,12493,12465,12540,12473,12467,12523,12490,12467,12540,12507,12442,12469,12452,12463,12523,12469,12531,12481,12540,12512,12471,12522,12531,12463,12441,12475,12531,12481,12475,12531,12488,12479,12441,12540,12473,12486,12441,12471,12488,12441,12523,12488,12531,12490,12494,12494,12483,12488,12495,12452,12484,12495,12442,12540,12475,12531,12488,12495,12442,12540,12484,12495,12441,12540,12524,12523,12498,12442,12450,12473,12488,12523,12498,12442,12463,12523,12498,12442,12467,12498,12441,12523,12501,12449,12521,12483,12488,12441,12501,12451,12540,12488,12501,12441,12483,12471,12455,12523,12501,12521,12531,12504,12463,12479,12540,12523,12504,12442,12477,12504,12442,12491,12498,12504,12523,12484,12504,12442,12531,12473,12504,12442,12540,12471,12441,12504,12441,12540,12479,12507,12442,12452,12531,12488,12507,12441,12523,12488,12507,12531,12507,12442,12531,12488,12441,12507,12540,12523,12507,12540,12531,12510,12452,12463,12525,12510,12452,12523,12510,12483,12495,12510,12523,12463,12510,12531,12471,12519,12531,12511,12463,12525,12531,12511,12522,12511,12522,12495,
    12441,12540,12523,12513,12459,12441,12513,12459,12441,12488,12531,12513,12540,12488,12523,12516,12540,12488,12441,12516,12540,12523,12518,12450,12531,12522,12483,12488,12523,12522,12521,12523,12498,12442,12540,12523,12540,12501,12441,12523,12524,12512,12524,12531,12488,12465,12441,12531,12527,12483,12488,48,28857,49,28857,50,28857,51,28857,52,28857,53,28857,54,28857,55,28857,56,28857,57,28857,49,48,28857,49,49,28857,49,50,28857,49,51,28857,49,52,28857,49,53,28857,49,54,28857,49,55,28857,49,56,28857,49,57,28857,50,48,28857,50,49,28857,50,50,28857,50,51,28857,50,52,28857,104,80,97,100,97,65,85,98,97,114,111,86,112,99,100,109,100,109,50,100,109,51,73,85,24179,25104,26157,21644,22823,27491,26126,27835,26666,24335,20250,31038,112,65,110,65,956,65,109,65,107,65,75,66,77,66,71,66,99,97,108,107,99,97,108,112,70,110,70,956,70,956,103,109,103,107,103,72,122,107,72,122,77,72,122,71,72,122,84,72,122,956,108,109,108,100,108,107,108,102,109,110,109,956,109,109,109,99,109,107,109,109,109,50,99,109,50,109,50,107,109,50,109,109,51,99,109,51,109,51,107,109,51,109,8725,115,109,8725,115,50,80,97,107,80,97,77,80,97,71,80,97,114,97,100,114,97,100,8725,115,114,97,100,8725,115,50,112,115,110,115,956,115,109,115,112,86,110,86,956,86,109,86,107,86,77,86,112,87,110,87,956,87,109,87,107,87,77,87,107,937,77,937,97,46,109,46,66,113,99,99,99,100,67,8725,107,103,67,111,46,100,66,71,121,104,97,72,80,105,110,75,75,75,77,107,116,108,109,108,110,108,111,103,108,120,109,98,109,105,108,109,111,108,80,72,112,46,109,46,80,80,77,80,82,115,114,83,118,87,98,86,8725,109,65,8725,109,49,26085,50,26085,51,26085,52,26085,53,26085,54,26085,55,26085,56,26085,57,26085,49,48,26085,49,49,26085,49,50,26085,49,51,26085,49,52,26085,49,53,26085,49,54,26085,49,55,26085,49,56,26085,49,57,26085,50,48,26085,50,49,26085,50,50,26085,50,51,26085,50,52,26085,50,53,26085,50,54,26085,50,55,26085,50,56,26085,50,57,26085,51,48,26085,51,49,26085,103,97,108,1098,1100,42863,294,339,42791,43831,619,43858,35912,26356,36554,36040,28369,
    20018,21477,40860,40860,22865,37329,21895,22856,25078,30313,32645,34367,34746,35064,37007,27138,27931,28889,29662,33853,37226,39409,20098,21365,27396,29211,34349,40478,23888,28651,34253,35172,25289,33240,34847,24266,26391,28010,29436,37070,20358,20919,21214,25796,27347,29200,30439,32769,34310,34396,36335,38706,39791,40442,30860,31103,32160,33737,37636,40575,35542,22751,24324,31840,32894,29282,30922,36034,38647,22744,23650,27155,28122,28431,32047,32311,38475,21202,32907,20956,20940,31260,32190,33777,38517,35712,25295,27138,35582,20025,23527,24594,29575,30064,21271,30971,20415,24489,19981,27852,25976,32034,21443,22622,30465,33865,35498,27578,36784,27784,25342,33509,25504,30053,20142,20841,20937,26753,31975,33391,35538,37327,21237,21570,22899,24300,26053,28670,31018,38317,39530,40599,40654,21147,26310,27511,36706,24180,24976,25088,25754,28451,29001,29833,31178,32244,32879,36646,34030,36899,37706,21015,21155,21693,28872,35010,35498,24265,24565,25467,27566,31806,29557,20196,22265,23527,23994,24604,29618,29801,32666,32838,37428,38646,38728,38936,20363,31150,37300,38584,24801,20102,20698,23534,23615,26009,27138,29134,30274,34044,36988,40845,26248,38446,21129,26491,26611,27969,28316,29705,30041,30827,32016,39006,20845,25134,38520,20523,23833,28138,36650,24459,24900,26647,29575,38534,21033,21519,23653,26131,26446,26792,27877,29702,30178,32633,35023,35041,37324,38626,21311,28346,21533,29136,29848,34298,38563,40023,40607,26519,28107,33256,31435,31520,31890,29376,28825,35672,20160,33590,21050,20999,24230,25299,31958,23429,27934,26292,36667,34892,38477,35211,24275,20800,21952,22618,26228,20958,29482,30410,31036,31070,31077,31119,38742,31934,32701,34322,35576,36920,37117,39151,39164,39208,40372,37086,38583,20398,20711,20813,21193,21220,21329,21917,22022,22120,22592,22696,23652,23662,24724,24936,24974,25074,25935,26082,26257,26757,28023,28186,28450,29038,29227,29730,30865,31038,31049,31048,31056,31062,31069,31117,31118,31296,31361,31680,32244,32265,32321,32626,32773,33261,33401,
    33401,33879,35088,35222,35585,35641,36051,36104,36790,36920,38627,38911,38971,24693,148206,33304,20006,20917,20840,20352,20805,20864,21191,21242,21917,21845,21913,21986,22618,22707,22852,22868,23138,23336,24274,24281,24425,24493,24792,24910,24840,24974,24928,25074,25140,25540,25628,25682,25942,26228,26391,26395,26454,27513,27578,27969,28379,28363,28450,28702,29038,30631,29237,29359,29482,29809,29958,30011,30237,30239,30410,30427,30452,30538,30528,30924,31409,31680,31867,32091,32244,32574,32773,33618,33775,34681,35137,35206,35222,35519,35576,35531,35585,35582,35565,35641,35722,36104,36664,36978,37273,37494,38524,38627,38742,38875,38911,38923,38971,39698,40860,141386,141380,144341,15261,16408,16441,152137,154832,163539,40771,40846,102,102,102,105,102,108,102,102,105,102,102,108,115,116,115,116,1396,1398,1396,1381,1396,1387,1406,1398,1396,1389,1497,1460,1522,1463,1506,1488,1491,1492,1499,1500,1501,1512,1514,43,1513,1473,1513,1474,1513,1468,1473,1513,1468,1474,1488,1463,1488,1464,1488,1468,1489,1468,1490,1468,1491,1468,1492,1468,1493,1468,1494,1468,1496,1468,1497,1468,1498,1468,1499,1468,1500,1468,1502,1468,1504,1468,1505,1468,1507,1468,1508,1468,1510,1468,1511,1468,1512,1468,1513,1468,1514,1468,1493,1465,1489,1471,1499,1471,1508,1471,1488,1500,1649,1649,1659,1659,1659,1659,1662,1662,1662,1662,1664,1664,1664,1664,1658,1658,1658,1658,1663,1663,1663,1663,1657,1657,1657,1657,1700,1700,1700,1700,1702,1702,1702,1702,1668,1668,1668,1668,1667,1667,1667,1667,1670,1670,1670,1670,1671,1671,1671,1671,1677,1677,1676,1676,1678,1678,1672,1672,1688,1688,1681,1681,1705,1705,1705,1705,1711,1711,1711,1711,1715,1715,1715,1715,1713,1713,1713,1713,1722,1722,1723,1723,1723,1723,1749,1620,1749,1620,1729,1729,1729,1729,1726,1726,1726,1726,1746,1746,1746,1620,1746,1620,1709,1709,1709,1709,1735,1735,1734,1734,1736,1736,1735,1652,1739,1739,1733,1733,1737,1737,1744,1744,1744,1744,1609,1609,1610,1620,1575,1610,1620,1575,1610,1620,1749,1610,1620,1749,1610,1620,1608,1610,1620,1608,1610,1620,1735,
    1610,1620,1735,1610,1620,1734,1610,1620,1734,1610,1620,1736,1610,1620,1736,1610,1620,1744,1610,1620,1744,1610,1620,1744,1610,1620,1609,1610,1620,1609,1610,1620,1609,1740,1740,1740,1740,1610,1620,1580,1610,1620,1581,1610,1620,1605,1610,1620,1609,1610,1620,1610,1576,1580,1576,1581,1576,1582,1576,1605,1576,1609,1576,1610,1578,1580,1578,1581,1578,1582,1578,1605,1578,1609,1578,1610,1579,1580,1579,1605,1579,1609,1579,1610,1580,1581,1580,1605,1581,1580,1581,1605,1582,1580,1582,1581,1582,1605,1587,1580,1587,1581,1587,1582,1587,1605,1589,1581,1589,1605,1590,1580,1590,1581,1590,1582,1590,1605,1591,1581,1591,1605,1592,1605,1593,1580,1593,1605,1594,1580,1594,1605,1601,1580,1601,1581,1601,1582,1601,1605,1601,1609,1601,1610,1602,1581,1602,1605,1602,1609,1602,1610,1603,1575,1603,1580,1603,1581,1603,1582,1603,1604,1603,1605,1603,1609,1603,1610,1604,1580,1604,1581,1604,1582,1604,1605,1604,1609,1604,1610,1605,1580,1605,1581,1605,1582,1605,1605,1605,1609,1605,1610,1606,1580,1606,1581,1606,1582,1606,1605,1606,1609,1606,1610,1607,1580,1607,1605,1607,1609,1607,1610,1610,1580,1610,1581,1610,1582,1610,1605,1610,1609,1610,1610,1584,1648,1585,1648,1609,1648,32,1612,1617,32,1613,1617,32,1614,1617,32,1615,1617,32,1616,1617,32,1617,1648,1610,1620,1585,1610,1620,1586,1610,1620,1605,1610,1620,1606,1610,1620,1609,1610,1620,1610,1576,1585,1576,1586,1576,1605,1576,1606,1576,1609,1576,1610,1578,1585,1578,1586,1578,1605,1578,1606,1578,1609,1578,1610,1579,1585,1579,1586,1579,1605,1579,1606,1579,1609,1579,1610,1601,1609,1601,1610,1602,1609,1602,1610,1603,1575,1603,1604,1603,1605,1603,1609,1603,1610,1604,1605,1604,1609,1604,1610,1605,1575,1605,1605,1606,1585,1606,1586,1606,1605,1606,1606,1606,1609,1606,1610,1609,1648,1610,1585,1610,1586,1610,1605,1610,1606,1610,1609,1610,1610,1610,1620,1580,1610,1620,1581,1610,1620,1582,1610,1620,1605,1610,1620,1607,1576,1580,1576,1581,1576,1582,1576,1605,1576,1607,1578,1580,1578,1581,1578,1582,1578,1605,1578,1607,1579,1605,1580,1581,1580,1605,1581,1580,1581,1605,1582,
    1580,1582,1605,1587,1580,1587,1581,1587,1582,1587,1605,1589,1581,1589,1582,1589,1605,1590,1580,1590,1581,1590,1582,1590,1605,1591,1581,1592,1605,1593,1580,1593,1605,1594,1580,1594,1605,1601,1580,1601,1581,1601,1582,1601,1605,1602,1581,1602,1605,1603,1580,1603,1581,1603,1582,1603,1604,1603,1605,1604,1580,1604,1581,1604,1582,1604,1605,1604,1607,1605,1580,1605,1581,1605,1582,1605,1605,1606,1580,1606,1581,1606,1582,1606,1605,1606,1607,1607,1580,1607,1605,1607,1648,1610,1580,1610,1581,1610,1582,1610,1605,1610,1607,1610,1620,1605,1610,1620,1607,1576,1605,1576,1607,1578,1605,1578,1607,1579,1605,1579,1607,1587,1605,1587,1607,1588,1605,1588,1607,1603,1604,1603,1605,1604,1605,1606,1605,1606,1607,1610,1605,1610,1607,1600,1614,1617,1600,1615,1617,1600,1616,1617,1591,1609,1591,1610,1593,1609,1593,1610,1594,1609,1594,1610,1587,1609,1587,1610,1588,1609,1588,1610,1581,1609,1581,1610,1580,1609,1580,1610,1582,1609,1582,1610,1589,1609,1589,1610,1590,1609,1590,1610,1588,1580,1588,1581,1588,1582,1588,1605,1588,1585,1587,1585,1589,1585,1590,1585,1591,1609,1591,1610,1593,1609,1593,1610,1594,1609,1594,1610,1587,1609,1587,1610,1588,1609,1588,1610,1581,1609,1581,1610,1580,1609,1580,1610,1582,1609,1582,1610,1589,1609,1589,1610,1590,1609,1590,1610,1588,1580,1588,1581,1588,1582,1588,1605,1588,1585,1587,1585,1589,1585,1590,1585,1588,1580,1588,1581,1588,1582,1588,1605,1587,1607,1588,1607,1591,1605,1587,1580,1587,1581,1587,1582,1588,1580,1588,1581,1588,1582,1591,1605,1592,1605,1575,1611,1575,1611,1578,1580,1605,1578,1581,1580,1578,1581,1580,1578,1581,1605,1578,1582,1605,1578,1605,1580,1578,1605,1581,1578,1605,1582,1580,1605,1581,1580,1605,1581,1581,1605,1610,1581,1605,1609,1587,1581,1580,1587,1580,1581,1587,1580,1609,1587,1605,1581,1587,1605,1581,1587,1605,1580,1587,1605,1605,1587,1605,1605,1589,1581,1581,1589,1581,1581,1589,1605,1605,1588,1581,1605,1588,1581,1605,1588,1580,1610,1588,1605,1582,1588,1605,1582,1588,1605,1605,1588,1605,1605,1590,1581,1609,1590,1582,1605,1590,1582,1605,1591,1605,1581,
    1591,1605,1581,1591,1605,1605,1591,1605,1610,1593,1580,1605,1593,1605,1605,1593,1605,1605,1593,1605,1609,1594,1605,1605,1594,1605,1610,1594,1605,1609,1601,1582,1605,1601,1582,1605,1602,1605,1581,1602,1605,1605,1604,1581,1605,1604,1581,1610,1604,1581,1609,1604,1580,1580,1604,1580,1580,1604,1582,1605,1604,1582,1605,1604,1605,1581,1604,1605,1581,1605,1581,1580,1605,1581,1605,1605,1581,1610,1605,1580,1581,1605,1580,1605,1605,1582,1580,1605,1582,1605,1605,1580,1582,1607,1605,1580,1607,1605,1605,1606,1581,1605,1606,1581,1609,1606,1580,1605,1606,1580,1605,1606,1580,1609,1606,1605,1610,1606,1605,1609,1610,1605,1605,1610,1605,1605,1576,1582,1610,1578,1580,1610,1578,1580,1609,1578,1582,1610,1578,1582,1609,1578,1605,1610,1578,1605,1609,1580,1605,1610,1580,1581,1609,1580,1605,1609,1587,1582,1609,1589,1581,1610,1588,1581,1610,1590,1581,1610,1604,1580,1610,1604,1605,1610,1610,1581,1610,1610,1580,1610,1610,1605,1610,1605,1605,1610,1602,1605,1610,1606,1581,1610,1602,1605,1581,1604,1581,1605,1593,1605,1610,1603,1605,1610,1606,1580,1581,1605,1582,1610,1604,1580,1605,1603,1605,1605,1604,1580,1605,1606,1580,1581,1580,1581,1610,1581,1580,1610,1605,1580,1610,1601,1605,1610,1576,1581,1610,1603,1605,1605,1593,1580,1605,1589,1605,1605,1587,1582,1610,1606,1580,1610,1589,1604,1746,1602,1604,1746,1575,1604,1604,1607,1575,1603,1576,1585,1605,1581,1605,1583,1589,1604,1593,1605,1585,1587,1608,1604,1593,1604,1610,1607,1608,1587,1604,1605,1589,1604,1609,1589,1604,1609,32,1575,1604,1604,1607,32,1593,1604,1610,1607,32,1608,1587,1604,1605,1580,1604,32,1580,1604,1575,1604,1607,1585,1740,1575,1604,44,12289,12290,58,59,33,63,12310,12311,46,46,46,46,46,8212,8211,95,95,40,41,123,125,12308,12309,12304,12305,12298,12299,12296,12297,12300,12301,12302,12303,91,93,32,773,32,773,32,773,32,773,95,95,95,44,12289,46,59,58,63,33,8212,40,41,123,125,12308,12309,35,38,42,43,45,60,62,61,92,36,37,64,32,1611,1600,1611,32,1612,32,1613,32,1614,1600,1614,32,1615,1600,1615,32,1616,1600,1616,32,1617,1600,1617,32,1618,1600,
    1618,1569,1575,1619,1575,1619,1575,1620,1575,1620,1608,1620,1608,1620,1575,1621,1575,1621,1610,1620,1610,1620,1610,1620,1610,1620,1575,1575,1576,1576,1576,1576,1577,1577,1578,1578,1578,1578,1579,1579,1579,1579,1580,1580,1580,1580,1581,1581,1581,1581,1582,1582,1582,1582,1583,1583,1584,1584,1585,1585,1586,1586,1587,1587,1587,1587,1588,1588,1588,1588,1589,1589,1589,1589,1590,1590,1590,1590,1591,1591,1591,1591,1592,1592,1592,1592,1593,1593,1593,1593,1594,1594,1594,1594,1601,1601,1601,1601,1602,1602,1602,1602,1603,1603,1603,1603,1604,1604,1604,1604,1605,1605,1605,1605,1606,1606,1606,1606,1607,1607,1607,1607,1608,1608,1609,1609,1610,1610,1610,1610,1604,1575,1619,1604,1575,1619,1604,1575,1620,1604,1575,1620,1604,1575,1621,1604,1575,1621,1604,1575,1604,1575,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,64,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,91,92,93,94,95,96,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,10629,10630,12290,12300,12301,12289,12539,12530,12449,12451,12453,12455,12457,12515,12517,12519,12483,12540,12450,12452,12454,12456,12458,12459,12461,12463,12465,12467,12469,12471,12473,12475,12477,12479,12481,12484,12486,12488,12490,12491,12492,12493,12494,12495,12498,12501,12504,12507,12510,12511,12512,12513,12514,12516,12518,12520,12521,12522,12523,12524,12525,12527,12531,12441,12442,4448,4352,4353,4522,4354,4524,4525,4355,4356,4357,4528,4529,4530,4531,4532,4533,4378,4358,4359,4360,4385,4361,4362,4363,4364,4365,4366,4367,4368,4369,4370,4449,4450,4451,4452,4453,4454,4455,4456,4457,4458,4459,4460,4461,4462,4463,4464,4465,4466,4467,4468,4469,162,163,172,32,772,166,165,8361,9474,8592,8593,8594,8595,9632,9675,69785,69818,69787,69818,69797,69818,69937,69927,69938,69927,70471,70462,70471,70487,70841,70842,70841,70832,70841,70845,71096,71087,71097,71087,119127,119141,119128,119141,119128,119141,119150,119128,119141,119151,
    119128,119141,119152,119128,119141,119153,119128,119141,119154,119225,119141,119226,119141,119225,119141,119150,119226,119141,119150,119225,119141,119151,119226,119141,119151,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,97,98,99,100,101,102,103,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,65,67,68,71,74,75,78,79,80,81,83,84,85,86,87,88,89,90,97,98,99,100,102,104,105,106,107,108,109,110,112,113,114,115,116,117,118,119,120,121,122,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,65,66,68,69,70,71,74,75,76,77,78,79,80,81,83,84,85,86,87,88,89,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,65,66,68,69,70,71,73,74,75,76,77,79,83,84,85,86,87,88,89,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,97,98,99,100,101,102,103,104,105,
    106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,305,567,913,914,915,916,917,918,919,920,921,922,923,924,925,926,927,928,929,920,931,932,933,934,935,936,937,8711,945,946,947,948,949,950,951,952,953,954,955,956,957,958,959,960,961,962,963,964,965,966,967,968,969,8706,949,952,954,966,961,960,913,914,915,916,917,918,919,920,921,922,923,924,925,926,927,928,929,920,931,932,933,934,935,936,937,8711,945,946,947,948,949,950,951,952,953,954,955,956,957,958,959,960,961,962,963,964,965,966,967,968,969,8706,949,952,954,966,961,960,913,914,915,916,917,918,919,920,921,922,923,924,925,926,927,928,929,920,931,932,933,934,935,936,937,8711,945,946,947,948,949,950,951,952,953,954,955,956,957,958,959,960,961,962,963,964,965,966,967,968,969,8706,949,952,954,966,961,960,913,914,915,916,917,918,919,920,921,922,923,924,925,926,927,928,929,920,931,932,933,934,935,936,937,8711,945,946,947,948,949,950,951,952,953,954,955,956,957,958,959,960,961,962,963,964,965,966,967,968,969,8706,949,952,954,966,961,960,913,914,915,916,917,918,919,920,921,922,923,924,925,926,927,928,929,920,931,932,933,934,935,936,937,8711,945,946,947,948,949,950,951,952,953,954,955,956,957,958,959,960,961,962,963,964,965,966,967,968,969,8706,949,952,954,966,961,960,988,989,48,49,50,51,52,53,54,55,56,57,48,49,50,51,52,53,54,55,56,57,48,49,50,51,52,53,54,55,56,57,48,49,50,51,52,53,54,55,56,57,48,49,50,51,52,53,54,55,56,57,1575,1576,1580,1583,1608,1586,1581,1591,1610,1603,1604,1605,1606,1587,1593,1601,1589,1602,1585,1588,1578,1579,1582,1584,1590,1592,1594,1646,1722,1697,1647,1576,1580,1607,1581,1610,1603,1604,1605,1606,1587,1593,1601,1589,1602,1588,1578,
    1579,1582,1590,1594,1580,1581,1610,1604,1606,1587,1593,1589,1602,1588,1582,1590,1594,1722,1647,1576,1580,1607,1581,1591,1610,1603,1605,1606,1587,1593,1601,1589,1602,1588,1578,1579,1582,1590,1592,1594,1646,1697,1575,1576,1580,1583,1607,1608,1586,1581,1591,1610,1604,1605,1606,1587,1593,1601,1589,1602,1585,1588,1578,1579,1582,1584,1590,1592,1594,1576,1580,1583,1608,1586,1581,1591,1610,1604,1605,1606,1587,1593,1601,1589,1602,1585,1588,1578,1579,1582,1584,1590,1592,1594,48,46,48,44,49,44,50,44,51,44,52,44,53,44,54,44,55,44,56,44,57,44,40,65,41,40,66,41,40,67,41,40,68,41,40,69,41,40,70,41,40,71,41,40,72,41,40,73,41,40,74,41,40,75,41,40,76,41,40,77,41,40,78,41,40,79,41,40,80,41,40,81,41,40,82,41,40,83,41,40,84,41,40,85,41,40,86,41,40,87,41,40,88,41,40,89,41,40,90,41,12308,83,12309,67,82,67,68,87,90,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,72,86,77,86,83,68,83,83,80,80,86,87,67,77,67,77,68,68,74,12411,12363,12467,12467,12469,25163,23383,21452,12486,12441,20108,22810,35299,22825,20132,26144,28961,26009,21069,24460,20877,26032,21021,32066,29983,36009,22768,21561,28436,25237,25429,19968,19977,36938,24038,20013,21491,25351,36208,25171,31105,31354,21512,28288,26377,26376,30003,21106,21942,12308,26412,12309,12308,19977,12309,12308,20108,12309,12308,23433,12309,12308,28857,12309,12308,25171,12309,12308,30423,12309,12308,21213,12309,12308,25943,12309,24471,21487,20029,20024,20033,131362,20320,20398,20411,20482,20602,20633,20711,20687,13470,132666,20813,20820,20836,20855,132380,13497,20839,20877,132427,20887,20900,20172,20908,20917,168415,20981,20995,13535,21051,21062,21106,21111,13589,21191,21193,21220,21242,21253,21254,21271,21321,21329,21338,21363,21373,21375,21375,21375,133676,28784,21450,21471,133987,21483,21489,21510,21662,21560,21576,21608,21666,21750,21776,21843,21859,21892,21892,21913,21931,21939,21954,22294,22022,22295,22097,22132,20999,22766,22478,22516,22541,22411,22578,22577,22700,136420,22770,22775,22790,22810,22818,22882,136872,
    136938,23020,23067,23079,23000,23142,14062,14076,23304,23358,23358,137672,23491,23512,23527,23539,138008,23551,23558,24403,23586,14209,23648,23662,23744,23693,138724,23875,138726,23918,23915,23932,24033,24034,14383,24061,24104,24125,24169,14434,139651,14460,24240,24243,24246,24266,172946,24318,140081,140081,33281,24354,24354,14535,144056,156122,24418,24427,14563,24474,24525,24535,24569,24705,14650,14620,24724,141012,24775,24904,24908,24910,24908,24954,24974,25010,24996,25007,25054,25074,25078,25104,25115,25181,25265,25300,25424,142092,25405,25340,25448,25475,25572,142321,25634,25541,25513,14894,25705,25726,25757,25719,14956,25935,25964,143370,26083,26360,26185,15129,26257,15112,15076,20882,20885,26368,26268,32941,17369,26391,26395,26401,26462,26451,144323,15177,26618,26501,26706,26757,144493,26766,26655,26900,15261,26946,27043,27114,27304,145059,27355,15384,27425,145575,27476,15438,27506,27551,27578,27579,146061,138507,146170,27726,146620,27839,27853,27751,27926,27966,28023,27969,28009,28024,28037,146718,27956,28207,28270,15667,28363,28359,147153,28153,28526,147294,147342,28614,28729,28702,28699,15766,28746,28797,28791,28845,132389,28997,148067,29084,148395,29224,29237,29264,149000,29312,29333,149301,149524,29562,29579,16044,29605,16056,16056,29767,29788,29809,29829,29898,16155,29988,150582,30014,150674,30064,139679,30224,151457,151480,151620,16380,16392,30452,151795,151794,151833,151859,30494,30495,30495,30538,16441,30603,16454,16534,152605,30798,30860,30924,16611,153126,31062,153242,153285,31119,31211,16687,31296,31306,31311,153980,154279,154279,31470,16898,154539,31686,31689,16935,154752,31954,17056,31976,31971,32000,155526,32099,17153,32199,32258,32325,17204,156200,156231,17241,156377,32634,156478,32661,32762,32773,156890,156963,32864,157096,32880,144223,17365,32946,33027,17419,33086,23221,157607,157621,144275,144284,33281,33284,36766,17515,33425,33419,33437,21171,33457,33459,33469,33510,158524,33509,33565,33635,33709,33571,33725,33767,33879,33619,33738,33740,
    33756,158774,159083,158933,17707,34033,34035,34070,160714,34148,159532,17757,17761,159665,159954,17771,34384,34396,34407,34409,34473,34440,34574,34530,34681,34600,34667,34694,17879,34785,34817,17913,34912,34915,161383,35031,35038,17973,35066,13499,161966,162150,18110,18119,35488,35565,35722,35925,162984,36011,36033,36123,36215,163631,133124,36299,36284,36336,133342,36564,36664,165330,165357,37012,37105,37137,165678,37147,37432,37591,37592,37500,37881,37909,166906,38283,18837,38327,167287,18918,38595,23986,38691,168261,168474,19054,19062,38880,168970,19122,169110,38923,38923,38953,169398,39138,19251,39209,39335,39362,39422,19406,170800,39698,40000,40189,19662,19693,40295,172238,19704,172293,172558,172689,40635,19798,40697,40702,40709,40719,40726,40763,173568
};

} // namespace unilib
} // namespace nametag
} // namespace ufal
