% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PlotEnrichment.R
\name{PlotEnrichment}
\alias{PlotEnrichment}
\title{Plot the Phylostratum or Divergence Stratum Enrichment of a given Gene Set}
\usage{
PlotEnrichment(ExpressionSet, test.set, use.only.map = FALSE,
  measure = "log-foldchange", complete.bg = TRUE, legendName = "",
  over.col = "steelblue", under.col = "midnightblue", epsilon = 1e-05,
  cex.legend = 1, cex.asterisk = 1, plot.bars = TRUE,
  p.adjust.method = NULL, ...)
}
\arguments{
\item{ExpressionSet}{a standard PhyloExpressionSet or DivergenceExpressionSet object (in case \code{only.map = FALSE}).}

\item{test.set}{a character vector storing the gene ids for which PS/DS enrichment analyses should be performed.}

\item{use.only.map}{a logical value indicating whether instead of a standard \code{ExpressionSet} only a \code{Phylostratigraphic Map} or \code{Divergene Map} is passed to this function.}

\item{measure}{a character string specifying the measure that should be used to quantify over and under representation of PS/DS. Measures can either be \code{measure = "foldchange"} (odds) or \code{measure = "log-foldchange"} (log-odds).}

\item{complete.bg}{a logical value indicating whether the entire background set
of the input \code{ExpressionSet} should be considered when performing Fisher's exact
test (\code{complete.bg = TRUE}) or whether genes that are stored in \code{test.set}
should be excluded from the background set before performing Fisher's exact test
(\code{complete.bg = FALSE}).}

\item{legendName}{a character string specifying whether "PS" or "DS" are used to compute relative expression profiles.}

\item{over.col}{color of the overrepresentation bars.}

\item{under.col}{color of the underrepresentation bars.}

\item{epsilon}{a small value to shift values by epsilon to avoid log(0) computations.}

\item{cex.legend}{the \code{cex} value for the legend.}

\item{cex.asterisk}{the \code{cex} value for the asterisk.}

\item{plot.bars}{a logical value specifying whether or not bars should be visualized or whether only \code{p.values} and \code{enrichment.matrix} should be returned.}

\item{p.adjust.method}{correction method to adjust p-values for multiple comparisons (see \code{\link{p.adjust}} for possible methods). E.g., \code{p.adjust.method = "BH"} (Benjamini & Hochberg (1995)) or \code{p.adjust.method = "bonferroni"} (Bonferroni correction).}

\item{...}{default graphics parameters.}
}
\description{
This function computes and visualizes the significance of enriched (over or underrepresented) Phylostrata or Divergence Strata within an input \code{test.set}.
}
\details{
This \emph{Phylostratum} or \emph{Divergence Stratum} enrichment analysis is motivated 
by Sestak and Domazet-Loso (2015) who perform \emph{Phylostratum} or \emph{Divergence Stratum} enrichment analyses to correlate organ evolution with the origin of organ specific genes. 

In detail this function takes the \emph{Phylostratum} or \emph{Divergence Stratum} distribution of all genes stored in the input \code{ExpressionSet} as background set and
the \emph{Phylostratum} or \emph{Divergence Stratum} distribution of the \code{test.set} and performes a \code{\link{fisher.test}} for each \emph{Phylostratum} or \emph{Divergence Stratum} to quantify the statistical significance of over- or underrepresentated \emph{Phylostrata} or \emph{Divergence Strata} within the set of selected \code{test.set} genes. 

To visualize the odds or log-odds of over or underrepresented genes within the \code{test.set} the following procedure is performed:

\itemize{
\item N_ij denotes the number of genes in group j and deriving from PS i, with \emph{i = 1, .. , n} and where \emph{j = 1} denotes the background set and \emph{j = 2} denotes the \code{test.set}
\item N_i. denotes the total number of genes within PS i
\item N_.j denotes the total number of genes within group j
\item N_.. is the total number of genes within all groups j and all PS i
\item f_ij = N_ij / N_.. and g_ij = f_ij / f_.j denote relative frequencies between groups
\item f_i. denotes the between group sum of f_ij
}

 The result is the fold-change value (odds) denoted as C = g_i2 / f_i. which is visualized above and below zero. 
 
 In case a large number of Phylostrata or Divergence Strata is included in the input 
 \code{ExpressionSet}, p-values returned by \code{PlotEnrichment} should be adjusted for
 multiple comparisons which can be done by specifying the \code{p.adjust.method} argument.
}
\examples{

data(PhyloExpressionSetExample)

set.seed(123)
test_set <- sample(PhyloExpressionSetExample[ , 2],10000)

## Examples with complete.bg = TRUE
## Hence: the entire background set of the input ExpressionSet is considered 
## when performing Fisher's exact test 

# measure: log-foldchange
PlotEnrichment(ExpressionSet = PhyloExpressionSetExample,
               test.set      = test_set , 
               legendName    = "PS", 
               measure       = "log-foldchange")
               
               
# measure: foldchange
PlotEnrichment(ExpressionSet = PhyloExpressionSetExample,
               test.set      = test_set , 
               legendName    = "PS", 
               measure       = "foldchange")
   
               
## Examples with complete.bg = FALSE
## Hence: the test.set genes are excluded from the background set before
## Fisher's exact test is performed
     
                                       
# measure: log-foldchange
PlotEnrichment(ExpressionSet = PhyloExpressionSetExample,
               test.set      = test_set ,
                complete.bg  = FALSE,
               legendName    = "PS", 
               measure       = "log-foldchange")
               
               
# measure: foldchange
PlotEnrichment(ExpressionSet = PhyloExpressionSetExample,
               test.set      = test_set , 
               complete.bg   = FALSE,
               legendName    = "PS", 
               measure       = "foldchange")     
               
}
\author{
Hajk-Georg Drost
}
\references{
Sestak and Domazet-Loso (2015). Phylostratigraphic Profiles in Zebrafish Uncover Chordate Origins of the Vertebrate Brain. Mol. Biol. Evol. 32(2): 299-312.
}
\seealso{
\code{\link{EnrichmentTest}}, \code{\link{fisher.test}}
}

