# MODELTIME FIT WORKFLOWSET ----

#' Fit a `workflowset` object to one or multiple time series
#'
#' This is a wrapper for `fit()` that takes a
#' `workflowset` object and fits each model on one or multiple
#' time series either sequentially or in parallel.
#'
#' @param object A workflow_set object, generated with the workflowsets::workflow_set function.
#' @param data A `tibble` that contains data to fit the models.
#' @param control An object used to modify the fitting process. See [control_fit_workflowset()].
#' @param ... Not currently used.
#'
#' @seealso
#' [control_fit_workflowset()]
#'
#' @examples
#' library(tidymodels)
#' library(modeltime)
#' library(workflowsets)
#' library(tidyverse)
#' library(lubridate)
#' library(timetk)
#'
#' data_set <- m4_monthly
#'
#' # SETUP WORKFLOWSETS
#'
#' rec1 <- recipe(value ~ date + id, data_set) %>%
#'     step_mutate(date_num = as.numeric(date)) %>%
#'     step_mutate(month_lbl = lubridate::month(date, label = TRUE)) %>%
#'     step_dummy(all_nominal(), one_hot = TRUE)
#'
#' mod1 <- linear_reg() %>% set_engine("lm")
#'
#' mod2 <- prophet_reg() %>% set_engine("prophet")
#'
#' wfsets <- workflowsets::workflow_set(
#'     preproc = list(rec1 = rec1),
#'     models  = list(
#'         mod1 = mod1,
#'         mod2 = mod2
#'     ),
#'     cross   = TRUE
#' )
#'
#' # FIT WORKFLOWSETS
#' # - Returns a Modeltime Table with fitted workflowsets
#'
#' wfsets %>% modeltime_fit_workflowset(data_set)
#'
#' @return
#' A Modeltime Table containing one or more fitted models.
#' @export
modeltime_fit_workflowset <- function(object, data, ..., control = control_fit_workflowset()) {

    if (!inherits(object, "workflow_set")){
        rlang::abort("object argument must be a `workflow_set` object generated by workflowsets::workflow_set() function.")
    }

    # Parallel or Sequential
    if ((control$cores > 1) && control$allow_par) {
        models <- modeltime_fit_workflowset_parallel(object, data = data, control = control, ...)
    } else {
        models <- modeltime_fit_workflowset_sequential(object, data = data, control = control, ...)
    }

    names(models) <- NULL

    modeltime_tbl <- models %>% as_modeltime_table_from_workflowset()

    return(modeltime_tbl)

}

modeltime_fit_workflowset_sequential <- function(object, data, control, ...) {

    t1 <- Sys.time()


    .models  <- object %>% split(.$wflow_id)
    safe_fit <- purrr::safely(parsnip::fit, otherwise = NULL, quiet = TRUE)

    # Setup progress

    # BEGIN LOOP
    # if (control$verbose) {
    #     t <- Sys.time()
    #     message(stringr::str_glue("Beginning Sequential Loop | {round(t-t1, 3)} seconds"))
    # }

    models <- .models %>%
        purrr::imap(
            .f = function(obj, id) {

                if (control$verbose) {
                    cli::cli_alert_info(cli::col_grey("Fitting Model: {id}"))
                }

                mod <- obj %>%
                    dplyr::pull(2) %>%
                    purrr::pluck(1, 'workflow', 1)

                ret <- safe_fit(mod, data = data)

                res <- ret %>% purrr::pluck("result")

                if (!is.null(ret$error)) message(stringr::str_glue("Model {id} Error: {ret$error}"))

                if (control$verbose) {
                    if (is.null(res)) {
                        cli::cli_alert_danger(cli::col_grey("Model Failed Fitting: {id}"))
                    } else {
                        cli::cli_alert_success(cli::col_grey("Model Successfully Fitted: {id}"))
                    }
                }

                return(res)


            }
        )

    # PRINT TOTAL TIME
    if (control$verbose) {
        t <- Sys.time()
        message(stringr::str_glue("Total time | {round(t-t1, 3)} seconds"))
    }

    return(models)


}

modeltime_fit_workflowset_parallel <- function(object, data, control, ...) {

    t1 <- Sys.time()

    is_par_setup <- foreach::getDoParWorkers() > 1

    .models <- object %>% split(.$wflow_id)

    clusters_made <- FALSE

    # If parallel processing is not set up, set up parallel backend
    if ((control$cores > 1) && control$allow_par && (!is_par_setup)){
        if (control$verbose) message(stringr::str_glue("Starting parallel backend with {control$cores} clusters (cores)..."))
        cl <- parallel::makeCluster(control$cores)
        doParallel::registerDoParallel(cl)
        parallel::clusterCall(cl, function(x) .libPaths(x), .libPaths())
        clusters_made <- TRUE

        if (control$verbose) {
            t <- Sys.time()
            message(stringr::str_glue(" Parallel Backend Setup | {round(t-t1, 3)} seconds"))
        }

    } else if (!is_par_setup) {
        # Run sequentially if parallel is not set up, cores == 1 or allow_par == FALSE
        if (control$verbose) message(stringr::str_glue("Running sequential backend. If parallel was intended, set `allow_par = TRUE` and `cores > 1`."))
        foreach::registerDoSEQ()
    } else {
        # Parallel was set up externally by user - Do nothing.
        if (control$verbose) message(stringr::str_glue("Using existing parallel backend with {foreach::getDoParWorkers()} clusters (cores)..."))
    }

    # Setup Foreach
    get_operator <- function(allow_par = TRUE) {
        is_par <- foreach::getDoParWorkers() > 1

        cond <- allow_par && is_par
        if (cond) {
            res <- foreach::`%dopar%`
        } else {
            res <- foreach::`%do%`
        }
        return(res)
    }

    `%op%` <- get_operator(allow_par = control$allow_par)

    # Setup Safe Modeling
    safe_fit <- purrr::safely(parsnip::fit, otherwise = NULL, quiet = FALSE)

    # BEGIN LOOP
    if (control$verbose) {
        t <- Sys.time()
        message(stringr::str_glue(" Beginning Parallel Loop | {round(t-t1, 3)} seconds"))
    }

    ret <- foreach::foreach(
        this_model          = .models,
        .inorder            = TRUE,
        .packages           = control$packages,
        .verbose            = FALSE
    ) %op% {

        mod <- this_model %>%
            dplyr::pull(2) %>%
            purrr::pluck(1, 'workflow', 1) %>%
            safe_fit(data)

        res <- mod %>%
            purrr::pluck("result")

        err <- mod %>%
            purrr::pluck("error")

        return(list(res = res, err = err))

    }

    # Collect models
    models <- ret %>% purrr::map(~ .x$res)

    # Collect errors
    error_messages <- ret %>% purrr::map(~ .x$err)
    purrr::iwalk(
        error_messages,
        function (e, id) {
            if (!is.null(e)) message(stringr::str_glue("Model {id} Error: {e}"))
        }
    )

    # Finish Parallel Backend. Close clusters if we set up internally.
    t <- Sys.time()
    if (clusters_made) {
        # We set up parallel processing internally. We should close.
        doParallel::stopImplicitCluster()
        parallel::stopCluster(cl)
        foreach::registerDoSEQ()
        if (control$verbose) {
            message(stringr::str_glue(" Finishing parallel backend. Closing clusters. | {round(t-t1, 3)} seconds)"))
        }
    } else if ((control$cores > 1) && control$allow_par) {
        if (control$verbose) {
            message(stringr::str_glue(" Finishing parallel backend. Clusters are remaining open. | {round(t-t1, 3)} seconds"))
            message(" Close clusters by running: `parallel_stop()`.")
        }
    } else {
        if (control$verbose) {
            message(stringr::str_glue(" Finishing sequential backend. | {round(t-t1, 3)} seconds"))
        }
    }

    if (control$verbose) {
        t <- Sys.time()
        message(stringr::str_glue(" Total time | {round(t-t1, 3)} seconds"))
    }


    return(models)

}



# HELPERS -----

as_modeltime_table_from_workflowset <- function(.l) {

    ret <- tibble::tibble(
        .model = .l
    ) %>%
        tibble::rowid_to_column(var = ".model_id")

    # CHECKS
    # validate_model_classes(ret, accept_classes = c("model_fit", "workflow", "mdl_time_ensemble"))
    # validate_models_are_trained(ret)
    validate_models_are_not_null(ret, msg_main = "Some models failed during fitting: modeltime_fit_workflowset()")

    # CREATE MODELTIME OBJECT
    ret <- ret %>%
        dplyr::mutate(.model_desc = purrr::map_chr(.model, .f = get_model_description))

    class(ret) <- c("mdl_time_tbl", class(ret))

    return(ret)
}
