% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/3_distances_disc.R
\name{hellinger.disc}
\alias{hellinger.disc}
\alias{hellinger.boot.disc}
\title{Estimation of a Discrete Mixture Complexity Based on Hellinger Distance}
\usage{
hellinger.disc(obj, j.max = 10, threshold = "SBC", control = c(trace = 0))
                                 
hellinger.boot.disc(obj, j.max = 10, B = 100, ql = 0.025, qu = 0.975, 
                    control = c(trace = 0), ...)
}
\arguments{
\item{obj}{object of class \code{\link{datMix}}.}

\item{j.max}{integer, stating the maximal number of components to be considered.}

\item{threshold}{function or character string in \code{c("AIC", "SBC")} specifying which threshold should be used to compare two mixture estimates of complexities \eqn{j} and \eqn{j+1}. If the difference in minimized squared distances is smaller than the relevant threshold, \eqn{j} will be returned as complexity estimate.}

\item{control}{control list of optimization parameters, see \code{\link[Rsolnp]{solnp}}.}

\item{B}{integer, specifying the number of bootstrap replicates.}

\item{ql}{numeric between \eqn{0} and \eqn{1} specifying the lower quantile to which the observed difference in minimized squared distances will be compared.}

\item{qu}{numeric between \eqn{0} and \eqn{1} specifying the upper quantile to which the observed difference in minimized squared distances will be compared.}

\item{\dots}{further arguments passed to the \code{\link[boot]{boot}} function.}
}
\value{
Object of class \code{paramEst} with the following attributes:
     \item{dat}{data based on which the complexity is estimated.}
     \item{dist}{character string stating the (abbreviated) name of the component distribution, such that the function \code{ddist} evaluates its density/ mass function and \code{rdist} generates random variates.}
     \item{ndistparams}{integer specifying the number of parameters identifying the component distribution, i.e. if \eqn{\theta} is in {R^d} then \code{ndistparams}\eqn{ = d}.}
     \item{formals.dist}{string vector specifying the names of the formal arguments identifying the distribution \code{dist} and used in \code{ddist} and \code{rdist}, e.g. for a gaussian mixture (\code{dist = norm}) amounts to \code{mean} and \code{sd}, as these are the formal arguments used by \code{dnorm} and \code{rnorm}.}
     \item{discrete}{logical flag indicating whether the underlying mixture distribution is discrete. Will always be \code{TRUE} in this case.}
     \item{mle.fct}{attribute \code{MLE.function} of \code{obj}.}
     \item{pars}{say the complexity estimate is equal to some \eqn{j}. Then \code{pars} is a numeric vector of size \eqn{(d+1)*j-1} specifying the component weight and parameter estimates, given as \deqn{(w_1, ... w_{j-1}, \theta 1_1, ... \theta 1_j, \theta 2_1, ... \theta d_j).}}
     \item{values}{numeric vector of function values gone through during optimization at iteration \eqn{j}, the last entry being the value at the optimum.}
     \item{convergence}{integer indicating whether the solver has converged (0) or not (1 or 2) at iteration \eqn{j}.}
}
\description{
Estimation of a discrete mixture complexity as well as its component weights and parameters by minimizing the squared Hellinger distance to the empirical probability mass function.
}
\details{
Define the \eqn{complexity} of a finite discrete mixture \eqn{F} as the smallest integer \eqn{p}, such that its probability mass function (pmf) \eqn{f} can be written as
\deqn{f(x) = w_1*g(x;\theta_1) + \dots + w_p*g(x;\theta_p).}
Let \eqn{g, f} be two probability mass functions. The squared Hellinger distance between \eqn{g} and \eqn{f} is given by
\deqn{H^2(g,f) = \sum (\sqrt{g(x)} - \sqrt{f(x)})^2,}
where \eqn{\sqrt{g(x)}} and \eqn{\sqrt{f(x)}} denote the square roots of the respective probability mass functions at point x.
To estimate \eqn{p}, \code{hellinger.disc} iteratively increases the assumed complexity \eqn{j} and finds the "best" estimate for the pmf of a mixture with \eqn{j} and the pmf of a mixture with \eqn{j+1} components, by calculating the parameters that minimize the sum of squared Hellinger distances to the empirical probability mass function at given points. Once these parameters have been obtained, the difference in squared distances is compared to a predefined \code{threshold}. If this difference is smaller than the threshold, the algorithm terminates and the true complexity is estimated as \eqn{j}, otherwise \eqn{j} is increased by 1. The predefined thresholds are the \code{"AIC"} given by 
\deqn{(d+1)/n} 
and the \code{"SBC"} given by 
\deqn{(d+1)log(n)/(2n),} 
\eqn{n} being the sample size and \eqn{d} the number of component parameters, i.e. \eqn{\theta} is in \eqn{R^d}. Note that, if a customized function is to be used, it may only take the arguments \code{j} and \code{n}, so if the user wants to include the number of component parameters \eqn{d}, it has to be entered explicitly.
\code{hellinger.boot.disc} works similarly to \code{hellinger.disc} with the exception that the difference in squared distances is compared to a value generated via a bootstrap procedure instead of being compared to a predefined threshold. At every iteration (of \eqn{j}), the function sequentially tests \eqn{p = j} versus \eqn{p = j+1} for \eqn{j = 1,2, \dots}, using a parametric bootstrap to generate \code{B} samples of size \eqn{n} from a \eqn{j}-component mixture given the previously calculated "best" parameter values. For each of the bootstrap samples, again the "best" estimates corresponding to pmfs with \eqn{j} and \eqn{j+1} components are computed, as well as their difference in squared Hellinger distances from the empirical probability mass function. The null hypothesis \eqn{H_0: p = j} is rejected and \eqn{j} increased by 1 if the original difference in squared distances lies outside of the interval \eqn{[ql, qu]}, specified by \code{ql} and \code{qu}, empirical quantiles of the bootstrapped differences. Otherwise, \eqn{j} is returned as the complexity estimate.
To calculate the minimum of the Hellinger distance (and the corresponding parameter values), the solver \code{\link[Rsolnp]{solnp}} is used. The initial values supplied to the solver are calculated as follows: the data is clustered into \eqn{j} groups by the function \code{\link[cluster]{clara}} and the data corresponding to each group is given to \code{MLE.function} (if supplied to the \code{\link{datMix}} object \code{obj}, otherwise numerical optimization is used here as well). The size of the groups is taken as initial component weights and the MLE's are taken as initial component parameter estimates.
}
\examples{

## create 'Mix' object
poisMix <- Mix("pois", , discrete = TRUE, w = c(0.45, 0.45, 0.1), lambda = c(1, 5, 10))

## create random data based on 'Mix' object (gives back 'rMix' object)
set.seed(0)
poisRMix <- rMix(1000, obj = poisMix)

## create 'datMix' object for estimation

# generate list of parameter bounds
poisList <- list("lambda" = c(0,Inf))

# generate MLE function
MLE.pois <- function(dat){
  mean(dat)
}

# generating 'datMix' object
pois.dM <- RtoDat(poisRMix, theta.bound.list = poisList, MLE.function = MLE.pois)

## complexity and parameter estimation
set.seed(0)
res <- hellinger.disc(pois.dM)
plot(res)

}
\references{
M.-J. Woo and T. Sriram, "Robust estimation of mixture complexity for count data", Computational Statistics and Data Analysis 51, 4379-4392, 2007.
}
\seealso{
\code{\link{L2.disc}} for the same estimation method using the L2 distance, \code{\link{hellinger.cont}} for the same estimation method for continuous mixtures, \code{\link[Rsolnp]{solnp}} for the solver, \code{\link{datMix}} for the creation of the \code{datMix} object.
}
\keyword{cluster}
