% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evaluate_imputation_parameters.R
\name{evaluate_imputation_parameters}
\alias{evaluate_imputation_parameters}
\title{Evaluate estimated parameters after imputation}
\usage{
evaluate_imputation_parameters(
  ds_imp,
  ds_orig = NULL,
  pars_true = NULL,
  parameter = "mean",
  criterion = "RMSE",
  cols_which = seq_len(ncol(ds_imp)),
  tolerance = sqrt(.Machine$double.eps),
  ...,
  imp_ds,
  true_pars,
  which_cols
)
}
\arguments{
\item{ds_imp}{A data frame or matrix with imputed values.}

\item{ds_orig}{A data frame or matrix with original (true) values.}

\item{pars_true}{True parameters, normally a vector or a matrix.}

\item{parameter}{A string specifying the estimated parameters for comparison.}

\item{criterion}{A string specifying the used criterion for comparing the
imputed and original values.}

\item{cols_which}{Indices or names of columns used for evaluation.}

\item{tolerance}{Numeric, only used for \code{criterion = "precision"}:
numeric differences smaller than tolerance are treated as zero/equal.}

\item{...}{Further arguments passed to the function for parameter estimation.}

\item{imp_ds}{Deprecated, renamed to \code{ds_imp}.}

\item{true_pars}{Deprecated, renamed to \code{pars_true}.}

\item{which_cols}{Deprecated, renamed to \code{cols_which}.}
}
\value{
A numeric vector of length one.
}
\description{
Compares estimated parameters after imputation to true parameters or
estimates based on the original dataset
}
\details{
Either \code{ds_orig} or \code{pars_true} must be supplied and the
other one must be \code{NULL} (default: both are \code{NULL}, just supply
one, see examples). The following \code{parameter}s are implemented:
"mean", "median", "var", "sd", "quantile", "cov", "cov_only", cor", "cor_only".
Some details follow:
\itemize{
\item{"var", "cov" and "cov_only": For "var" only the variances of the
columns (the diagonal elements of the covariance matrix) are compared. For
"cov" the whole covariance matrix is compared. For "cov_only" only the upper
triangle (excluding the diagonal) of the covariance matrix is compared.}
\item{"cor", "cor_only": For "cor" the whole correlation matrix is compared.
For "cor_only" only the upper triangle (excluding the diagonal) of the
correlation matrix is compared.}
\item{"quantile": the quantiles can be set via the additional
argument \code{probs} (see examples). Otherwise, the default quantiles from
\code{\link[stats]{quantile}} will be used.}
}

The argument \code{cols_which} allows the selection of columns for comparison
(see examples). If \code{pars_true} is used, it is assumed that only relevant
parameters are supplied (see examples).

Possible choices for the argument \code{criterion} are documented in
\code{\link{evaluate_imputed_values}}
}
\examples{
# only ds_orig known
ds_orig <- data.frame(X = 1:10, Y = 101:101)
ds_imp <- impute_mean(delete_MCAR(ds_orig, 0.4))
evaluate_imputation_parameters(ds_imp, ds_orig = ds_orig)

# true parameters known
ds_orig <- data.frame(X = rnorm(100), Y = rnorm(100, mean = 10))
ds_imp <- impute_mean(delete_MCAR(ds_orig, 0.3))
evaluate_imputation_parameters(ds_imp, pars_true = c(0, 10), parameter = "mean")
evaluate_imputation_parameters(ds_imp, pars_true = c(1, 1), parameter = "var")

# set quantiles
evaluate_imputation_parameters(ds_imp,
  pars_true = c(qnorm(0.3), qnorm(0.3, mean = 10)),
  parameter = "quantile", probs = 0.3
)

# compare only column Y
evaluate_imputation_parameters(ds_imp,
  pars_true = c(Y = 10), parameter = "mean",
  cols_which = "Y"
)
}
\references{
Cetin-Berber, D. D., Sari, H. I., & Huggins-Manley, A. C. (2019).
  Imputation Methods to Deal With Missing Responses in Computerized Adaptive
  Multistage Testing. \emph{Educational and psychological measurement},
  79(3), 495-511.
}
\seealso{
Other evaluation functions: 
\code{\link{evaluate_imputed_values}()},
\code{\link{evaluate_parameters}()}
}
\concept{evaluation functions}
