% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/misvm.R
\name{misvm}
\alias{misvm}
\alias{misvm.default}
\alias{misvm.formula}
\alias{misvm.mi_df}
\alias{misvm.mild_df}
\title{Fit MI-SVM model to the data}
\usage{
\method{misvm}{default}(
  x,
  y,
  bags,
  cost = 1,
  method = c("heuristic", "mip", "qp-heuristic"),
  weights = TRUE,
  control = list(kernel = "linear", sigma = if (is.vector(x)) 1 else 1/ncol(x),
    nystrom_args = list(m = nrow(x), r = nrow(x), sampling = "random"), max_step = 500,
    type = "C-classification", scale = TRUE, verbose = FALSE, time_limit = 60, start =
    FALSE),
  ...
)

\method{misvm}{formula}(formula, data, ...)

\method{misvm}{mi_df}(x, ...)

\method{misvm}{mild_df}(x, .fns = list(mean = mean, sd = stats::sd), cor = FALSE, ...)
}
\arguments{
\item{x}{A data.frame, matrix, or similar object of covariates, where each
row represents an instance. If a \code{mi_df} object is passed, \verb{y, bags} are
automatically extracted, and all other columns will be used as predictors.
If a \code{mild_df} object is passed, \verb{y, bags, instances} are automatically
extracted, and all other columns will be used as predictors.}

\item{y}{A numeric, character, or factor vector of bag labels for each
instance.  Must satisfy \code{length(y) == nrow(x)}. Suggest that one of the
levels is 1, '1', or TRUE, which becomes the positive class; otherwise, a
positive class is chosen and a message will be supplied.}

\item{bags}{A vector specifying which instance belongs to each bag.  Can be a
string, numeric, of factor.}

\item{cost}{The cost parameter in SVM. If \code{method = 'heuristic'}, this will
be fed to \code{kernlab::ksvm()}, otherwise it is similarly in internal
functions.}

\item{method}{The algorithm to use in fitting (default  \code{'heuristic'}).  When
\code{method = 'heuristic'}, which employs an algorithm similar to Andrews et
al. (2003). When \code{method = 'mip'}, the novel MIP method will be used.  When
\verb{method = 'qp-heuristic}, the heuristic algorithm is computed using the
dual SVM.  See details.}

\item{weights}{named vector, or \code{TRUE}, to control the weight of the cost
parameter for each possible y value.  Weights multiply against the cost
vector. If \code{TRUE}, weights are calculated based on inverse counts of
instances with given label, where we only count one positive instance per
bag. Otherwise, names must match the levels of \code{y}.}

\item{control}{list of additional parameters passed to the method that
control computation with the following components:
\itemize{
\item \code{kernel} either a character the describes the kernel ('linear' or
'radial') or a kernel matrix at the instance level.
\item \code{sigma} argument needed for radial basis kernel.
\item \code{nystrom_args} a list of parameters to pass to \code{\link[=kfm_nystrom]{kfm_nystrom()}}. This is
used when \code{method = 'mip'} and \code{kernel = 'radial'} to generate a Nystrom
approximation of the kernel features.
\item \code{max_step} argument used when \code{method = 'heuristic'}. Maximum steps of
iteration for the heuristic algorithm.
\item \code{type}: argument used when \code{method = 'heuristic'}. The \code{type} argument is
passed to \code{e1071::svm()}.
\item \code{scale} argument used for all methods. A logical for whether to rescale
the input before fitting.
\item \code{verbose} argument used when \code{method = 'mip'}. Whether to message output
to the console.
\item \code{time_limit} argument used when \code{method = 'mip'}. \code{FALSE}, or a time
limit (in seconds) passed to \code{gurobi()} parameters.  If \code{FALSE}, no time
limit is given.
\item \code{start} argument used when \code{method = 'mip'}.  If \code{TRUE}, the mip program
will be warm_started with the solution from \code{method = 'qp-heuristic'} to
potentially improve speed.
}}

\item{...}{Arguments passed to or from other methods.}

\item{formula}{a formula with specification \code{mi(y, bags) ~ x} which uses the
\code{mi} function to create the bag-instance structure. This argument is an
alternative to the \verb{x, y, bags} arguments, but requires the \code{data}
argument. See examples.}

\item{data}{If \code{formula} is provided, a data.frame or similar from which
formula elements will be extracted.}

\item{.fns}{(argument for \code{misvm.mild_df()} method) list of functions to
summarize instances over.}

\item{cor}{(argument for \code{misvm.mild_df()} method) logical, whether to
include correlations between all features in the summarization.}
}
\value{
An object of class \code{misvm.}  The object contains at least the
following components:
\itemize{
\item \verb{*_fit}: A fit object depending on the \code{method} parameter.  If \code{method =   'heuristic'}, this will be an \code{svm} fit from the e1071 package.  If
\verb{method = 'mip', 'qp-heuristic'} this will be \code{gurobi_fit} from a model
optimization.
\item \code{call_type}: A character indicating which method \code{misvm()} was called
with.
\item \code{features}: The names of features used in training.
\item \code{levels}: The levels of \code{y} that are recorded for future prediction.
\item \code{cost}: The cost parameter from function inputs.
\item \code{weights}: The calculated weights on the \code{cost} parameter.
\item \code{repr_inst}: The instances from positive bags that are selected to be
most representative of the positive instances.
\item \code{n_step}: If \code{method \%in\% c('heuristic', 'qp-heuristic')}, the total
steps used in the heuristic algorithm.
\item \code{x_scale}: If \code{scale = TRUE}, the scaling parameters for new predictions.
}
}
\description{
This function fits the MI-SVM model, first proposed by Andrews et al. (2003).
It is a variation on the traditional SVM framework that carefully treats data
from the multiple instance learning paradigm, where instances are grouped
into bags, and a label is only available for each bag.
}
\details{
Several choices of fitting algorithm are available, including a version of
the heuristic algorithm proposed by Andrews et al. (2003) and a novel
algorithm that explicitly solves the mixed-integer programming (MIP) problem
using the gurobi package optimization back-end.
}
\section{Methods (by class)}{
\itemize{
\item \code{default}: Method for data.frame-like objects

\item \code{formula}: Method for passing formula

\item \code{mi_df}: Method for \code{mi_df} objects, automatically handling bag
names, labels, and all covariates.

\item \code{mild_df}: Method for \code{mild_df} objects. Summarize samples to the
instance level based on specified functions, then perform \code{misvm()} on
instance level data.
}}

\examples{
set.seed(8)
mil_data <- generate_mild_df(nbag = 20,
                             positive_prob = 0.15,
                             sd_of_mean = rep(0.1, 3))
df <- build_instance_feature(mil_data, seq(0.05, 0.95, length.out = 10))

# Heuristic method
mdl1 <- misvm(x = df[, 4:123], y = df$bag_label,
              bags = df$bag_name, method = "heuristic")
mdl2 <- misvm(mi(bag_label, bag_name) ~ X1_mean + X2_mean + X3_mean, data = df)

# MIP method
if (require(gurobi)) {
  mdl3 <- misvm(x = df[, 4:123], y = df$bag_label,
                bags = df$bag_name, method = "mip")
}

predict(mdl1, new_data = df, type = "raw", layer = "bag")

# summarize predictions at the bag layer
library(dplyr)
df \%>\%
  bind_cols(predict(mdl2, df, type = "class")) \%>\%
  bind_cols(predict(mdl2, df, type = "raw")) \%>\%
  distinct(bag_name, bag_label, .pred_class, .pred)


}
\references{
Andrews, S., Tsochantaridis, I., & Hofmann, T. (2002). Support
vector machines for multiple-instance learning. \emph{Advances in neural
information processing systems}, \emph{15}.

Kent, S., & Yu, M. (2022). Non-convex SVM for cancer diagnosis based on
morphologic features of tumor microenvironment \emph{arXiv preprint}
\href{https://arxiv.org/abs/2206.14704}{arXiv:2206.14704}
}
\seealso{
\itemize{
\item \code{\link[=predict.misvm]{predict.misvm()}} for prediction on new data.
\item \code{\link[=cv_misvm]{cv_misvm()}} for cross-validation fitting.
}
}
\author{
Sean Kent, Yifei Liu
}
