% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/es_from_stand_OR.R
\name{es_from_or_se}
\alias{es_from_or_se}
\title{Convert an odds ratio value and its standard error into several effect size measures}
\usage{
es_from_or_se(
  or,
  logor,
  logor_se,
  baseline_risk,
  small_margin_prop,
  n_exp,
  n_nexp,
  n_cases,
  n_controls,
  n_sample,
  or_to_rr = "metaumbrella_cases",
  or_to_cor = "pearson",
  reverse_or
)
}
\arguments{
\item{or}{odds ratio value}

\item{logor}{log odds ratio value}

\item{logor_se}{the standard error of the log odds ratio}

\item{baseline_risk}{proportion of cases in the non-exposed group}

\item{small_margin_prop}{smallest margin proportion of cases/events in the underlying 2x2 table}

\item{n_exp}{number of participants in the exposed group}

\item{n_nexp}{number of participants in the non-exposed group}

\item{n_cases}{number of cases/events across exposed/non-exposed groups}

\item{n_controls}{number of controls/no-event across exposed/non-exposed groups}

\item{n_sample}{total number of participants in the sample}

\item{or_to_rr}{formula used to convert the \code{or} value into a risk ratio (see details).}

\item{or_to_cor}{formula used to convert the \code{or} value into a correlation coefficient (see details).}

\item{reverse_or}{a logical value indicating whether the direction of the generated effect sizes should be flipped.}
}
\value{
This function estimates and converts between several effect size measures.

\tabular{ll}{
\code{natural effect size measure} \tab OR\cr
\tab \cr
\code{converted effect size measure} \tab RR + NNT\cr
\code{} \tab D + G + R + Z\cr
\tab \cr
\code{required input data} \tab See 'Section 2. Odds Ratio'\cr
\tab https://metaconvert.org/input.html\cr
\tab \cr
}
}
\description{
Convert an odds ratio value and its standard error into several effect size measures
}
\details{
This function converts the log odds ratio into a Risk ratio (RR), Cohen's d (D), Hedges' g (G)
and correlation coefficients (R/Z).

\strong{To estimate the Cohen's d value and its standard error}
The following formulas are used (Cooper et al., 2019):
\deqn{d = \log(or) * \frac{\sqrt{3}}{\pi}}
\deqn{d\_se = \sqrt{\frac{logor\_se^2 * 3}{\pi^2}}}

\strong{To estimate the risk ratio and its standard error, various formulas can be used.}

\strong{A.} First, the approach described in Grant (2014) can be used.
However, in the paper, only the formula to convert an OR value to a RR value
is described.
To derive the variance, we used this formula to convert the bounds of the 95\% CI, which
were then used to obtain the variance.

This argument requires (or + baseline_risk + or_ci_lo + or_ci_up) to generate a RR.
The following formulas are used (br = baseline_risk):
\deqn{rr = \frac{or}{1 - br + br*or}}
\deqn{rr\_ci\_lo = \frac{or\_ci\_lo}{1 - br + br*or\_ci\_lo}}
\deqn{rr\_ci\_up = \frac{or\_ci\_up}{1 - br + br*or\_ci\_up}}
\deqn{logrr\_se = \frac{log(rr\_ci\_up) - log(rr\_ci\_lo)}{2 * qnorm(.975)}}

\strong{B.} Second, the formulas implemented in the metaumbrella package can be used
(\code{or_to_rr = "metaumbrella_cases"} or \code{or_to_rr = "metaumbrella_exp"}).
This argument requires (or + logor_se + n_cases + n_controls) or (or + logor_se + n_exp + n_nexp)
to generate a RR.
More precisely, when the OR value and its standard error, plus either
(i) the number of cases and controls or
(ii) the number of participants in the exposed and non-exposed groups,
are available, we previously developed functions that simulate all combinations of the possible
number of cases and controls
in the exposed and non-exposed groups compatible with the actual value of the OR.
Then, the functions select the contingency table whose standard error coincides best with
the standard error reported.
The RR value and its standard are obtained from this estimated contingency table.

\strong{C.} Third, it is possible to transpose the RR to a OR (\code{or_to_rr = "transpose"}).
This argument requires (or + logor_se) to generate a OR.
It is known that OR and RR are similar when the baseline risk is small.
Therefore, users can request to simply transpose the OR value & standard error into a RR value & standard error.
\deqn{rr = or}
\deqn{logrr\_se = logor\_se}

\strong{D.} Fourth, it is possible to recreate the 2x2 table using the dipietrantonj's formulas (\code{or_to_rr = "dipietrantonj"}).
This argument requires (or + logor_ci_lo + logor_ci_lo) to generate a RR. Information on this approach can be retrieved in
Di Pietrantonj (2006).

\strong{To estimate the NNT, the formulas used are :}
\deqn{\frac{(1 - br * (1 - or))}{(1 - br) * (br * (1 - or))}}

\strong{To estimate a correlation coefficient, various formulas can be used.}

\strong{A.} First, the approach described in Pearson (1900) can be used (\code{or_to_cor = "pearson"}).
This argument requires (or + logor_se) to generate a R/Z.
It converts the OR value and its standard error to a tetrachoric correlation.
Note that the formula assumes that each cell of the 2x2 used to estimate the OR has been added 1/2 before estimating the OR value and its standard error.
If it is not the case, formulas can produce slightly less accurate results.

\deqn{c = \frac{1}{2}}
\deqn{r = \cos{\frac{\pi}{1+or^c}}}
\deqn{r\_se = logor\_se * ((\pi * c * or^c) * \frac{\sin(\pi / (1+or^c))}{1+or^c})^2}
\deqn{or\_ci\_lo = exp(log(or) - qnorm(.975)*logor\_se)}
\deqn{or\_ci\_up = exp(log(or) + qnorm(.975)*logor\_se)}
\deqn{r\_ci\_lo = cos(\frac{\pi}{1 + or\_ci\_lo^c})}
\deqn{r\_ci\_up = cos(\frac{\pi}{1 + or\_ci\_up^c})}
\deqn{z = atanh(r)}
\deqn{z\_se = \sqrt{\frac{r\_se^2}{(1 - r^2)^2}}}
\deqn{z\_ci\_lo = atanh(r\_lo)}
\deqn{z\_ci\_up = atanh(r\_up)}

\strong{B.} Second, the approach described in Digby (1983) can be used (\code{or_to_cor = "digby"}).
This argument requires (or + logor_se) to generate a R/Z.
It converts the OR value and its standard error to a tetrachoric correlation.
Note that the formula assumes that each cell of the 2x2 used to estimate the OR has been added 1/2 before estimating the OR value and its standard error.
If it is not the case, formulas can produce slightly less accurate results.

\deqn{c = \frac{3}{4}}
\deqn{r = \frac{or^c - 1}{or^c + 1}}
\deqn{r\_se = \sqrt{\frac{c^2}{4} * (1 - r^2)^2 * logor\_se}}
\deqn{z = atanh(r)}
\deqn{z\_se = \sqrt{\frac{r\_se^2}{(1 - r^2)^2}}}
\deqn{z\_ci\_lo = z - qnorm(.975)*\sqrt{\frac{c^2}{4} * logor\_se}}
\deqn{z\_ci\_up = z + qnorm(.975)*\sqrt{\frac{c^2}{4} * logor\_se}}
\deqn{r\_ci\_lo = tanh(z\_lo)}
\deqn{r\_ci\_up = tanh(z\_up)}

\strong{C.} Third, the approach described in Bonett (2005) can be used (\code{or_to_cor = "bonett"}).
This argument requires (or + logor_se + n_cases + n_exp + small_margin_prop) to generate a R/Z.
Note that the formula assumes that each cell of the 2x2 used to estimate the OR has been added 1/2 before estimating the OR value and its standard error.
If it is not the case, formulas can produce slightly less accurate results.

\deqn{c = \frac{\frac{1 - |n\_exp - n\_cases|}{5} - (0.5 - small\_margin\_prop)^2}{2}}
\deqn{r = \cos{\frac{\pi}{1+or^c}}}
\deqn{r\_se = logor\_se * ((\pi * c * or^c) * \frac{\sin(\frac{\pi}{1+or^c})}{1+or^c})^2}
\deqn{or\_ci\_lo = exp(log(or) - qnorm(.975)*logor\_se)}
\deqn{or\_ci\_up = exp(log(or) + qnorm(.975)*logor\_se)}
\deqn{r\_ci\_lo = cos(\frac{\pi}{1 + or\_ci\_lo^c})}
\deqn{r\_ci\_up = cos(\frac{\pi}{1 + or\_ci\_up^c})}
\deqn{z = atanh(r)}
\deqn{z\_se = \sqrt{\frac{r\_se^2}{(1 - r^2)^2}} }
\deqn{z\_ci\_lo = atanh(r\_lo)}
\deqn{z\_ci\_up = atanh(r\_up)}

\strong{D.} Last, the approach described in Cooper et al. (2019) can be used (\code{or_to_cor = "lipsey_cooper"}).
This argument requires (or + logor_se + n_exp + n_nexp) to generate a R/Z.
As shown above, the function starts to estimate a SMD from the OR.
Then, as described in \code{\link{es_from_cohen_d}}, it converts this Cohen's d value into a correlation
coefficient using the \code{"lipsey_cooper"} formulas.
}
\examples{
es_from_or_se(or = 2.12, logor_se = 0.242, n_exp = 120, n_nexp = 44)
}
\references{
Bonett, Douglas G. and Robert M. Price. (2005). Inferential Methods for the Tetrachoric Correlation Coefficient. Journal of Educational and Behavioral Statistics 30:213-25.

Bonett, D. G., & Price, R. M. (2007). Statistical inference for generalized Yule coefficients in 2× 2 contingency tables. Sociological methods & research, 35(3), 429-446.

Cooper, H., Hedges, L. V., & Valentine, J. C. (Eds.). (2019). The handbook of research synthesis and meta-analysis. Russell Sage Foundation.

Di Pietrantonj C. (2006). Four-fold table cell frequencies imputation in meta analysis. Statistics in medicine, 25(13), 2299–2322. https://doi.org/10.1002/sim.2287

Digby, Peter G. N. (1983). Approximating the Tetrachoric Correlation Coefficient. Biometrics 39:753-7.

Gosling, C. J., Solanes, A., Fusar-Poli, P., & Radua, J. (2023). metaumbrella: the first comprehensive suite to perform data analysis in umbrella reviews with stratification of the evidence. BMJ mental health, 26(1), e300534. https://doi.org/10.1136/bmjment-2022-300534

Grant R. L. (2014). Converting an odds ratio to a range of plausible relative risks for better communication of research findings. BMJ (Clinical research ed.), 348, f7450. https://doi.org/10.1136/bmj.f7450

Pearson, K. (1900). Mathematical Contributions to the Theory of Evolution. VII: On the Correlation of Characters Not Quantitatively Measurable. Philosophical Transactions of the Royal Statistical Society of London, Series A 19:1-47

Veroniki, A. A., Pavlides, M., Patsopoulos, N. A., & Salanti, G. (2013). Reconstructing 2x2 contingency tables from odds ratios using the Di Pietrantonj method: difficulties, constraints and impact in meta-analysis results. Research synthesis methods, 4(1), 78–94. https://doi.org/10.1002/jrsm.1061
}
