% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_lagr.R
\name{fit_lagr.mcgf}
\alias{fit_lagr.mcgf}
\title{Parameter estimation for Lagrangian correlation functions for an \code{mcgf}
object.}
\usage{
\method{fit_lagr}{mcgf}(
  x,
  model = c("lagr_tri", "lagr_askey", "none"),
  method = c("wls", "mle"),
  optim_fn = c("nlminb", "optim", "other"),
  par_fixed = NULL,
  par_init,
  lower = NULL,
  upper = NULL,
  other_optim_fn = NULL,
  dists_base = FALSE,
  dists_lagr = NULL,
  ...
)
}
\arguments{
\item{x}{An \code{mcgf} object containing attributes \code{dists}, \code{acfs}, \code{ccfs}, and
\code{sds}. \code{x} must have been passed to \code{add_base()} or \verb{base<-}}

\item{model}{Base model, one of \code{lagr_tri}, \code{lagr_askey}, or \code{none}.
If \code{none}, NULLs are returned.}

\item{method}{Parameter estimation methods, weighted least square (\code{wls}) or
maximum likelihood estimation (\code{mle}).}

\item{optim_fn}{Optimization functions, one of \code{nlminb}, \code{optim}, \code{other}.
When \code{optim_fn = other}, supply \code{other_optim_fn}.}

\item{par_fixed}{Fixed parameters.}

\item{par_init}{Initial values for parameters to be optimized.}

\item{lower}{Optional; lower bounds of parameters lambda, v1, v2, and k.}

\item{upper}{Optional: upper bounds of parameters lambda, v1, v2, and k.}

\item{other_optim_fn}{Optional, other optimization functions. The first two
arguments must be initial values for the parameters and a function to be
minimized respectively (same as that of \code{optim} and \code{nlminb}).}

\item{dists_base}{Logical; if TRUE \code{dists_base} from the base model is used
as the distance.}

\item{dists_lagr}{List of distance matrices/arrays. Used when \code{dists_base} is
FALSE. If NULL, \code{dists(x)} is used.}

\item{...}{Additional arguments passed to \code{optim_fn}.}
}
\value{
A list containing outputs from optimization functions of \code{optim_fn}.
}
\description{
Parameter estimation for Lagrangian correlation functions for an \code{mcgf}
object.
}
\details{
This function fits the Lagrangian models using weighted least squares and
maximum likelihood estimation. The base model must be fitted first using
\code{add_base()} or \verb{base<-}. Optimization functions such as \code{nlminb} and \code{optim}
are supported. Other functions are also supported by setting
\code{optim_fn = "other"} and supplying \code{other_optim_fn}. \code{lower} and \code{upper} are
lower and upper bounds of parameters in \code{par_init} and default bounds are
used if they are not specified.

Note that both \code{wls} and \code{mle} are heuristic approaches when \code{x} contains
observations from a subset of the discrete spatial domain, though estimation
results are close to that using the full spatial domain for large sample
sizes.

Since parameters for the base model and the Lagrangian model are estimated
sequentially, more accurate estimation may be obtained if the full model is
fitted all at once.
}
\examples{
data(sim1)
sim1_mcgf <- mcgf(sim1$data, dists = sim1$dists)
sim1_mcgf <- add_acfs(sim1_mcgf, lag_max = 5)
sim1_mcgf <- add_ccfs(sim1_mcgf, lag_max = 5)

# Fit a separable model and store it to 'sim1_mcgf'
fit_sep <- fit_base(
    sim1_mcgf,
    model = "sep",
    lag = 5,
    par_init = c(
        c = 0.001,
        gamma = 0.5,
        a = 0.3,
        alpha = 0.5
    ),
    par_fixed = c(nugget = 0)
)
sim1_mcgf <- add_base(sim1_mcgf, fit_base = fit_sep)

# Fit a Lagrangian model
fit_lagr <- fit_lagr(
    sim1_mcgf,
    model = "lagr_tri",
    par_init = c(v1 = 300, v2 = 300, lambda = 0.15),
    par_fixed = c(k = 2)
)
fit_lagr$fit
}
\seealso{
Other functions on fitting an mcgf: 
\code{\link{add_base.mcgf}()},
\code{\link{add_lagr.mcgf}()},
\code{\link{fit_base.mcgf}()},
\code{\link{krige.mcgf}()}
}
\concept{functions on fitting an mcgf}
