% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/construct.R
\name{matrixset}
\alias{matrixset}
\title{Matrix Set}
\usage{
matrixset(
  ...,
  expand = NULL,
  row_info = NULL,
  column_info = NULL,
  row_key = "rowname",
  column_key = "colname",
  row_tag = ".rowname",
  column_tag = ".colname"
)
}
\arguments{
\item{...}{A single list of matrices (must be a named list), or
individual matrices, e.g. \code{mat1 = m1}, \code{mat2 = m2}, etc.
\code{NULL} elements are accepted. This allows to create a
placeholder that can be filled later on.}

\item{expand}{By default (\code{NULL}), input matrix expansion is disabled.
Setting this parameter to \code{TRUE} will enable the expansion
feature. See the section \sQuote{Matrix Expansion} for more
details of what it is, as well as other possible options
for \code{expand}.}

\item{row_info}{a data frame, used to annotate matrix rows. The link
between the matrix row names and the data frame is given
in column "rowname". A different column can be used if one
provides a different \code{row_key}.}

\item{column_info}{a data frame, used to annotate matrix columns. The link
between the matrix column names and the data frame is given
in column "colname". A different column can be used if one
provides a different \code{column_key}.}

\item{row_key}{column name in `row_info`` data frame that will
link the row names with the row information. A string is
expected.}

\item{column_key}{column name in \code{col_info} data frame that will
link the column names with the row information. A string is
expected.}

\item{row_tag}{A string, giving the row annotation data frame column that
will link the row names to the data frame. While
\code{row_key} specifies the column name of the data frame
at input, \code{row_tag} specifies the column name that
will be used throughout in the \code{matrixset} object.}

\item{column_tag}{A string, giving the column annotation data frame column
that will link the row names to the data frame. While
\code{column_key} specifies the column name of the data
frame at input, \code{column_tag} specifies the column
name that will be used throughout in the \code{matrixset}
object.}
}
\value{
Returns a \code{matrixset}, a collection of matrices (see \sQuote{Details}).
}
\description{
Creates a matrix set, possibly annotated for rows and/or columns. These
annotations are referred as traits.
}
\details{
A \code{matrixset} is a collection of matrices that share the same dimensions and,
if applicable, dimnames. It is designed to hold different measures for the
same rows/columns. For example, each matrix could be a different time point
for the same subjects.

Traits, which are annotations, can be provided in the form of data frames
for rows and/or columns. If traits are provided, the \code{data.frame} must
contain only one entry per row/column (see examples).

Row or column names are not mandatory to create a proper \code{matrixset}. The
only way for this to work however is to leave traits (annotations) empty.
If provided, each matrices must have the same dimnames as well.

If dimnames are missing, note that most of the operations for matrixsets
won't be available. For instance, operations that use traits will not work,
e.g., \code{\link[=filter_row]{filter_row()}}.

It is allowed for matrix elements of a \code{matrixset} to be \code{NULL} - see
examples.
}
\section{Matrix Expansion}{

The concept of matrix expansion allows to provide input matrices that do not
share the same dimensions.

This works by taking the union of the dimnames and padding, if necessary,
each matrix with a special value for the missing rows/columns.

Because the dimnames are used, they must necessarily be non-\code{NULL} in the
provided matrices.

An interesting side-effect is that one can use this option to match the
dimnames and provide a common row/column order among the matrices.

The padding special value is, by default (\code{expand = TRUE}), \code{NA}. It can be
changed by providing any value (e.g, \code{-1}) to \code{expand}, in which case the
same padding value is used for all matrices.

If different padding values are needed for each matrices, a list can be
provided to \code{expand}. If the list is unnamed, it must match the number of
input matrices in length and the padding values are assigned to the matrices
in order.

A named list can be provided as well. In that case, \code{expand} names and
matrix names are matched. All matrices must have a match in the \code{expand} list
(more \code{expand} values can be provided, though).
}

\examples{
# A single NULL element will create an empty matrixset (it doesn't hold
# any matrices)
lst <- NULL
matrixset(lst)

# This will hold to empty matrices
lst <- list(a = NULL, b = NULL)
matrixset(lst)
# this is equivalent
matrixset(a = NULL, b = NULL)

# A basic example
lst <- list(a = matrix(0, 2, 3))
matrixset(lst)
# equivalent
matrixset(a = matrix(0, 2, 3))

# can mix with NULL too
lst <- list(a = NULL, b = matrix(0, 2, 3), c = matrix(0, 2, 3))
matset <- matrixset(lst)

# dimnames are also considered to be traits
lst <- list(a = NULL, b = matrix(0, 2, 3), c = matrix(0, 2, 3))
rownames(lst$b) <- c("r1", "r2")
rownames(lst$c) <- c("r1", "r2")
matrixset(lst)

# You don't have to annotate both rows and columns. But you need to provide
# the appropriate dimnames when you provide traits
lst <- list(a = matrix(0, 2, 3), b = matrix(0, 2, 3), c = NULL)
rownames(lst$a) <- c("r1", "r2")
rownames(lst$b) <- c("r1", "r2")
colnames(lst$a) <- c("c1", "c2", "c3")
colnames(lst$b) <- c("c1", "c2", "c3")
ri <- data.frame(rowname = c("r1", "r2"), g = 1:2)
matset <- matrixset(lst, row_info = ri)

# You can provide a column name that contains the keys
ri <- data.frame(foo = c("r1", "r2"), g = 1:2)
matset <- matrixset(lst, row_info = ri, row_key = "foo")

lst <- list(a = matrix(0, 2, 3), b = matrix(0, 2, 3), c = NULL)
rownames(lst$a) <- c("r1", "r2")
rownames(lst$b) <- c("r1", "r2")
colnames(lst$a) <- c("c1", "c2", "c3")
colnames(lst$b) <- c("c1", "c2", "c3")
ri <- data.frame(rowname = c("r1", "r2"), g = 1:2)
ci <- data.frame(colname = c("c1", "c2", "c3"), h = 1:3)
matset <- matrixset(lst, row_info = ri, column_info = ci)

# This is not allowed, because the row trait data frame has more than one
# entry for "r1"
lst <- list(a = matrix(0, 2, 3), b = matrix(0, 2, 3), c = NULL)
rownames(lst$a) <- c("r1", "r2")
rownames(lst$b) <- c("r1", "r2")
colnames(lst$a) <- c("c1", "c2", "c3")
colnames(lst$b) <- c("c1", "c2", "c3")
ri <- data.frame(rowname = c("r1", "r2", "r1"), g = 1:3)
ci <- data.frame(colname = c("c1", "c2", "c3"), h = 1:3)
ans <- tryCatch(matrixset(lst, row_info = ri, column_info = ci),
                error = function(e) e)
is(ans, "error")

}
\seealso{
\code{\link[=as_matrixset]{as_matrixset()}}
}
