% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/distance_corr.R
\name{distance_corr}
\alias{distance_corr}
\alias{print.distance_corr}
\alias{plot.distance_corr}
\title{Pairwise Distance Correlation (dCor)}
\usage{
distance_corr(data)

\method{print}{distance_corr}(x, digits = 4, max_rows = NULL, max_cols = NULL, ...)

\method{plot}{distance_corr}(
  x,
  title = "Distance correlation heatmap",
  low_color = "white",
  high_color = "steelblue1",
  value_text_size = 4,
  ...
)
}
\arguments{
\item{data}{A numeric matrix or a data frame with at least two numeric
columns. All non-numeric columns are dropped. Columns must be numeric
and contain no \code{NA}s.}

\item{x}{An object of class \code{distance_corr}.}

\item{digits}{Integer; number of decimal places to print.}

\item{max_rows}{Optional integer; maximum number of rows to display.
If \code{NULL}, all rows are shown.}

\item{max_cols}{Optional integer; maximum number of columns to display.
If \code{NULL}, all columns are shown.}

\item{...}{Additional arguments passed to \code{ggplot2::theme()} or other
\code{ggplot2} layers.}

\item{title}{Plot title. Default is \code{"Distance correlation heatmap"}.}

\item{low_color}{Colour for zero correlation. Default is \code{"white"}.}

\item{high_color}{Colour for strong correlation. Default is \code{"steelblue1"}.}

\item{value_text_size}{Font size for displaying values. Default is \code{4}.}
}
\value{
A symmetric numeric matrix where the \code{(i, j)} entry is the
unbiased distance correlation between the \code{i}-th and \code{j}-th
numeric columns. The object has class \code{distance_corr} with attributes
\code{method = "distance_correlation"}, \code{description}, and
\code{package = "matrixCorr"}.

Invisibly returns \code{x}.

A \code{ggplot} object representing the heatmap.
}
\description{
Computes all pairwise \emph{distance correlations} using the unbiased
U-statistic estimator for the numeric columns of a matrix or data frame,
via a high-performance 'C++' backend ('OpenMP'-parallelised).
Distance correlation detects general (including non-linear and non-monotonic)
dependence between variables; unlike Pearson or Spearman, it is zero
(in population) if and only if the variables are independent.

Prints a summary of the distance correlation matrix with
optional truncation for large objects.

Generates a ggplot2 heatmap of the distance correlation matrix.
Distance correlation is non-negative; the fill scale spans \code{[0, 1]}.
}
\details{
Let \eqn{x \in \mathbb{R}^n} and \eqn{D^{(x)}} be the pairwise distance matrix
with zero diagonal: \eqn{D^{(x)}_{ii} = 0}, \eqn{D^{(x)}_{ij} = |x_i - x_j|} for
\eqn{i \neq j}. Define row sums \eqn{r^{(x)}_i = \sum_{k \neq i} D^{(x)}_{ik}} and
grand sum \eqn{S^{(x)} = \sum_{i \neq k} D^{(x)}_{ik}}. The U-centred matrix is
\deqn{A^{(x)}_{ij} =
      \begin{cases}
        D^{(x)}_{ij} - \dfrac{r^{(x)}_i + r^{(x)}_j}{n - 2}
        + \dfrac{S^{(x)}}{(n - 1)(n - 2)}, & i \neq j,\\[6pt]
        0, & i = j~.
      \end{cases}}
For two variables \eqn{x,y}, the unbiased distance covariance and variances are
\deqn{\widehat{\mathrm{dCov}}^2_u(x,y) = \frac{2}{n(n-3)} \sum_{i<j} A^{(x)}_{ij} A^{(y)}_{ij}
\;=\; \frac{1}{n(n-3)} \sum_{i \neq j} A^{(x)}_{ij} A^{(y)}_{ij},}
with \eqn{\widehat{\mathrm{dVar}}^2_u(x)} defined analogously from \eqn{A^{(x)}}.
The unbiased distance correlation is
\deqn{\widehat{\mathrm{dCor}}_u(x,y) =
      \frac{\widehat{\mathrm{dCov}}_u(x,y)}
           {\sqrt{\widehat{\mathrm{dVar}}_u(x)\,\widehat{\mathrm{dVar}}_u(y)}} \in [0,1].}
}
\note{
Requires \eqn{n \ge 4}. Columns with (near) zero unbiased distance
variance yield \code{NA} in their row/column. Computation is \eqn{O(n^2)} per
pair.
}
\examples{
##Independent variables -> dCor ~ 0
set.seed(1)
X <- cbind(a = rnorm(200), b = rnorm(200))
D <- distance_corr(X)
print(D, digits = 3)

## Non-linear dependence: Pearson ~ 0, but unbiased dCor > 0
set.seed(42)
n <- 200
x <- rnorm(n)
y <- x^2 + rnorm(n, sd = 0.2)
XY <- cbind(x = x, y = y)
D2 <- distance_corr(XY)
# Compare Pearson vs unbiased distance correlation
round(c(pearson = cor(XY)[1, 2], dcor = D2["x", "y"]), 3)
plot(D2, title = "Unbiased distance correlation (non-linear example)")

## Small AR(1) multivariate normal example
set.seed(7)
p <- 5; n <- 150; rho <- 0.6
Sigma <- rho^abs(outer(seq_len(p), seq_len(p), "-"))
X3 <- MASS::mvrnorm(n, mu = rep(0, p), Sigma = Sigma)
colnames(X3) <- paste0("V", seq_len(p))
D3 <- distance_corr(X3)
print(D3[1:3, 1:3], digits = 2)

}
\references{
Székely, G. J., Rizzo, M. L., & Bakirov, N. K. (2007).
Measuring and testing dependence by correlation of distances.
\emph{Annals of Statistics}, 35(6), 2769–2794.

Székely, G. J., & Rizzo, M. L. (2013).
The distance correlation t-test of independence.
\emph{Journal of Multivariate Analysis}, 117, 193-213.
}
\author{
Thiago de paula Oliveira
}
