% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/measure_centrality.R
\name{measure_central_close}
\alias{measure_central_close}
\alias{node_closeness}
\alias{node_harmonic}
\alias{node_reach}
\alias{node_information}
\alias{node_eccentricity}
\alias{node_distance}
\alias{node_vitality}
\alias{tie_closeness}
\alias{net_closeness}
\alias{net_reach}
\alias{net_harmonic}
\title{Measures of closeness-like centrality and centralisation}
\usage{
node_closeness(.data, normalized = TRUE, direction = "out", cutoff = NULL)

node_harmonic(.data, normalized = TRUE, cutoff = -1)

node_reach(.data, normalized = TRUE, cutoff = 2)

node_information(.data, normalized = TRUE)

node_eccentricity(.data, normalized = TRUE)

node_distance(.data, from, to, normalized = TRUE)

node_vitality(.data, normalized = TRUE)

tie_closeness(.data, normalized = TRUE)

net_closeness(.data, normalized = TRUE, direction = c("all", "out", "in"))

net_reach(.data, normalized = TRUE, cutoff = 2)

net_harmonic(.data, normalized = TRUE, cutoff = 2)
}
\arguments{
\item{.data}{An object of a manynet-consistent class:
\itemize{
\item matrix (adjacency or incidence) from \code{{base}} R
\item edgelist, a data frame from \code{{base}} R or tibble from \code{{tibble}}
\item igraph, from the \code{{igraph}} package
\item network, from the \code{{network}} package
\item tbl_graph, from the \code{{tidygraph}} package
}}

\item{normalized}{Logical scalar, whether the centrality scores are normalized.
Different denominators are used depending on whether the object is one-mode or two-mode,
the type of centrality, and other arguments.}

\item{direction}{Character string, “out” bases the measure on outgoing ties,
“in” on incoming ties, and "all" on either/the sum of the two.
For two-mode networks, "all" uses as numerator the sum of differences
between the maximum centrality score for the mode
against all other centrality scores in the network,
whereas "in" uses as numerator the sum of differences
between the maximum centrality score for the mode
against only the centrality scores of the other nodes in that mode.}

\item{cutoff}{Maximum path length to use during calculations.}

\item{from, to}{Index or name of a node to calculate distances from or to.}
}
\description{
These functions calculate common closeness-related centrality measures
that rely on path-length for one- and two-mode networks:
\itemize{
\item \code{node_closeness()} measures the closeness centrality of nodes in a
network.
\item \code{node_reach()} measures nodes' reach centrality,
or how many nodes they can reach within \emph{k} steps.
\item \code{node_harmonic()} measures nodes' harmonic centrality or valued
centrality, which is thought to behave better than reach centrality
for disconnected networks.
\item \code{node_information()} measures nodes' information centrality or
current-flow closeness centrality.
\item \code{node_eccentricity()} measures nodes' eccentricity or maximum distance
from another node in the network.
\item \code{node_distance()} measures nodes' geodesic distance from or to a
given node.
\item \code{tie_closeness()} measures the closeness of each tie to other ties
in the network.
\item \code{net_closeness()} measures a network's closeness centralization.
\item \code{net_reach()} measures a network's reach centralization.
\item \code{net_harmonic()} measures a network's harmonic centralization.
}

All measures attempt to use as much information as they are offered,
including whether the networks are directed, weighted, or multimodal.
If this would produce unintended results,
first transform the salient properties using e.g. \code{\link[=to_undirected]{to_undirected()}} functions.
All centrality and centralization measures return normalized measures by default,
including for two-mode networks.
}
\section{Closeness centrality}{

Closeness centrality or status centrality is defined as the reciprocal of
the farness or distance, \eqn{d},
from a node to all other nodes in the network:
\deqn{C_C(i) = \frac{1}{\sum_j d(i,j)}}
When (more commonly) normalised, the numerator is instead \eqn{N-1}.
}

\section{Harmonic centrality}{

Harmonic centrality or valued centrality reverses the sum and reciprocal
operations compared to closeness centrality:
\deqn{C_H(i) = \sum_{i, i \neq j} \frac{1}{d(i,j)}}
where \eqn{\frac{1}{d(i,j)} = 0} where there is no path between \eqn{i} and
\eqn{j}. Normalization is by \eqn{N-1}.
Since the harmonic mean performs better than the arithmetic mean on
unconnected networks, i.e. networks with infinite distances,
harmonic centrality is to be preferred in these cases.
}

\section{Reach centrality}{

In some cases, longer path lengths are irrelevant and 'closeness' should
be defined as how many others are in a local neighbourhood.
How many steps out this neighbourhood should be defined as is given by
the 'cutoff' parameter.
This is usually termed \eqn{k} or \eqn{m} in equations,
which is why this is sometimes called (\eqn{m}- or)
\eqn{k}-step reach centrality:
\deqn{C_R(i) = \sum_j d(i,j) \leq k}
The maximum reach score is \eqn{N-1}, achieved when the node can reach all
other nodes in the network in \eqn{k} steps or less,
but the normalised version, \eqn{\frac{C_R}{N-1}}, is more common.
Note that if \eqn{k = 1} (i.e. cutoff = 1), then this returns the node's degree.
At higher cutoff reach centrality returns the size of the node's component.
}

\section{Information centrality}{

Information centrality, also known as current-flow centrality,
is a hybrid measure relating to both path-length and walk-based measures.
The information centrality of a node is the harmonic average of the
“bandwidth” or inverse path-length for all paths originating from the node.

As described in the \code{{sna}} package,
information centrality works on an undirected but potentially weighted
network excluding isolates (which take scores of zero).
It is defined as:
\deqn{C_I = \frac{1}{T + \frac{\sum T - 2 \sum C_1}{|N|}}}
where \eqn{C = B^-1} with \eqn{B} is a pseudo-adjacency matrix replacing
the diagonal of \eqn{1-A} with \eqn{1+k},
and \eqn{T} is the trace of \eqn{C} and \eqn{S_R} an arbitrary row sum
(all rows in \eqn{C} have the same sum).

Nodes with higher information centrality have a large number of short paths
to many others in the network, and are thus considered to have greater
control of the flow of information.
}

\section{Eccentricity centrality}{

Eccentricity centrality, graph centrality, or the Koenig number,
is the (if normalized, inverse of) the distance to the furthest node:
\deqn{C_E(i) = \frac{1}{max_{j \in N} d(i,j)}}
where the distance from \eqn{i} to \eqn{j} is \eqn{\infty} if unconnected.
As such it is only well defined for connected networks.
}

\section{Closeness vitality centrality}{

The closeness vitality of a node is the change in the sum of all distances
in a network, also known as the Wiener Index, when that node is removed.
Note that the closeness vitality may be negative infinity if
removing that node would disconnect the network.
}

\examples{
node_closeness(ison_southern_women)
node_reach(ison_adolescents)
(ec <- tie_closeness(ison_adolescents))
ison_adolescents \%>\% mutate_ties(weight = ec)
net_closeness(ison_southern_women, direction = "in")
}
\references{
\subsection{On closeness centrality}{

Bavelas, Alex. 1950.
"Communication Patterns in Task‐Oriented Groups".
\emph{The Journal of the Acoustical Society of America}, 22(6): 725–730.
\doi{10.1121/1.1906679}

Harary, Frank. 1959.
"Status and Contrastatus".
\emph{Sociometry}, 22(1): 23–43.
\doi{10.2307/2785610}
}

\subsection{On harmonic centrality}{

Marchiori, Massimo, and Vito Latora. 2000.
"Harmony in the small-world".
\emph{Physica A} 285: 539-546.
\doi{10.1016/S0378-4371(00)00311-3}

Dekker, Anthony. 2005.
"Conceptual distance in social network analysis".
\emph{Journal of Social Structure} 6(3).
}

\subsection{On reach centrality}{

Borgatti, Stephen P., Martin G. Everett, and J.C. Johnson. 2013.
\emph{Analyzing social networks}.
London: SAGE Publications Limited.
}

\subsection{On information centrality}{

Stephenson, Karen, and Marvin Zelen. 1989.
"Rethinking centrality: Methods and examples".
\emph{Social Networks} 11(1):1-37.
\doi{10.1016/0378-8733(89)90016-6}

Brandes, Ulrik, and Daniel Fleischer. 2005.
"Centrality Measures Based on Current Flow".
\emph{Proc. 22nd Symp. Theoretical Aspects of Computer Science} LNCS 3404: 533-544.
\doi{10.1007/978-3-540-31856-9_44}
}

\subsection{On eccentricity centrality}{

Hage, Per, and Frank Harary. 1995.
"Eccentricity and centrality in networks".
\emph{Social Networks}, 17(1): 57-63.
\doi{10.1016/0378-8733(94)00248-9}
}

Koschuetzki, Dirk, Katharina Lehmann, Leon Peeters, Stefan Richter,
Dagmar Tenfelde-Podehl, and Oliver Zlotowski. 2005.
"Centrality Indices", in
Brandes, Ulrik, and Thomas Erlebach (eds.).
\emph{Network Analysis: Methodological Foundations}.
Springer: Berlin, pp. 16-61.
}
\seealso{
Other centrality: 
\code{\link{measure_central_between}},
\code{\link{measure_central_degree}},
\code{\link{measure_central_eigen}}

Other measures: 
\code{\link{measure_attributes}},
\code{\link{measure_central_between}},
\code{\link{measure_central_degree}},
\code{\link{measure_central_eigen}},
\code{\link{measure_closure}},
\code{\link{measure_cohesion}},
\code{\link{measure_diffusion_infection}},
\code{\link{measure_diffusion_net}},
\code{\link{measure_diffusion_node}},
\code{\link{measure_features}},
\code{\link{measure_heterogeneity}},
\code{\link{measure_hierarchy}},
\code{\link{measure_holes}},
\code{\link{measure_periods}},
\code{\link{measure_properties}},
\code{\link{member_diffusion}}
}
\concept{centrality}
\concept{measures}
