#' Print and plot methods
#'
#' @export
#'
#' @param x A list with class \code{'loo'} (as returned by the \code{\link{loo}}
#'   function or, for \code{print} only, the \code{\link{waic}} function). For
#'   \code{print}, \code{x} can also have class \code{'compare.loo'} (as
#'   returned by \code{\link{compare}}).
#' @param ... Ignored for \code{print}. For \code{plot} you can use \code{...}
#'   to pass arguments to \code{\link[graphics]{text}} if \code{label_points =
#'   TRUE}.
#' @param digits Passed to \code{\link[base]{round}}.
#' @param warn Logical. If \code{TRUE} (the default), a warning message will be
#'   printed if any estimates for the Pareto shape parameter \eqn{k} are
#'   problematic. See section PSIS-LOO in \code{\link{loo-package}}. Ignored if
#'   \code{x} was generated by \code{\link{waic}}.
#' @param plot_k Logical. If \code{TRUE} the estimates of \eqn{k} are plotted.
#'   Ignored if \code{x} was generated by \code{\link{waic}}. To just plot
#'   \eqn{k} without printing use \code{plot(x)}.
#' @param label_points Logical. If \code{TRUE} the observation numbers
#'   corresponding to any values of \eqn{k} greater than 0.5 will be displayed
#'   in the plot. Any arguments specified in \code{...} will be passed to
#'   \code{\link[graphics]{text}} and can be used to control the appearance of
#'   the labels.
#'
#' @return The print methods return \code{x} invisibly.
#'
print.loo <- function(x, ..., digits = 1, warn = TRUE, plot_k = FALSE) {
  lldims <- paste(attr(x, "log_lik_dim"), collapse = " by ")
  cat("Computed from", lldims, "log-likelihood matrix\n\n")
  z <- x[-grep("pointwise|pareto_k", names(x))]
  uz <- unlist(z)
  nms <- names(uz)
  ses <- grepl("se", nms)
  out <- data.frame(Estimate = uz[!ses], SE = uz[ses])
  out <- format(round(out, digits), nsmall = digits)
  print(out, quote = FALSE)
  if ("pareto_k" %in% names(x)) {
    if (warn)
      k_warnings(x$pareto_k, digits)
    if (plot_k)
      plot(x)
  }
  invisible(x)
}

#' @rdname print.loo
#' @export
print.compare.loo <- function(x, ..., digits = 1) {
  print(format(round(x, digits), nsmall = digits), quote = FALSE)
}

#' @rdname print.loo
#' @export
#'
plot.loo <- function(x, ..., label_points = FALSE) {
  if (!("pareto_k" %in% names(x)))
    stop("No Pareto k values found.", call. = FALSE)
  k <- x$pareto_k
  if (any(!is.finite(k))) {
    warning(signif(100 * mean(!is.finite(k)), 2),
            "% of k estimates are Inf/NA/NaN and not plotted.")
  }
  plot_k(k, ..., label_points = label_points)
}
