\name{logConDens}
\alias{logConDens}
\title{Compute log-concave density estimator and related quantities}
\description{Compute the log-concave and smoothed log-concave density
  estimator.

  This function is as it is in the logcondens package except we've added
  the 'prec' variable as an argument, and modified the values returned
  as output, to be in line with the \code{activeSetLogCon.mode}
  function.

}
\usage{logConDens(x, xgrid = NULL, smoothed = TRUE, print = FALSE, 
    gam = NULL, xs = NULL, prec=10^-10)}

\arguments{
  \item{x}{Vector of independent and identically distributed numbers, not necessarily unique.}
  \item{xgrid}{Governs the generation of weights for observations. See \code{\link{preProcess}} for details.}
  \item{smoothed}{If \code{TRUE}, the smoothed version of the log-concave density estimator is also computed.}
  \item{print}{\code{print = TRUE} outputs the log-likelihood in every loop, \code{print = FALSE} does not. Make sure to tell \code{R} to output (press CTRL+W).}
  \item{gam}{Only necessary if \code{smoothed = TRUE}. The standard deviation of the normal kernel. If equal to \code{NULL}, \code{gam} is chosen
             such that the variances of the original sample \eqn{x_1, \ldots, x_n} and \eqn{\widehat f_n^*}{\hat f_n^*} coincide.}
  \item{xs}{Only necessary if \code{smoothed = TRUE}. Either provide a vector of support points where the smoothed estimator should be computed at, or
  leave as \code{NULL}. Then, a sufficiently width equidistant grid of
  points will be used.}
  \item{prec}{Governs precision of various subfunctions,
    e.g. the Newton-Raphson procedure.}
}

\details{See \code{\link{activeSetLogCon}} for details on the
  computations.}


\value{ \code{\link{logConDens}} returns an object of class
\code{"dlc"}, a list containing the following components: \code{xn},
\code{x}, \code{w}, \code{L}, \code{IsKnot}, \code{knots}, \code{phi},
\code{fhat},
\code{Fhat}, \code{H}, \code{n}, \code{m}, \code{mode}, \code{dlcMode},
 \code{sig}, \code{phi.f}, \code{fhat.f}, \code{Fhat.f}, \code{E.f},
 \code{phiPL},
 \code{phiPR}, \code{phiPL.f}, and \code{phiPR.f},
 as generated by \code{\link{activeSetLogCon}}. If
\code{smoothed = TRUE}, then the returned object additionally contains
\code{f.smoothed}, \code{F.smoothed}, \code{gam}, and \code{xs} as
generated by \code{\link{evaluateLogConDens}}. Finally, the entry
\code{smoothed} of type \code{"logical"} returnes the value of
\code{smoothed}.

The methods \code{\link{summary.dlc}} and \code{\link{plot.dlc}} are
used to obtain a summary and generate plots of the estimated density.  }

\references{
Duembgen, L, Huesler, A. and Rufibach, K. (2010).
Active set and EM algorithms for log-concave densities based on complete and censored data. 
Technical report 61, IMSV, Univ. of Bern, available at \url{http://arxiv.org/abs/0707.4643}.

Duembgen, L. and Rufibach, K. (2009).
Maximum likelihood estimation of a log--concave density and its distribution function: basic properties and uniform consistency. 
\emph{Bernoulli}, \bold{15(1)}, 40--68.  

Duembgen, L. and Rufibach, K. (2011).
logcondens: Computations Related to Univariate Log-Concave Density Estimation. 
\emph{Journal of Statistical Software}, \bold{39(6)}, 1--28. \url{http://www.jstatsoft.org/v39/i06}
}
\author{
Kaspar Rufibach, \email{kaspar.rufibach@gmail.com}, \cr \url{http://www.kasparrufibach.ch} 

Lutz Duembgen, \email{duembgen@stat.unibe.ch}, \cr \url{http://www.staff.unibe.ch/duembgen}
}
\examples{
## ===================================================
## Illustrate on simulated data
## ===================================================

## Set parameters
n <- 50
x <- rnorm(n)

res <- logConDens(x, smoothed = TRUE, print = FALSE, gam = NULL, 
    xs = NULL)
summary(res)
plot(res, which = "density", legend.pos = "topright")
plot(res, which = "log-density")
plot(res, which = "CDF")

## Compute slopes and intercepts of the linear functions that 
## compose phi
slopes <- diff(res$phi) / diff(res$x)
intercepts <- -slopes * res$x[-n] + res$phi[-n]


## ===================================================
## Illustrate method on reliability data
## Reproduce Fig. 2 in Duembgen & Rufibach (2009)
## ===================================================

## Set parameters
data(reliability)
x <- reliability
n <- length(x)
res <- logConDens(x, smooth = TRUE, print = TRUE)
phi <- res$phi
f <- exp(phi)

## smoothed log-concave PDF
f.smoothed <- res$f.smoothed
xs <- res$xs

## compute kernel density
sig <- sd(x)
h <- sig / sqrt(n)
f.kernel <- rep(NA, length(xs))
for (i in 1:length(xs)){
    xi <- xs[i]
    f.kernel[i] <- mean(dnorm(xi, mean = x, sd = h))
}

## compute normal density
mu <- mean(x)
f.normal <- dnorm(xs, mean = mu, sd = sig)

## ===================================================
## Plot resulting densities, i.e. reproduce Fig. 2
## in Duembgen and Rufibach (2009)
## ===================================================
plot(0, 0, type = 'n', xlim = range(xs), ylim = c(0, 6.5 * 10^-3))
rug(res$x)
lines(res$x, f, col = 2)
lines(xs, f.normal, col = 3)
lines(xs, f.kernel, col = 4)
lines(xs, f.smoothed, lwd = 3, col = 5)
legend("topleft", c("log-concave", "normal", "kernel", 
    "log-concave smoothed"), lty = 1, col = 2:5, bty = "n")


## ===================================================
## Plot log-densities
## ===================================================
plot(0, 0, type = 'n', xlim = range(xs), ylim = c(-20, -5))
legend("bottomright", c("log-concave", "normal", "kernel", 
    "log-concave smoothed"), lty = 1, col = 2:5, bty = "n")
rug(res$x)
lines(res$x, phi, col = 2)
lines(xs, log(f.normal), col = 3)
lines(xs, log(f.kernel), col = 4)
lines(xs, log(f.smoothed), lwd = 3, col = 5)


## ===================================================
## Confidence intervals at a fixed point for the density
## see help file for logConCI()
## ===================================================
}
\keyword{htest}
\keyword{nonparametric}
