% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coregistration.R
\name{coregistration}
\alias{coregistration}
\title{Tree inventory and canopy height model coregistration}
\usage{
coregistration(
  chm,
  trees,
  mask = NULL,
  buffer = 19,
  step = 0.5,
  dm = 2,
  plot = TRUE
)
}
\arguments{
\item{chm}{raster. canopy height model}

\item{trees}{data.frame. the first two columns contain xy coordinates, and 
the third is the value to correlate to the chm (e.g. tree heights or diameters)}

\item{mask}{raster. raster mask of tree inventory area}

\item{buffer}{numeric. radius of the circular buffer area of possible translations}

\item{step}{numeric. increment step of translations within buffer area to 
compute correlation values, should be a multiple of raster resolution}

\item{dm}{numeric. minimum distance between two local maxima in meters}

\item{plot}{boolean. whether to display the results or not}
}
\value{
A list with two elements : first the correlation raster returned by 
\code{\link{rasters_moving_cor}}, second a data.frame returned by 
\code{\link{raster_local_max}}
}
\description{
Computes the correlation between the canopy height model and a virtual canopy 
height model simulated from tree locations, for different translations of tree 
inventory positions, and outputs the translation corresponding to best 
estimated co-registration.
}
\examples{
# tree inventory
trees <- data.frame(x = c(22.2, 18.3, 18.1), y = c(22.1, 22.7, 18.4), 
z = c(15, 10, 15))

# mask of inventory area
# empty raster with extent
tree_mask <- circle2Raster(20, 20, 9, resolution = 1)
# fill binary mask
tree_mask <- raster_xy_mask(rbind(c(20, 20), c(20, 20)), c(9, 9), tree_mask, 
binary = TRUE)

# simulate chm raster
chm <- raster::raster()
raster::extent(chm) <- c(0, 40, 0, 40)
raster::res(chm) <- 1
xy <- raster::xyFromCell(chm, 1:length(chm))

# add Gaussian surfaces to simulate tree crowns
z1 <- trees$z[1] * exp(-((xy[, 1] - trees$x[1])^2 + (xy[, 2] - trees$y[1])^2 / 2) * trees$z[1] / 50)
z2 <- trees$z[2] * exp(-((xy[, 1] - trees$x[2])^2 + (xy[, 2] - trees$y[2])^2 / 2) * trees$z[2] / 50)
z3 <- trees$z[3] * exp(-((xy[, 1] - trees$x[3])^2 + (xy[, 2] - trees$y[3])^2 / 2) * trees$z[3] / 50)
chm <- raster::rasterFromXYZ(cbind(xy, pmax(z1, z2, z3))) #+rnorm(length(z1),0,1)))

# translate trees
trees$x <- trees$x + 1
trees$y <- trees$y + 2

coreg <- coregistration(chm, trees, mask = tree_mask, buffer = 5, step = 1, dm = 1, plot = FALSE)
coreg$local_max[, c("dx1", "dy1")]

# plot raster
raster::plot(coreg$correlation_raster)
abline(h = 0, lty = 2)
abline(v = 0, lty = 2)
# add location of two local maxima
graphics::points(coreg$local_max[1, c("dx1", "dx2")],
  coreg$local_max[1, c("dy1", "dy2")],
  cex = c(1, 0.5), pch = 3, col = "red"
)
}
\references{
Monnet, J.-M. and Mermin, E. 2014. Cross-Correlation of Diameter 
Measures for the Co-Registration of Forest Inventory Plots with Airborne Laser 
Scanning Data. Forests 2014, 5(9), 2307-2326, \doi{10.3390/f5092307}
}
\seealso{
\code{\link{rasters_moving_cor}}, \code{\link{raster_local_max}}
}
