% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Logger.R
\name{Logger}
\alias{Logger}
\alias{Loggers}
\alias{LoggerGlue}
\title{Loggers}
\description{
A Logger produces a \link{LogEvent} that contains the log message along with
metadata (timestamp, calling function) and dispatches it to one or several
\link{Appenders} which are responsible for the output (console, file, ...) of the
event. \strong{lgr} comes with a single pre-configured Logger called the
\code{root Logger} that can be accessed via \code{lgr$<...>}. Instantiation of new
Loggers is only necessary if you want to take advantage of hierarchical
logging as outlined in \code{vignette("lgr", package = "lgr")}.
}
\section{Usage}{
\preformatted{# Cannonical way to initialize a new Logger (see "Creating Loggers")
lg <- get_logger("logger")
}

\preformatted{# R6 constructor (not recommended for productive use)
lg <- Logger$new(name = "(unnamed logger)", appenders = list(), threshold =
  NULL, filters = list(), exception_handler = default_exception_handler,
  propagate = TRUE)

lg$log(level, msg, ..., timestamp = Sys.time(), caller = get_caller(-7))
lg$fatal(msg, ..., caller = get_caller(-8L))
lg$error(msg, ..., caller = get_caller(-8L))
lg$warn(msg, ..., caller = get_caller(-8L))
lg$info(msg, ..., caller = get_caller(-8L))
lg$debug(msg, ..., caller = get_caller(-8L))
lg$trace(msg, ..., caller = get_caller(-8L))
lg$config(cfg = NULL, file = NULL, text = NULL)
lg$add_appender(appender, name = NULL)
lg$remove_appender(pos)
lg$handle_exception(...)
lg$set_name(x)
lg$set_exception_handler(fun)
lg$set_propagate(x)
lg$set_threshold(level)
lg$set_appenders(x)
lg$spawn(name, ...)
lg$filter(event)
lg$add_filter(filter, name = NULL)
lg$remove_filter(pos)
lg$set_filters(filters)

lg$ancestry
lg$appenders
lg$exception_handler
lg$inherited_appenders
lg$last_event
lg$name
lg$parent
lg$propagate
lg$threshold
lg$filters

}
}

\section{Creating Loggers}{


If you are a package developer you should define a new Logger for each
package, but you do not need to configure it. Usually only the root logger
needs to be configured (new Appenders added/removed, Layouts modified,
etc...).

Loggers should never be instantiated directly with \code{Logger$new()} but rather
via \code{get_logger("name")}. If \code{"name"} does not exist, a new
Logger with that name will be created, otherwise the function returns a
Reference to the existing Logger.

The \code{name} is potentially a \code{/} separated hierarchical value like
\code{foo/bar/baz}. Loggers further down the hierarchy are children of the loggers
above.

All calls to \code{get_logger()} with the same name return the same Logger
instance. This means that Logger instances never need to be passed between
different parts of an application.

If you just want to log to an additional output (like a log file), you want
a new \link{Appender}, not a new Logger.
}

\section{Fields}{


You can modify the fields of an existing Logger with
\code{logger$set_<fieldname>(value)} (see examples). Another way to configure
loggers is via its \code{$config()} method.

\describe{
\item{\code{name}, \code{set_name(x)}}{\code{character} scalar. A name for the Logger that should be
unique among Loggers. This logger name is used in the Loggers print
method and can be used by Appenders to indicate which logger the
log message came from. If you define a Logger for an R Package (the most
common case for defining new Loggers), the logger name should be the
same name as the Package name. If you do not define a Logger name
manually, a warning will be thrown.}

\item{\code{appenders}, \code{set_appenders(x)}}{A single \link{Appender} or a \code{list}
thereof. Appenders control the output of a Logger. Be aware that a Logger
also inherits the Appenders of its ancestors
(see \code{vignette("lgr", package = "lgr")} for more info about Logger
inheritance structures).}

\item{\code{threshold}, \code{set_threshold(level)}}{\code{character} or \code{integer} scalar.
The minimum \link[=log_levels]{log level} that triggers this Logger}

\item{\code{exception_handler}, \code{set_exception_handler()}}{a \code{function} that
takes a single argument \code{e}. The function used to handle errors that occur
during logging. Defaults to demoting errors to \link{warnings}.}

\item{\code{propagate}, \code{set_propagate()}}{\code{TRUE} or \code{FALSE}. Should LogEvents
be passed on to the appenders of the ancestral Loggers?}
}



\describe{
\item{\code{filters}, \code{set_filters(filters)}}{a \code{list} that may contain
\code{functions} or any \R object with a \code{filter()} method. These functions
must have exactly one argument: \code{event} which will get passed the
LogEvent when the Filterable's \code{filter()} method is invoked.
If all of these functions evaluate to \code{TRUE} the LogEvent is passed on.
Since LogEvents have reference semantics, filters can also be abused to
modify them before they are passed on. Look at the source code of
\code{\link[=with_log_level]{with_log_level()}} or \code{\link[=with_log_value]{with_log_value()}} for examples.
}
}
}

\section{Read-Only Bindings}{


In addition to the active bindings used to access the fields described above,
Loggers also have the following additional read-only bindings:

\describe{
\item{\code{ancestry}}{A named \code{logical} vector of containing
the propagate value of each Logger upper the inheritance tree. The names
are the names of the appenders. \code{ancestry} is an S3 class with a custom
\code{format()}/\code{print()} method, so if you want to use the plain logical
vector use \code{unclass(lg$ancestry)}}

\item{\code{inherited_appenders}}{A \code{list} of all inherited
appenders from ancestral Loggers of the current Logger}

\item{\code{last_event}}{The last LogEvent produced by the current Logger}
}
}

\section{Methods}{


\describe{
\item{\code{fatal(msg, ..., caller = get_caller(-8L))}}{Logs a message with
level \code{fatal} on this logger. If there are \emph{unnamed} arguments in \code{...},
they will be passed to \code{base::sprintf()} along with message. Named
arguments will be passed as custom fields to \link{LogEvent}. If there are named
arguments the names must be unique. \code{caller} refers to the name of the
calling function and if specified manually must be a \code{character} scalar.}

\item{\code{error(msg, ..., caller = get_caller(-8L))}}{Logs a message with level \code{error} on this logger.
The arguments are interpreted as for \code{fatal()}.}

\item{\code{warn(msg, ..., caller = get_caller(-8L))}}{Logs a message with level \code{warn} on this logger.
The arguments are interpreted as for \code{fatal()}.}

\item{\code{info(msg, ..., caller = get_caller(-8L))}}{Logs a message with level \code{info} on this logger.
The arguments are interpreted as for \code{fatal()}.}

\item{\code{debug(msg, ..., caller = get_caller(-8L))}}{Logs a message with level \code{debug} on this logger.
The arguments are interpreted as for \code{fatal()}.}

\item{\code{trace(msg, ..., caller = get_caller(-8L))}}{Logs a message with level \code{trace} on this logger.
The arguments are interpreted as for \code{fatal()}.}

\item{\code{log(level, msg, ..., timestamp, caller)}}{
If the \code{level} passes the Logger \code{threshold} a new \link{LogEvent}
with \code{level}, \code{msg}, \code{timestamp} and \code{caller} is created. Unnamed
arguments in \code{...} will be combined with \code{msg} via \code{base::sprintf()}.
Named arguments in \code{...} will be passed on to \code{LogEvent$new()} as custom
fields. If no unnamed arguments are present, \code{msg} will \emph{not} be passed
to \code{sprintf()}, so in that case you do not have to escape \code{"\%"}
characters.
If the new LogEvent passes this Loggers filters, it will be dispatched
to the relevant \link{Appenders} and checked against their thresholds and
filters.
}

\item{\code{config(cfg  = NULL, file = NULL, text = NULL}}{Load a Logger
configuration. \code{cfg} can be either a special list object, the path to
a YAML config file, or a character scalar containing YAML. The arguments
\code{file} and \code{text} can used alternatively and enforce that the supplied
argument is of the specified type.
See \link{logger_config} for details.
}

\item{\code{add_appender(appender, name = NULL)}, \code{remove_appender(pos)}}{
Add or remove an \link{Appender}. Supplying a \code{name} is optional but
recommended. After adding an Appender with
\code{logger$add_appender(AppenderConsole$new(), name = "console")} you can
refer to it via \code{logger$appenders$console}. \code{remove_appender()} can
remove an Appender by position or name.
}

\item{\code{spawn(...)}}{Spawn a child Logger.
\code{get_logger("foo/bar")$spawn("baz")} is equivalent to
\code{get_logger("foo/bar/baz")}, but can be convenient for programmatic use
when the name of the parent Logger is not known.}
}


\describe{
\item{\code{filter(event)}}{Determine whether the LogEvent \code{x} should be passed
on to Appenders (\code{TRUE}) or not (\code{FALSE}). See also the active binding
\code{filters}}
\item{\code{add_filter(filter, name = NULL)}, \code{remove_filter(pos)}}{
Add or remove a filter. When adding a filter an optional \code{name} can
be specified. \code{remove_filter()} can remove by position or name (if one
was specified)
}
}
}

\section{LoggerGlue}{


\code{LoggerGlue} uses \code{\link[glue:glue]{glue::glue()}} instead of \code{\link[base:sprintf]{base::sprintf()}} to construct
log messages. \strong{glue} is a very well designed package for
string interpolation. It makes composing log messages
more flexible and comfortable at the price of an additional dependency and
slightly less performance than \code{sprintf()}.

\code{glue()} lets you define temporary named variables inside the call.
As with the normal Logger, these named arguments get turned into custom
fields; however, you can suppress this behaviour by making named argument
start with a \code{"."}. Please refer to \code{vignette("lgr", package = "lgr")} for
examples.
}

\examples{
# lgr::lgr is the root logger that is always available
lgr$info("Today is \%s", Sys.Date() )
lgr$fatal("This is a serious error")

# You can create new loggers with Logger$new(). The following creates a
# logger that logs to a temporary file.
tf <- tempfile()
lg <- Logger$new("mylogger", appenders = AppenderFile$new(tf))

# The new logger passes the log message on to the appenders of its parent
# logger, which is by default the root logger. This is why the following
# writes not only the file 'tf', but also to the console.
lg$fatal("blubb")
readLines(tf)

# This logger's print() method depicts this relationship
lg2 <- Logger$new("lg/child")
print(lg2)
print(lg2$ancestry)
print(lg2$name)

# use formatting strings and custom fields
tf2 <- tempfile()
lg$add_appender(AppenderFile$new(tf2, layout = LayoutJson$new()))
lg$info("Not all \%s support custom fields", "appenders", type = "test")
cat(readLines(tf), sep = "\\n")
cat(readLines(tf2), sep = "\\n")

# The following works because if no unnamed `...` are present, msg is not
# passed through sprintf() (otherwise you would have to escape the "\%")
lg$fatal("100\%")

# LoggerGlue
lg <- LoggerGlue$new("glue")
lg$fatal("blah ", "fizz is set to: {fizz}", foo = "bar", fizz = "buzz")
}
\seealso{
\href{https://glue.tidyverse.org/}{glue}
}
