% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gpOptimizationInterface.R
\name{gpAdaptiveLasso}
\alias{gpAdaptiveLasso}
\title{gpAdaptiveLasso}
\usage{
gpAdaptiveLasso(
  par,
  regularized,
  weights = NULL,
  fn,
  gr = NULL,
  lambdas = NULL,
  nLambdas = NULL,
  reverse = TRUE,
  curve = 1,
  ...,
  method = "glmnet",
  control = lessSEM::controlGlmnet()
)
}
\arguments{
\item{par}{labeled vector with starting values}

\item{regularized}{vector with names of parameters which are to be regularized.}

\item{weights}{labeled vector with adaptive lasso weights. NULL will use 1/abs(par)}

\item{fn}{R function which takes the parameters as input and returns the
fit value (a single value)}

\item{gr}{R function which takes the parameters as input and returns the
gradients of the objective function. If set to NULL, numDeriv will be used
to approximate the gradients}

\item{lambdas}{numeric vector: values for the tuning parameter lambda}

\item{nLambdas}{alternative to lambda: If alpha = 1, lessSEM can automatically
compute the first lambda value which sets all regularized parameters to zero.
It will then generate nLambda values between 0 and the computed lambda.}

\item{reverse}{if set to TRUE and nLambdas is used, lessSEM will start with the
largest lambda and gradually decrease lambda. Otherwise, lessSEM will start with
the smallest lambda and gradually increase it.}

\item{curve}{Allows for unequally spaced lambda steps (e.g., .01,.02,.05,1,5,20).
If curve is close to 1 all lambda values will be equally spaced, if curve is large
lambda values will be more concentrated close to 0. See ?lessSEM::curveLambda for more information.}

\item{...}{additional arguments passed to fn and gr}

\item{method}{which optimizer should be used? Currently implemented are ista
and glmnet.}

\item{control}{used to control the optimizer. This element is generated with
the controlIsta and controlGlmnet functions. See ?controlIsta and ?controlGlmnet
for more details.}
}
\value{
Object of class gpRegularized
}
\description{
Implements adaptive lasso regularization for general purpose optimization problems.
The penalty function is given by:
\deqn{p( x_j) = p( x_j) = \frac{1}{w_j}\lambda| x_j|}
Adaptive lasso regularization will set parameters to zero if \eqn{\lambda} is large enough.
}
\details{
The interface is similar to that of optim. Users have to supply a vector
with starting values (important: This vector \emph{must} have labels) and a fitting
function. This fitting functions \emph{must} take a labeled vector with parameter
values as first argument. The remaining arguments are passed with the ... argument.
This is similar to optim.

The gradient function gr is optional. If set to NULL, the \pkg{numDeriv} package
will be used to approximate the gradients. Supplying a gradient function
can result in considerable speed improvements.

Adaptive lasso regularization:
\itemize{
\item Zou, H. (2006). The adaptive lasso and its oracle properties. Journal of the American Statistical Association,
101(476), 1418–1429. https://doi.org/10.1198/016214506000000735
}

For more details on GLMNET, see:
\itemize{
\item Friedman, J., Hastie, T., & Tibshirani, R. (2010).
Regularization Paths for Generalized Linear Models via Coordinate Descent.
Journal of Statistical Software, 33(1), 1–20. https://doi.org/10.18637/jss.v033.i01
\item Yuan, G.-X., Chang, K.-W., Hsieh, C.-J., & Lin, C.-J. (2010).
A Comparison of Optimization Methods and Software for Large-scale
L1-regularized Linear Classification. Journal of Machine Learning Research, 11, 3183–3234.
\item Yuan, G.-X., Ho, C.-H., & Lin, C.-J. (2012).
An improved GLMNET for l1-regularized logistic regression.
The Journal of Machine Learning Research, 13, 1999–2030. https://doi.org/10.1145/2020408.2020421
}

For more details on ISTA, see:
\itemize{
\item Beck, A., & Teboulle, M. (2009). A Fast Iterative Shrinkage-Thresholding
Algorithm for Linear Inverse Problems. SIAM Journal on Imaging Sciences, 2(1),
183–202. https://doi.org/10.1137/080716542
\item Gong, P., Zhang, C., Lu, Z., Huang, J., & Ye, J. (2013).
A General Iterative Shrinkage and Thresholding Algorithm for Non-convex
Regularized Optimization Problems. Proceedings of the 30th International
Conference on Machine Learning, 28(2)(2), 37–45.
\item Parikh, N., & Boyd, S. (2013). Proximal Algorithms. Foundations and
Trends in Optimization, 1(3), 123–231.
}
}
\examples{
# This example shows how to use the optimizers
# for other objective functions. We will use
# a linear regression as an example. Note that
# this is not a useful application of the optimizers
# as there are specialized packages for linear regression
# (e.g., glmnet)

library(lessSEM)
set.seed(123)

# first, we simulate data for our
# linear regression.
N <- 100 # number of persons
p <- 10 # number of predictors
X <- matrix(rnorm(N*p),	nrow = N, ncol = p) # design matrix
b <- c(rep(1,4), 
       rep(0,6)) # true regression weights
y <- X\%*\%matrix(b,ncol = 1) + rnorm(N,0,.2)

# First, we must construct a fiting function
# which returns a single value. We will use
# the residual sum squared as fitting function.

# Let's start setting up the fitting function:
fittingFunction <- function(par, y, X, N){
  # par is the parameter vector
  # y is the observed dependent variable
  # X is the design matrix
  # N is the sample size
  pred <- X \%*\% matrix(par, ncol = 1) #be explicit here: 
  # we need par to be a column vector
  sse <- sum((y - pred)^2)
  # we scale with .5/N to get the same results as glmnet
  return((.5/N)*sse)
}

# let's define the starting values:
b <- c(solve(t(X)\%*\%X)\%*\%t(X)\%*\%y) # we will use the lm estimates
names(b) <- paste0("b", 1:length(b))
# names of regularized parameters
regularized <- paste0("b",1:p)

# define the weight for each of the parameters
weights <- 1/abs(b)
# we will re-scale the weights for equivalence to glmnet.
# see ?glmnet for more details
weights <- length(b)*weights/sum(weights)

# optimize
adaptiveLassoPen <- gpAdaptiveLasso(
  par = b, 
  regularized = regularized, 
  weights = weights,
  fn = fittingFunction, 
  lambdas = seq(0,1,.01), 
  X = X,
  y = y,
  N = N
)
plot(adaptiveLassoPen)
# You can access the fit results as follows:
adaptiveLassoPen@fits
# Note that we won't compute any fit measures automatically, as
# we cannot be sure how the AIC, BIC, etc are defined for your objective function 

# for comparison:
# library(glmnet)
# coef(glmnet(x = X,
#            y = y,
#            penalty.factor = weights,
#            lambda = adaptiveLassoPen@fits$lambda[20],
#            intercept = FALSE,
#            standardize = FALSE))[,1]
# adaptiveLassoPen@parameters[20,]
}
