% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dbscan.R
\name{hdbscan}
\alias{hdbscan}
\title{hdbscan}
\usage{
hdbscan(edges, minPts = 20, K = 5, neighbors = NULL, threads = NULL,
  verbose = getOption("verbose", TRUE))
}
\arguments{
\item{edges}{An edge matrix of the type returned by \code{\link{buildEdgeMatrix}}.}

\item{minPts}{The minimum number of points in a cluster.}

\item{K}{The number of points in the core neighborhood. (See details.)}

\item{neighbors}{An adjacency matrix of the type returned by \code{\link{randomProjectionTreeSearch}}.}

\item{threads}{The maximum number of threads to spawn. Determined automatically if \code{NULL} (the default).}

\item{verbose}{Verbosity.}
}
\value{
An object of type \code{hdbscan} with the following fields:
\describe{
   \item{'clusters'}{A vector of the cluster membership for each vertex. Outliers
   are given \code{NA}}
   \item{'probabilities'}{A vector of the degree of each vertex' membership. This
   is calculated as each vertex' \eqn{\lambda_p} over the highest \eqn{\lambda_p}
   in the cluster. }
   \item{'tree'}{The minimum spanning tree used to generate the clustering.}
   \item{'hierarchy'}{A representation of the condensed cluster hierarchy.}
   \item{'call'}{The call.}
 }

 The hierarchy describes the complete post-condensation structure of the tree:
 \describe{
 \item{'nodemembership'}{The node ID of the vertex's immediate parent, after condensation.}
 \item{'lambda'}{\eqn{\lambda_p}}
 \item{'parent'}{The node ID of each node's parent.}
 \item{'stability'}{The node's stability, taking into account child-node stabilities.}
 \item{'selected'}{Whether the node was selected.}
 \item{'coredistances'}{The core distance determined for each vertex.}
 }
}
\description{
Implemenation of the hdbscan algorithm.
}
\details{
The hyperparameter \code{K} controls the size of core neighborhoods.
The algorithm measures the density around a point as 1 / the distance between
that point and its \eqn{k}th nearest neighbor. A low value of \code{K} is similar
to clustering nearest neighbors rather than based on density. A high value of
\code{K} may cause the algorithm to miss some (usually contrived) clustering
patterns, such as where clusters are made up of points arranged in lines to form
shapes.

If \code{neighbors} is specified, some costly sorts of neighbors in the edge
matrix may be avoided.

The function must be provided sufficient nearest-neighbor data for whatever
is specified for \eqn{k}. If \eqn{k} = 5, for example, the edge matrix (and
neighbor matrix, if specified) must contain data on at least 5 neighbors for
each point. This should not be problematic in typical use in connection with
\code{\link{largeVis}}, which is ordinarily run with a far higher \eqn{k}-value
than hdbscan.
}
\note{
This is not precisely the \code{HDBSCAN} algorithm because it relies on the
nearest neighbor data generated by the \code{LargeVis} algorithm. In particular,
\code{HDBSCAN} assumes that all points can be connected in a single minimal-spanning
tree. This implementation uses a minimal-spanning forest, because the graph may not
be fully connected depending on the amount of nearest-neighbor data provided.
If, for example, the data has distinct clusters where no member of some cluster is a
nearest neighbor of a point in any other cluster, which can easily happen, the algorithm will
generate distinct trees for each disconnected set of points. In testing, this
improved the performance of the algorithm.

Do not rely on the content of the \code{probabilities} field for outliers. A future version
will hopefully provide some measure of outlier factor.
}
\examples{
\dontrun{
library(largeVis)
library(clusteringdatasets)
data(spiral)
dat <- as.matrix(spiral[, 1:2])
neighbors <- randomProjectionTreeSearch(t(dat), K = 10, tree_threshold = 100,
                                       max_iter = 5)
edges <- buildEdgeMatrix(t(dat), neighbors)
clusters <- hdbscan(edges, verbose = FALSE)

largeHighDimensionalDataset <- matrix(rnorm(50000), ncol = 50)
vis <- largeVis(largeHighDimensionalDataset)
edges <- buildEdgeMatrix(largeHighDimensionalDataset, vis$knns)
clustering <- hdbscan(edges)
gplot(clustering, t(vis$coords))
}
}
\references{
R. Campello, D. Moulavi, and J. Sander, Density-Based Clustering Based on Hierarchical Density Estimates In: Advances in Knowledge Discovery and Data Mining, Springer, pp 160-172. 2013
}
\seealso{
\url{https://github.com/lmcinnes/hdbscan}
}

