% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/jointSeg.R
\name{jointSeg}
\alias{jointSeg}
\title{Joint segmentation of multivariate signals}
\usage{
jointSeg(Y, method = "RBS", stat = NULL, dpStat = stat,
  segFUN = NULL, jitter = NULL,
  modelSelectionMethod = ifelse(match(method, c("DynamicProgramming",
  "RBS", "GFLars"), nomatch = 0) > 0, "Lebarbier", "none"),
  modelSelectionOnDP = (match(method, c("DynamicProgramming", "RBS",
  "GFLars"), nomatch = 0) > 0), ..., profile = FALSE, verbose = FALSE)
}
\arguments{
\item{Y}{The signal to be segmented (a matrix or a numeric vector)}

\item{method}{A \code{character} value, the type of segmentation method used.
May be one of:  \describe{ \item{"RBS"}{Recursive Binary Segmentation (the 
default), see \code{\link{segmentByRBS}} as described in Gey and Lebarbier 
(2005)} \item{"GFLars"}{Group fused LARS as described in Bleakley and Vert 
(2011).} \item{"DP"}{Dynamic Programming (Bellman, 1956). For univariate 
signals the pruned DP of  Rigaill et al (2010) is used.} \item{"other"}{The
segmentation method is passed as a function using argument \code{segFUN} 
(see examples in directory \code{otherMethods} of the \code{jointseg} 
package).}}}

\item{stat}{A vector containing the names or indices of the columns of 
\code{Y} to be segmented}

\item{dpStat}{A vector containing the names or indices of the columns of 
\code{Y} to be segmented at the second round of segmentation. Defaults to 
the value of argument \code{stat}.}

\item{segFUN}{The segmentation function to be used when \code{method} is set 
to \code{other}. Not used otherwise.}

\item{jitter}{Uncertainty on breakpoint position after initial segmentation. 
Defaults to \code{NULL}.  See Details.}

\item{modelSelectionMethod}{A character value, the name of the method used to
perform model selection.}

\item{modelSelectionOnDP}{A logical value. If \code{TRUE} (the default), 
model selection is performed on segmentation after dynamic programming; 
else model selection is performed on initial segmentation.  Only applies to
methods "DP", "RBS" and "GFLars".}

\item{...}{Further arguments to be passed to the lower-level segmentation 
method determined by argument \code{method}.}

\item{profile}{A \code{logical} value: trace time and memory usage ?}

\item{verbose}{A \code{logical} value: should extra information be output ? 
Defaults to \code{FALSE}.}
}
\description{
Joint segmentation of multivariate signals in two steps: \enumerate{ 
\item{first-pass segmentation.  By default, a fast, greedy approach is used 
(see \code{method}).} \item{pruning of the candidate change points obtained 
by dynamic programming} }
}
\details{
If \code{modelSelectionOnDP} is set to \code{FALSE}, then model selection is 
run on the sets of the form \code{bkp[1:k]} for \eqn{1 \leq k \leq 
length(bkp)}, where \code{bkp} is the set of breakpoints identified by the 
initial segmentation.  In particular, this implies that the candidate 
breakpoints in \code{bkp} are sorted by order of appearance and not by 
position.

If \code{jitter} is not \code{NULL}, it should be a vector of integer
indices. The set of candidate breakpoints passed on to dynamic programming is
augmented by all indices distant from an element of \code{jitter} from one of
the candidates. For example, if \code{jitter==c(-1, 0, 1)} and the initial
set of breakpoints is \code{c(1,5)} then dynamic programming is run on
\code{c(1,2,4,5,6)}.

If the return value of the initial segmentation has an element named
\code{dpseg}, then initial segmentation results are not pruned by dynamic
programming.
}
\examples{
## A two-dimensional signal
p <- 2
trueK <- 10
len <- 1e4
sim <- randomProfile(len, trueK, 1, p)
Y <- sim$profile
K <- 2*trueK
res <- jointSeg(Y, method="RBS", K=K)
bkp <- res$bestBkp
getTpFp(bkp, sim$bkp, tol=5, relax = -1)   ## true and false positives
plotSeg(Y, list(sim$bkp, res$bestBkp), col=1)

## Now we add some NA:s in one dimension
jj <- sim$bkp[1]
Y[jj-seq(-10,10), p] <- NA
res2 <- jointSeg(Y, method="RBS", K=K, verbose=TRUE)
bkp <- res2$bestBkp
getTpFp(res2$bestBkp, sim$bkp, tol=5, relax = -1)   ## true and false positives
}
\references{
Bleakley, K., & Vert, J. P. (2011). The group fused lasso for
  multiple change-point detection. arXiv preprint arXiv:1106.4199.

Vert, J. P., & Bleakley, K. (2010). Fast detection of multiple
  change-points shared by many signals using group LARS. Advances in Neural
  Information Processing Systems, 23, 2343-2351.

Gey, S., & Lebarbier, E. (2008). Using CART to Detect Multiple
  Change Points in the Mean for Large Sample.
  http://hal.archives-ouvertes.fr/hal-00327146/

Rigaill, G. (2010). Pruned dynamic programming for optimal
  multiple change-point detection. arXiv preprint arXiv:1004.0887.
}
\seealso{
\code{\link{pruneByDP}}
}
