% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/planning.R
\name{planning}
\alias{planning}
\title{Frequentist and Bayesian Planning for Audit Sampling}
\usage{
planning(confidence = 0.95, expectedError = 0, likelihood = "poisson", N = NULL, 
          materiality = NULL, minPrecision = NULL, 
          prior = FALSE, kPrior = 0, nPrior = 0,
          increase = 1, maxSize = 5000)
}
\arguments{
\item{confidence}{the confidence level desired from the confidence bound (on a scale from 0 to 1). Defaults to 0.95, or 95\% confidence.}

\item{expectedError}{a fraction representing the percentage of expected mistakes in the sample relative to the total size, or a number (>= 1) that represents the number of expected mistakes.}

\item{likelihood}{can be one of \code{binomial}, \code{poisson}, or \code{hypergeometric}.}

\item{N}{the population size (required for hypergeometric calculations).}

\item{materiality}{a value between 0 and 1 representing the materiality of the audit as a fraction of the total size or value. Can be \code{NULL}, but \code{minPrecision} should be specified in that case.}

\item{minPrecision}{The minimum precision to be obtained. Can be \code{NULL}, but \code{materiality} should be specified in that case.}

\item{prior}{whether to use a prior distribution when planning. Defaults to \code{FALSE} for frequentist planning. If \code{TRUE}, the prior distribution is updated by the specified likelihood. Chooses a conjugate gamma distribution for the Poisson likelihood, a conjugate beta distribution for the binomial likelihood, and a conjugate beta-binomial distribution for the hypergeometric likelihood.}

\item{kPrior}{the prior parameter \eqn{\alpha} (number of errors in the assumed prior sample).}

\item{nPrior}{the prior parameter \eqn{\beta} (total number of observations in the assumed prior sample).}

\item{increase}{the desired increase step for the sample size calculation.}

\item{maxSize}{the maximum sample size that is considered for calculations. Defaults to 5000 for efficiency. Increase this value if the sample size cannot be found due to it being too large (e.g., for a low materiality).}
}
\value{
An object of class \code{jfaPlanning} containing:

\item{confidence}{the confidence level for the desired population statement.}
\item{expectedError}{the specified number of errors as a fraction or as a number.}
\item{likelihood}{the specified likelihood.}
\item{N}{the population size (only returned in case of a hypergeometric likelihood).}
\item{materiality}{the value of the specified materiality. Can be \code{NULL}.}
\item{minPrecision}{The minimum precision to be obtained. Can be \code{NULL}.}
\item{sampleSize}{the resulting sample size.}
\item{errorType}{whether the expected errors where specified as a percentage or as an integer.}
\item{expectedSampleError}{the number of full errors that are allowed to occur in the sample.}
\item{expectedBound}{a value specifying the expected upper bound if the sample goes according to plan.}
\item{expectedPrecision}{a value specifying the expected precision if the sample goes according to plan.}
\item{populationK}{the assumed population errors (only returned in case of a hypergeometric likelihood).}
\item{prior}{an object of class \code{jfaPrior} that represents the prior distribution.}
\item{expectedPosterior}{an object of class \code{jfaPosterior} that represents the expected posterior distribution.}
}
\description{
This function calculates the required sample size for an audit, based on the Poisson, binomial, or hypergeometric likelihood. A prior can be specified to perform Bayesian planning. The returned object is of class \code{jfaPlanning} and can be used with associated \code{print()} and \code{plot()} methods.

For more details on how to use this function see the package vignette:
\code{vignette("jfa", package = "jfa")}
}
\details{
This section elaborates on the available likelihoods and corresponding prior distributions for the \code{likelihood} argument.

\itemize{
 \item{\code{poisson}:          The Poisson likelihood is used as a likelihood for monetary unit sampling (MUS). Its likelihood function is defined as: \deqn{p(x) = \frac{\lambda^x e^{-\lambda}}{x!}} The conjugate \emph{gamma(\eqn{\alpha, \beta})} prior has probability density function: \deqn{f(x; \alpha, \beta) = \frac{\beta^\alpha x^{\alpha - 1} e^{-\beta x}}{\Gamma(\alpha)}}}
 \item{\code{binomial}:         The binomial likelihood is used as a likelihood for record sampling \emph{with} replacement. Its likelihood function is defined as: \deqn{p(x) = {n \choose k} p^k (1 - p)^{n - k}} The conjugate \emph{beta(\eqn{\alpha, \beta})} prior has probability density function: \deqn{f(x; \alpha, \beta) = \frac{1}{B(\alpha, \beta)} x^{\alpha - 1} (1 - x)^{\beta - 1}}}
 \item{\code{hypergeometric}:   The hypergeometric likelihood is used as a likelihood for record sampling \emph{without} replacement. Its likelihood function is defined as: \deqn{p(x = k) = \frac{{K \choose k} {N - K \choose n - k}}{{N \choose n}}} The conjugate \emph{beta-binomial(\eqn{\alpha, \beta})} prior (Dyer and Pierce, 1993) has probability density function: \deqn{f(k | n, \alpha, \beta) = {n \choose k} \frac{B(k + \alpha, n - k + \beta)}{B(\alpha, \beta)}} }
}
}
\examples{
library(jfa)

# Using the binomial distribution, calculates the required sample size for a 
# materiality of 5\% when 2.5\% mistakes are expected to be found in the sample.

# Frequentist planning with binomial likelihood:

p1 <- planning(confidence = 0.95, expectedError = 0.025, likelihood = "binomial",
               materiality = 0.05)
print(p1)

# ------------------------------------------------------------
#              jfa Planning Summary (Frequentist)
# ------------------------------------------------------------     
# Input:
# 
# Confidence:              95\% 
# Materiality:             5\% 
# Minimum precision:       Not specified 
# Likelihood:              binomial 
# Expected sample errors:  6 
# ------------------------------------------------------------
# Output:
#
# Sample size:             234 
# ------------------------------------------------------------
# Statistics:
#
# Expected upper bound:    5\% 
# Expected precision:      2.43\% 
# ------------------------------------------------------------ 

# Bayesian planning with prior:

prior <- auditPrior(confidence = 0.95, likelihood = "binomial", method = "arm", 
                    expectedError = 0.025, materiality = 0.05, cr = 0.6)

p3 <- planning(confidence = 0.95, expectedError = 0.025, materiality = 0.05,
               prior = prior)
print(p3)

# ------------------------------------------------------------
#              jfa Planning Summary (Bayesian)
# ------------------------------------------------------------
# Input:
# 
# Confidence:              95\%      
# Materiality:             5\% 
# Minimum precision:       Not specified 
# Likelihood:              binomial 
# Prior distribution:      beta(2.275, 50.725) 
# Expected sample errors:  4.23 
# ------------------------------------------------------------
# Output:
#
# Sample size:             169 
# Posterior distribution:  beta(6.5, 215.5) 
# ------------------------------------------------------------
# Statistics:
#
# Expected upper bound:    4.99\% 
# Expected precision:      2.49\% 
# Expected Bayes factor-+: 9.32 
# ------------------------------------------------------------ 

}
\references{
Dyer, D. and Pierce, R.L. (1993). On the Choice of the Prior Distribution in Hypergeometric Sampling. \emph{Communications in Statistics - Theory and Methods}, 22(8), 2125 - 2146.
}
\seealso{
\code{\link{auditPrior}} \code{\link{selection}} \code{\link{evaluation}}
}
\author{
Koen Derks, \email{k.derks@nyenrode.nl}
}
\keyword{audit}
\keyword{planning}
\keyword{sample}
\keyword{size}
