% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/groups.R
\name{iv-groups}
\alias{iv-groups}
\alias{iv_groups}
\alias{iv_identify_group}
\alias{iv_locate_groups}
\title{Group overlapping intervals}
\usage{
iv_groups(x, ..., abutting = TRUE)

iv_identify_group(x, ..., abutting = TRUE)

iv_locate_groups(x, ..., abutting = TRUE)
}
\arguments{
\item{x}{\verb{[iv]}

An interval vector.}

\item{...}{These dots are for future extensions and must be empty.}

\item{abutting}{\verb{[TRUE / FALSE]}

Should abutting intervals be grouped together?

If \code{TRUE}, \verb{[a, b)} and \verb{[b, c)} will merge as \verb{[a, c)}. If \code{FALSE}, they
will be kept separate. To be a minimal interval vector, all abutting
intervals must be grouped together.}
}
\value{
\itemize{
\item For \code{iv_groups()}, an iv with the same type as \code{x}.
\item For \code{iv_identify_group()}, an iv with the same type and size as \code{x}.
\item For \code{iv_locate_groups()}, a two column data frame with a \code{key} column
containing the result of \code{iv_groups()} and a \code{loc} list-column containing
integer vectors.
}
}
\description{
This family of functions revolves around grouping overlapping intervals
within a single iv. When multiple overlapping intervals are grouped together
they result in a wider interval containing the smallest \code{\link[=iv_start]{iv_start()}} and the
largest \code{\link[=iv_end]{iv_end()}} of the overlaps.
\itemize{
\item \code{iv_groups()} merges all overlapping intervals found within \code{x}. The
resulting intervals are known as the "groups" of \code{x}.
\item \code{iv_identify_group()} identifies the group that the current interval of \code{x}
falls in. This is particularly useful alongside \code{\link[dplyr:group_by]{dplyr::group_by()}}.
\item \code{iv_locate_groups()} returns a two column data frame with a \code{key} column
containing the result of \code{iv_groups()} and a \code{loc} list-column containing
integer vectors that map each interval in \code{x} to the group that it falls in.
}

Optionally, you can choose \emph{not} to group abutting intervals together with
\code{abutting = FALSE}, which can be useful if you'd like to retain those
boundaries.
\subsection{Minimal interval vectors}{

\code{iv_groups()} is particularly useful because it can generate a \emph{minimal}
interval vector, which covers the range of an interval vector in the most
compact form possible. In particular, a minimal interval vector:
\itemize{
\item Has no overlapping intervals
\item Has no abutting intervals
\item Is ordered on both \code{start} and \code{end}
}

A minimal interval vector is allowed to have a single missing interval,
which is located at the end of the vector.
}
}
\section{Graphical Representation}{


Graphically, generating groups looks like:

\figure{groups.png}

With \code{abutting = FALSE}, intervals that touch aren't grouped:

\figure{groups-abutting-keep.png}
}

\examples{
library(dplyr, warn.conflicts = FALSE)

x <- iv_pairs(
  c(1, 5),
  c(2, 3),
  c(NA, NA),
  c(5, 6),
  c(NA, NA),
  c(9, 12),
  c(11, 14)
)
x

# Grouping removes all redundancy while still covering the full range
# of values that were originally represented. If any missing intervals
# are present, a single one is retained.
iv_groups(x)

# Abutting intervals are typically grouped together, but you can choose not
# to group them if you want to retain those boundaries
iv_groups(x, abutting = FALSE)

# `iv_identify_group()` is useful alongside `group_by()` and `summarize()`
df <- tibble(x = x)
df <- mutate(df, u = iv_identify_group(x))
df

df \%>\%
  group_by(u) \%>\%
  summarize(n = n())

# The real workhorse here is `iv_locate_groups()`, which returns
# the groups and information on which observations in `x` fall in which
# group
iv_locate_groups(x)
}
