#' Solution path plot for \code{intsel()}
#'
#' @description
#' Plot the solution path generated by \code{\link{intsel}()}.
#' 
#' @param x Fitted \code{\link{intsel}} model.
#' @param type Graphical argument to be passed to \code{\link{matplot}()}, a character string (length 1 vector) or vector of 1-character strings indicating the type of plot for each column of y, see \link{plot.default} for all possible types. Default is "l" for lines.
#' @param log Graphical argument to be passed to \code{\link{matplot}()}, a character string which contains "x" if the x axis is to be logarithmic, "y" if the y axis is to be logarithmic, "" if neither, "xy" or "yx" if both axes are to be logarithmic. Default is "x".
#' @param ... Further arguments of \code{\link{matplot}()} and ultimately of \code{\link{plot.default}()} for some.
#' @return
#' Produces a coefficient profile plot of the coefficient paths for a fitted \code{\link{intsel}} model.
#' 
#' @examples
#' n <- 1000
#' p.int <- 5
#' p.noint <- 3
#' intercept <- TRUE
#' p.screen <- 5
#' 
#' p.int.expand <- p.int*(p.int-1)/2
#' p.main <- p.int + p.noint
#' x <- matrix(rnorm(n * p.main), nrow = n, ncol = p.main)
#' 
#' # true model
#' # logit(p) = 0.1 + 0.3 x1 + 0.3 x2 + 0.3 x8 + 0.2 * x1 * x2
#' 
#' beta.true <- rep(0, p.main)
#' beta.true[c(1, 2, p.main)] <- 0.3
#' eta <- x %*% beta.true + 0.2 * x[, 1] * x[, 2]
#' 
#' if (intercept) eta <- eta + 0.1
#' 
#' py <- 1/(1 + exp(-eta))
#' 
#' y <- rbinom(n, 1, py)
#' 
#' nlam <- 30
#' lambdas <- exp(seq(log(0.1), log(0.00005), length.out = nlam))
#' 
#' # All the pairwise two-way interactions for the first p.screen variables 
#' # are included in the model and screened in a data-driven manner.
#' fit <- intsel(x = x,
#'               y = y,
#'               p.screen = 5,
#'               intercept = intercept,
#'               lambda = lambdas)
#' plot(fit)
#' 
#' @seealso \code{\link{intsel}}, \code{\link{intsel_cv}}.
#' @method plot intsel
#' @rdname plot.intsel
#' @export
#' @importFrom graphics matplot
#' @importFrom graphics abline

plot.intsel <- function(
    x,
    type = "l",
    log = "x",
    ...
) {
  lambdas <- x$lambdas
  
  estimates <- x$estimates
  
  matplot(x = lambdas,
          y = t(estimates),
          xlab = expression(paste(lambda)),
          ylab = "",
          type = type,
          log = log,
          ...)
  abline(h = 0, lty = 2)
}
