% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_refinement.R
\name{smooth_coef}
\alias{smooth_coef}
\title{Smooth coefficients in the model}
\usage{
smooth_coef(model, x_cut, x_org, degree = NULL, breaks = NULL)
}
\arguments{
\item{model}{object of class glm/smooth}

\item{x_cut}{column name with breaks/cut}

\item{x_org}{column name where x_cut is based on}

\item{degree}{order of polynomial}

\item{breaks}{numerical vector with new clusters for x}
}
\value{
Object of class smooth
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}
Apply smoothing on the risk factors used in the model. \code{smooth_coef()}
must always be followed by \code{refit_glm()}.
}
\details{
Although smoothing could be applied either to the frequency or
the severity model, it is more appropriate to impose the smoothing
on the premium model. This can be achieved by calculating the pure
premium for each record (i.e. expected number of claims times the expected
claim amount), then fitting an "unrestricted" Gamma GLM to the pure premium,
and then imposing the restrictions in a final "restricted" Gamma GLM.
}
\examples{
\dontrun{
library(insurancerating)
library(dplyr)

# Fit GAM for claim frequency
age_policyholder_frequency <- fit_gam(data = MTPL,
                                      nclaims = nclaims,
                                      x = age_policyholder,
                                      exposure = exposure)

# Determine clusters
clusters_freq <- construct_tariff_classes(age_policyholder_frequency)

# Add clusters to MTPL portfolio
dat <- MTPL \%>\%
  mutate(age_policyholder_freq_cat = clusters_freq$tariff_classes) \%>\%
  mutate(across(where(is.character), as.factor)) \%>\%
  mutate(across(where(is.factor), ~biggest_reference(., exposure)))

# Fit frequency and severity model
freq <- glm(nclaims ~ bm + age_policyholder_freq_cat, offset = log(exposure), family = poisson(),
            data = dat)
sev <- glm(amount ~ bm + zip, weights = nclaims, family = Gamma(link = "log"),
           data = dat \%>\% filter(amount > 0))

# Add predictions for freq and sev to data, and calculate premium
premium_df <- dat \%>\%
  add_prediction(freq, sev) \%>\%
  mutate(premium = pred_nclaims_freq * pred_amount_sev)

# Fit unrestricted model
burn_unrestricted <- glm(premium ~ zip + bm + age_policyholder_freq_cat,
                         weights = exposure,
                         family = Gamma(link = "log"),
                         data = premium_df)

# Impose smoothing and create figure
burn_unrestricted \%>\%
  smooth_coef(x_cut = "age_policyholder_freq_cat",
              x_org = "age_policyholder",
              breaks = seq(18, 95, 5)) \%>\%
  autoplot()

# Impose smoothing and refit model
burn_restricted <- burn_unrestricted \%>\%
  smooth_coef(x_cut = "age_policyholder_freq_cat",
              x_org = "age_policyholder",
              breaks = seq(18, 95, 5)) \%>\%
  refit_glm()

# Show new rating factors
rating_factors(burn_restricted)
}

}
\seealso{
\code{\link[=refit_glm]{refit_glm()}} for refitting the smoothed model, and \code{\link[=autoplot.smooth]{autoplot.smooth()}}.

Other refit_glm: 
\code{\link{restrict_coef}()}
}
\author{
Martin Haringa
}
\concept{autoplot.smooth}
\concept{refit_glm}
