##' Subsetting 'incidence' objects
##'
##' Two functions can be used to subset incidence objects.  The function
##' \code{subset} permits to retain dates within a specified range and,
##' optionally, specific groups. The operator "[" can be used as for matrices,
##' using the syntax \code{x[i,j]} where 'i' is a subset of dates, and 'j' is a
##' subset of groups.
##'
##' @author Thibaut Jombart \email{thibautjombart@@gmail.com}
##'
##' @export
##'
##' @rdname subset
##'
##' @aliases "subset.incidence" "[.incidence"
##'
##' @seealso The \code{\link{incidence}} function to generate the 'incidence'
##' objects.
##'
##' @param x An incidence object, generated by the function
##' \code{\link{incidence}}.
##'
##' @param from The starting date; data strictly before this date are discarded.
##'
##' @param to The ending date; data strictly after this date are discarded.
##'
##' @param groups (optional) The groups to retained, indicated as subsets of the
##' columns of x$counts.
##'
##' @param ... Further arguments passed to other methods (not used).
##'
##' @examples
##' ## example using simulated dataset
##' if(require(outbreaks)) {
##'   onset <- ebola_sim$linelist$date.of.onset
##'
##'   ## weekly incidence
##'   inc <- incidence(onset, interval = 7)
##'   inc
##'   inc[1:10] # first 10 weeks
##'   plot(inc[1:10])
##'   inc[-c(11:15)] # remove weeks 11-15
##'   plot(inc[-c(11:15)])
##' }
##'

subset.incidence <- function(x, ..., from = min(x$dates), to = max(x$dates),
                             groups = TRUE){
    to.keep <- x$dates >= from & x$dates <= to

    if (sum(to.keep) < 1) {
        stop("No data retained.")
    }
    x[to.keep, groups]
}




##' @export
##' @rdname subset
##' @param i a subset of dates to retain
##' @param j a subset of groups to retain

"[.incidence" <- function(x, i, j){
    if (missing(i)) {
        i <- TRUE
    }

    if (missing(j)) {
        j <- TRUE
    }

    out <- x
    out$counts <- out$counts[i, j, drop = FALSE]
    out$dates <- out$dates[i]
    if ("isoweeks" %in% names(x)) {
      out$isoweeks <- out$isoweeks[i]
    }
    out$timespan <-  diff(range(out$dates, na.rm = TRUE)) + 1
    out$n <- sum(out$counts)
    out
}
