% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/psm.R
\name{PsmCurves}
\alias{PsmCurves}
\title{Partitioned survival curves}
\format{
An \link[R6:R6Class]{R6::R6Class} object.
}
\description{
Summarize N-1 survival curves for an N-state partitioned survival model.
}
\examples{
library("flexsurv")
N_SAMPLES <- 5 # Number of parameter samples for PSA

# Consider a 3-state model where there is a 
# progression-free survival (PFS) and an
# overall survival (OS) endpoint

# (0) Model setup
hesim_dat <- hesim_data(
  strategies = data.frame(
    strategy_id = c(1, 2),
    strategy_name = c("SOC", "New 1")
  ),
  patients = data.frame(
    patient_id = 1
  )
)

# (1) Parameterize survival models
## (1.1) If patient-level data is available, 
## we can fit survival models

### (1.1.1) Data for estimation (for simplicity, only use 2 strategies)
surv_est_data <- as_pfs_os(
  onc3[strategy_name != "New 2"], 
  patient_vars = c("patient_id", "strategy_name")
)
surv_est_data$strategy_name <- droplevels(surv_est_data$strategy_name)

### (1.1.2) Fit models
fit_pfs <- flexsurvreg(Surv(pfs_time, pfs_status) ~ strategy_name,
                       data = surv_est_data, dist = "exp")
fit_os <- flexsurvreg(Surv(os_time, os_status) ~ strategy_name,
                      data = surv_est_data, dist = "exp")
fits <- flexsurvreg_list(pfs = fit_pfs, os = fit_os)

## (1.2) If patient-level data is NOT available, 
## we can construct the parameter objects "manually"

### (1.2.1) Baseline hazard:
### Assume that we know the (log) rate parameters for both PFS and OS 
### for SOC (i.e., the intercept) and their standard error
logint_pfs_est <- -1.7470900
logint_pfs_se <-  0.03866223
logint_os_est <- -2.7487675
logint_os_se <- 0.04845015

### (1.2.2) Relative treatment effect:
### Assume we know the log hazard ratios (and their standard errors) 
### for comparing the new interventions to the SOC
loghr_pfs_est_new1 <- -0.1772028 
loghr_pfs_se_new1 <- 0.05420119
loghr_os_est_new1 <- -0.1603632
loghr_os_se_new1 <- 0.06948962

### (1.2.3) Create "params_surv_list" object by combining the baseline hazard 
### and relative treatment effects
params <- params_surv_list(
  #### Model for PFS
  pfs = params_surv(
    coefs = list( 
      rate = data.frame( # coefficients predict log rate
        intercept = rnorm(N_SAMPLES, logint_pfs_est, logint_pfs_se),
        new1 = rnorm(N_SAMPLES, loghr_pfs_est_new1, loghr_pfs_se_new1)
      )
    ),
    dist = "exp"
  ),
  
  #### Model for OS
  os = params_surv(
    coefs = list(
      rate = data.frame(
        intercept = rnorm(N_SAMPLES, logint_os_est, logint_os_se),
        new1 = rnorm(N_SAMPLES, loghr_os_est_new1, loghr_os_se_new1)
      )
    ),
    dist = "exp"
  )
)

#### The print (and summary) methods for the "params_surv_list" object will 
#### summarize each of the model terms, which is a good way to check
#### if it's been setup correctly
params 

# (2) Simulation
## (2.1) Construct the model
### (2.1.1) Case where patient-level data was available
### Use create_PsmCurves.params_flexsurvreg_list() method
surv_input_data <- expand(hesim_dat, by = c("strategies", "patients"))
psm_curves1 <- create_PsmCurves(fits, input_data = surv_input_data, 
                                n = N_SAMPLES,
                                uncertainty = "normal",
                                est_data = surv_est_data)

### (2.1.2) Case where patient-level data was NOT available
### Use create_PsmCurves.params_surv_list() method
surv_input_data$intercept <- 1
surv_input_data$new1 <- ifelse(surv_input_data$strategy_name == "New 1", 
                               1, 0)
psm_curves2 <- create_PsmCurves(params, input_data = surv_input_data)

## (2.2) Summarize survival models
## There are minor discrepancies between the case where models were fit
## with flexsurvreg() and the case where the "params_surv_list" object
## was constructed manually due to differences in the random draws
## of the parameter samples. These differences are decreasing in the size 
## of N_SAMPLES
times <- seq(0, 10, 1/12) # Monthly times

### Quantiles
head(psm_curves1$quantile(p = c(.25, .5, .75)))
head(psm_curves2$quantile(p = c(.25, .5, .75)))

### Survival curves
head(psm_curves1$survival(t = times))
head(psm_curves2$survival(t = times))

### Restricted mean survival
head(psm_curves1$rmst(t = c(2, 5)))
head(psm_curves2$rmst(t = c(2, 5)))
}
\seealso{
\code{PsmCurves} are conveniently created from either fitted models or
parameter objects with \code{\link[=create_PsmCurves]{create_PsmCurves()}}. A complete economic model can be
implemented with the \code{\link{Psm}} class. A longer example is provided in
\code{vignette("psm")}.
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{params}}{An object of class \code{\link{params_surv_list}}.}

\item{\code{input_data}}{An object of class \code{\link{input_mats}}. Each row in \code{X} must
be a unique treatment strategy and patient.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{PsmCurves$new()}}
\item \href{#method-hazard}{\code{PsmCurves$hazard()}}
\item \href{#method-cumhazard}{\code{PsmCurves$cumhazard()}}
\item \href{#method-survival}{\code{PsmCurves$survival()}}
\item \href{#method-rmst}{\code{PsmCurves$rmst()}}
\item \href{#method-quantile}{\code{PsmCurves$quantile()}}
\item \href{#method-check}{\code{PsmCurves$check()}}
\item \href{#method-clone}{\code{PsmCurves$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Create a new \code{PsmCurves} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{PsmCurves$new(params, input_data)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{params}}{The \code{params} field.}

\item{\code{input_data}}{The \code{input_data} field.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new \code{PsmCurves} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-hazard"></a>}}
\if{latex}{\out{\hypertarget{method-hazard}{}}}
\subsection{Method \code{hazard()}}{
Predict the hazard function for each survival curve as a function of time.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{PsmCurves$hazard(t)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{t}}{A numeric vector of times.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{data.table} with columns \code{sample}, \code{strategy_id},
\code{patient_id}, \code{grp_id}, \code{curve} (the curve number), \code{t}, and \code{hazard}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cumhazard"></a>}}
\if{latex}{\out{\hypertarget{method-cumhazard}{}}}
\subsection{Method \code{cumhazard()}}{
Predict the cumulative hazard function for each survival curve as a function of time.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{PsmCurves$cumhazard(t)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{t}}{A numeric vector of times.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{data.table} with columns \code{sample}, \code{strategy_id},
\code{patient_id}, \code{grp_id}, \code{curve}, \code{t}, and \code{cumhazard}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-survival"></a>}}
\if{latex}{\out{\hypertarget{method-survival}{}}}
\subsection{Method \code{survival()}}{
Predict survival probabilities for each survival curve as a function of time.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{PsmCurves$survival(t)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{t}}{A numeric vector of times.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An object of class \code{\link{survival}}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-rmst"></a>}}
\if{latex}{\out{\hypertarget{method-rmst}{}}}
\subsection{Method \code{rmst()}}{
Predict the restricted mean survival time up until time points \code{t}
for each survival curve.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{PsmCurves$rmst(t, dr = 0)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{t}}{A numeric vector of times.}

\item{\code{dr}}{Discount rate.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{data.table} with columns \code{sample}, \code{strategy_id},
\code{patient_id}, \code{grp_id}, \code{curve}, \code{t}, and \code{rmst}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-quantile"></a>}}
\if{latex}{\out{\hypertarget{method-quantile}{}}}
\subsection{Method \code{quantile()}}{
Predict quantiles of the survival distribution for each survival curve.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{PsmCurves$quantile(p)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{p}}{A numeric vector of probabilities for computing quantiles.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{data.table} with columns \code{sample}, \code{strategy_id},
\code{patient_id}, \code{grp_id}, \code{curve}, \code{p} and \code{quantile}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-check"></a>}}
\if{latex}{\out{\hypertarget{method-check}{}}}
\subsection{Method \code{check()}}{
Input validation for class. Checks that fields are the correct type.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{PsmCurves$check()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{PsmCurves$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
