hergm.postprocess <- function(sample = NULL,
                              burnin = NULL, 
                              relabel = TRUE,
                              name = NULL, 
                              ...)
# input: MCMC sample generated by function hergm, number of burn-in iterations, relabeling desired, output desired, root of name of output files 
# output: postprocessed MCMC sample
{

  # Extract arguments
  n <- sample$n
  max_number <- sample$max_number
  d1 <- sample$d1
  d2 <- sample$d2
  parallel <- sample$parallel
  sample_size <- sample$sample_size
  mcmc <- sample$sample 

  # Check arguments
  if (burnin > sample_size) 
    {
    cat("\n\n")
    error_message <- paste("number of burn-in iterations ", burnin, " exceeds number of recorded iterations ", sample_size, ".", sep = "")
    stop(error_message, call. = FALSE)
    }
  if ((relabel == TRUE) && (max_number > 10)) cat("\nWarning: relabeling time-consuming.\n")

  # Preprocess MCMC sample: delete burn-in iterations and transform vector into matrix, where rows correspond to MCMC draws
  d <- d1 + d2
  terms <- length_mcmc(d1, d2, max_number, n)  
  mcmc_sample <- NULL
  count <- 0
  for (i in 1:parallel) 
    {
    first <- count + (burnin * terms) + 1
    last <- count + (sample_size * terms)
    mcmc_sample <- append(mcmc_sample, mcmc[first:last])
    count <- count + (sample_size * terms)
    }
  mcmc_sample_size <- parallel * (sample_size - burnin)
  mcmc_sample <- matrix(mcmc_sample, nrow = mcmc_sample_size, ncol = terms, byrow = TRUE)

  # Initialize arrays
  output <- list()
  if (d1 > 0) output$ergm_theta <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d1) 
  output$base_mean <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d2)
  output$base_precision <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d2)
  output$hergm_theta <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d2 * (max_number + 1))
  output$indicator <- matrix(data = 0, nrow = mcmc_sample_size, ncol = n)
  output$size <- matrix(data = 0, nrow = mcmc_sample_size, ncol = max_number)
  output$p_k <- matrix(data = 0, nrow = mcmc_sample_size, ncol = max_number)
  output$alpha <- matrix(data = 0, nrow = mcmc_sample_size, ncol = 1)
  output$prediction <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d)

  # Process MCMC sample
  for (row in 1:mcmc_sample_size)
    {
    column <- 0
    if (d1 > 0)
      {
      for (i in 1:d1) 
        {
        column <- column + 1
        output$ergm_theta[row,i] <- mcmc_sample[row,column]
        }
      }
    for (i in 1:d2) 
      {
      column <- column + 1
      output$base_mean[row,i] <- mcmc_sample[row,column]
      }
    for (i in 1:d2) 
      {
      column <- column + 1
      output$base_precision[row,i] <- mcmc_sample[row,column]
      }
    for (i in 1:(d2 * (max_number + 1))) 
      {
      column <- column + 1
      output$hergm_theta[row,i] <- mcmc_sample[row,column]
      }
    for (i in 1:n) 
      {
      column <- column + 1
      output$indicator[row,i] <- mcmc_sample[row,column]
      }
    for (i in 1:max_number) 
      {
      column <- column + 1
      output$size[row,i] <- mcmc_sample[row,column]
      }
    for (i in 1:max_number) 
      {
      column <- column + 1
      output$p_k[row,i] <- mcmc_sample[row,column]
      }
    column <- column + 1
    output$alpha[row,1] <- mcmc_sample[row,column]
    for (i in 1:d) 
      {
      column <- column + 1
      output$prediction[row,i] <- mcmc_sample[row,column]
      }
    }

  # Write MCMC sample to files
  if (is.null(name)) name <- ""
  else name <- paste(name, "_", sep = "")
  if (d1 > 0) write(t(output$ergm_theta), paste(sep = "",name,"parameter.out"), ncolumns = d1)
  write(t(output$base_mean), paste(sep = "",name,"mean_block_parameter.out"), ncolumns = d2)
  write(t(output$base_precision), paste(sep = "",name,"precision_block_parameter.out"), ncolumns = d2)
  write(t(output$hergm_theta), paste(sep = "",name,"block_parameter.out"), ncolumns = d2 * (max_number + 1))
  write(t(output$indicator), paste(sep = "",name,"indicator.out"), ncolumns = n)
  write(t(output$size), paste(sep = "",name,"size.out"), ncolumns = max_number)
  write(t(output$p_k), paste(sep = "",name,"p.out"), ncolumns = max_number)
  write(t(output$alpha), paste(sep = "",name,"alpha.out"), ncolumns = 1)
  write(t(output$prediction), paste(sep = "", name, "statistics.out"), ncolumns = d)

  # Relabel sample
  if (relabel == TRUE)
    {
    minimizer <- hergm.min_loss(max_number, paste(sep = "", name, "indicator.out"), 0, 100) # Specify number of iterations of post-processing algorithm
    output$p_i_k <- minimizer$p
    index <- 0
    for (h_term in 1:d2)
      {
      index <- index + 1
      theta <- output$hergm_theta[,index]
      for (i in 2:max_number) 
        {
        index <- index + 1
        theta <- cbind(theta, output$hergm_theta[,index])
        }
      write(t(theta), paste(sep = "", name, "block_parameter_", h_term, ".out"), ncolumns = max_number)
      min_theta <- hergm.permute_mcmc(theta, max_number, minimizer$min_permutations) 
      write(t(min_theta), paste(sep = "", name, "block_parameter_min_", h_term, ".out"), ncolumns = max_number)
      }
    }

  cat("\n")

  output

}

